/*
 * scamper_addr.h
 *
 * $Id: scamper_addr.h,v 1.10 2006/11/23 20:10:06 mjl Exp $
 *
 * Copyright (C) 2004 The University of Waikato
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __SCAMPER_ADDR_H
#define __SCAMPER_ADDR_H

/*
 * the scamper_addr code has a number of purposes:
 *
 * first, it provides a way to record different address types in the same
 * structure regardless of the operating system scamper is used on.
 *
 * second, it provides a way to reference count the addresses contained, so
 * that users of addresses do not have to allocate copies of the address
 * each time they use the address internally.
 *
 * third, it can provide a 'cache' of addresses so that only one copy of
 * the address needs to be kept among all users of scamper_addr.
 */

/*
 * these are the types of addresses that scamper understands
 */
#define SCAMPER_ADDR_TYPE_IPV4     0x01 /* 32 bit IPv4 address */
#define SCAMPER_ADDR_TYPE_IPV6     0x02 /* 128 bit IPv6 address */
#define SCAMPER_ADDR_TYPE_ETHERNET 0x03 /* 48 bit ethernet mac address */
#define SCAMPER_ADDR_TYPE_FIREWIRE 0x04 /* 64 bit firewire link address */

/*
 * scamper_addr:
 *
 * this structure is used to store an address in scamper.
 * the contents of this will eventually be made private, so users of
 * addresses should not count on the contents of the struct remaining
 * public.
 */
typedef struct scamper_addr
{
  int   type;
  void *addr;
  int   refcnt;
  void *internal;
} scamper_addr_t;

/*
 * scamper_addr_alloc:
 *  allocate an address of the specified type; the reference count of the
 *  address is initialised to one
 *
 * scamper_addr_use:
 *  increment the reference count by one, and return a reference to the
 *  address passed in as a convenience.
 *
 * scamper_addr_free:
 *  decrement the reference count held on the address.  when the reference
 *  count becomes zero, the address is freed.
 *
 * scamper_addr_resolve:
 *  attempt to resolve the string as getaddrinfo would, and return the address
 */
scamper_addr_t *scamper_addr_alloc(const int type, const void *addr);
scamper_addr_t *scamper_addr_use(scamper_addr_t *sa);
void scamper_addr_free(scamper_addr_t *sa);
scamper_addr_t *scamper_addr_resolve(const int af, const char *str);

/*
 * scamper_addr_alloc_[ipv4|ipv6|ethernet|firewire]
 *
 * these macros are provided as a convenience as the type constants can
 * become unwieldy to use
 */
#define scamper_addr_alloc_ipv4(addr) \
 scamper_addr_alloc(SCAMPER_ADDR_TYPE_IPV4, addr)

#define scamper_addr_alloc_ipv6(addr) \
 scamper_addr_alloc(SCAMPER_ADDR_TYPE_IPV6, addr)

#define scamper_addr_alloc_ethernet(addr) \
 scamper_addr_alloc(SCAMPER_ADDR_TYPE_ETHERNET, addr)

#define scamper_addr_alloc_firewire(addr) \
 scamper_addr_alloc(SCAMPER_ADDR_TYPE_FIREWIRE, addr)

/*
 * scamper_addr_cmp:
 *  given two addresses, return their sort order.
 */
int scamper_addr_cmp(const scamper_addr_t *a, const scamper_addr_t *b);

/*
 * scamper_addr_tostr:
 *  given a scamper address, convert it to a string representation in the
 *  buffer of specified size.
 */
const char *scamper_addr_tostr(const scamper_addr_t *sa,
			       char *dst, const size_t size);

/*
 * scamper_addr_size
 *  return the size of the underlying address stored in the scamper_addr
 *  structure.  useful for writing address objects to disk...
 */
size_t scamper_addr_size(const scamper_addr_t *sa);

/*
 * scamper_addrcache:
 *  store identical addresses just once in this structure
 *
 * scamper_addrcache_alloc:
 *  allocate an empty address cache and return a pointer to it
 *
 * scamper_addrcache_free:
 *  free the address cache structure.  all addresses have their reference
 *  count decremented; if their reference count is zero, the underlying
 *  address is freed as well.
 */
typedef struct scamper_addrcache scamper_addrcache_t;
scamper_addrcache_t *scamper_addrcache_alloc(void);
void scamper_addrcache_free(scamper_addrcache_t *ac);

/*
 * scamper_addrcache_get:
 *  return a pointer to a scamper_addr_t which corresponds to the address
 *  out of the cache; allocate the address from scratch if necessary
 */
scamper_addr_t *scamper_addrcache_get(scamper_addrcache_t *ac,
				      const int type, const void *addr);

scamper_addr_t *scamper_addrcache_resolve(scamper_addrcache_t *ac,
					  const int af, const char *addr);

/*
 * scamper_addrcache_get_[ipv4|ipv6|ethernet|firewire]
 *
 * these macros are provided as a convenience as the type constants can
 * become unwieldy to use
 */
#define scamper_addrcache_get_ipv4(addrcache, addr) \
 scamper_addrcache_get(addrcache, SCAMPER_ADDR_TYPE_IPV4, addr)

#define scamper_addrcache_get_ipv6(addrcache, addr) \
 scamper_addrcache_get(addrcache, SCAMPER_ADDR_TYPE_IPV6, addr)

#define scamper_addrcache_get_ethernet(addrcache, addr) \
 scamper_addrcache_get(addrcache, SCAMPER_ADDR_TYPE_ETHERNET, addr)

#define scamper_addrcache_get_firewire(addrcache, addr) \
 scamper_addrcache_get(addrcache, SCAMPER_ADDR_TYPE_FIREWIRE, addr)

#endif /* __SCAMPER_ADDR_H */
