// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <string.h>

#include "frscale.h"

FrScale::FrScale()
{
  colorScaleType_ = LINEARSCALE;
  clipScope_ = LOCAL;
  clipMode_ = MINMAX;
  mmMode_ = SCAN;
  low_ = 1;
  high_ = 100;
  min_ = 1;
  max_ = 100;
  uLow_ = 1;
  uHigh_ = 100;
  zContrast_ = .25;
  zSampleSize_ = 600;
  zSampleLine_ = 120;
  mmIncr_ = 10;
  autoCutPer_ = 1;
  scanMode_ = DATASEC;

  histogramX_ = NULL;
  histogramY_ = NULL;
  histequ_ = NULL;

  histogramXSize_ = 0;
  histogramYSize_ = 0;
  histequSize_ = 0;

  preserve_ = 0;
}

FrScale::~FrScale()
{
  if (histogramX_)
    free(histogramX_);

  if (histogramY_)
    free(histogramY_);
  
  if (histequ_)
    delete histequ_;
}

FrScale::FrScale(const FrScale& a)
{
  colorScaleType_ = a.colorScaleType_;
  clipScope_ = a.clipScope_;
  clipMode_ = a.clipMode_;
  mmMode_ = a.mmMode_;
  low_ = a.low_;
  high_ = a.high_;
  min_ = a.min_;
  max_ = a.max_;
  uLow_ = a.uLow_;
  uHigh_ = a.uHigh_;
  zContrast_ = a.zContrast_;
  zSampleSize_ = a.zSampleSize_;
  zSampleLine_ = a.zSampleLine_;
  mmIncr_ = a.mmIncr_;
  autoCutPer_ = a.autoCutPer_;
  scanMode_ = a.scanMode_;

  if (a.histogramX_) {
    histogramX_ = (double*)calloc(a.histogramXSize_, sizeof(double));
    memcpy(histogramX_,a.histogramX_,a.histogramXSize_*sizeof(double));
  }
  else
    histogramX_ = NULL;

  if (a.histogramY_) {
    histogramY_ = (double*)calloc(a.histogramYSize_, sizeof(double));
    memcpy(histogramY_,a.histogramY_,a.histogramYSize_*sizeof(double));
  }
  else
    histogramY_ = NULL;

  if (a.histequ_) {
    histequ_ = new double[a.histequSize_];
    memcpy(histequ_,a.histequ_,a.histequSize_*sizeof(double));
  }
  else
    histequ_ = NULL;

  histogramXSize_ = a.histogramXSize_;
  histogramYSize_ = a.histogramYSize_;
  histequSize_ = a.histequSize_;

  preserve_ = a.preserve_;
}

FrScale& FrScale::operator=(const FrScale& a)
{
  colorScaleType_ = a.colorScaleType_;
  clipScope_ = a.clipScope_;
  clipMode_ = a.clipMode_;
  mmMode_ = a.mmMode_;
  low_ = a.low_;
  high_ = a.high_;
  min_ = a.min_;
  max_ = a.max_;
  uLow_ = a.uLow_;
  uHigh_ = a.uHigh_;
  zContrast_ = a.zContrast_;
  zSampleSize_ = a.zSampleSize_;
  zSampleLine_ = a.zSampleLine_;
  mmIncr_ = a.mmIncr_;
  autoCutPer_ = a.autoCutPer_;
  scanMode_ = a.scanMode_;

  if (histogramX_)
    free(histogramX_);
  histogramX_ = (double*)calloc(a.histogramXSize_, sizeof(double));
  memcpy(histogramX_,a.histogramX_,a.histogramXSize_*sizeof(double));

  if (histogramY_)
    free(histogramY_);
  histogramY_ = (double*)calloc(a.histogramYSize_, sizeof(double));
  memcpy(histogramY_,a.histogramY_,a.histogramYSize_*sizeof(double));

  if (histequ_)
    delete histequ_;
  histequ_ = new double[a.histequSize_];
  memcpy(histequ_,a.histequ_,a.histequSize_*sizeof(double));

  histogramXSize_ = a.histogramXSize_;
  histogramYSize_ = a.histogramYSize_;
  histequSize_ = a.histequSize_;

  preserve_ = a.preserve_;
}

double* FrScale::initHistogramX(int n) {
  deleteHistogramX();
  histogramXSize_ = n;
  return histogramX_ = (double*)calloc(n, sizeof(double));
}

double* FrScale::initHistogramY(int n) {
  deleteHistogramY();
  histogramYSize_ = n;
  return histogramY_ = (double*)calloc(n, sizeof(double));
}

double* FrScale::initHistequ(int n) {
  deleteHistequ();
  histequSize_ = n;
  return histequ_ = new double[n];
}

void FrScale::deleteHistogramX()
{
  if (histogramX_) 
    free(histogramX_); 
  histogramX_ = NULL; 
  histogramXSize_ = 0;
}

void FrScale::deleteHistogramY()
{
  if (histogramY_) 
    free(histogramY_); 
  histogramY_ = NULL; 
  histogramYSize_ = 0;
}

void FrScale::deleteHistequ()
{
  if (histequ_) 
    free(histequ_); 
  histequ_ = NULL; 
  histequSize_ = 0;
}

