// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "circle.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Circle::Circle(const Circle& a) : BaseEllipse(a) {}

Circle::Circle(FrameBase* p, const Vector& ctr,
	       double r, 
	       const char* clr, int wth, const char* fnt, const char* txt, 
	       unsigned short prop, const char* cmt,
	       const List<Tag>& tg, const List<CallBack>& cb)
  : BaseEllipse(p, ctr, 0, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = 1;
  annuli_ = new Vector[numAnnuli_];
  annuli_[0] = Vector(r,r);

  strcpy(type, "circle");
  numHandle = 4;

  updateBBox();
}

void Circle::edit(const Vector& v, int h)
{
  Matrix mm = bckCanvasMatrix();

  // calc dist between edge of circle and handle
  double d = annuli_[0].length() - annuli_[0][0];
  double r = (v * mm).length() - d;
  annuli_[0] = Vector(r,r);
  
  updateBBox();
  doCallBack(CallBack::EDITCB);
}

// list

void Circle::list(ostream& str, CoordSystem sys, SkyFrame sky,
		  SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(center);
  listPre(str, sys, sky, ptr, strip, 0);
  
  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      double r = ptr->mapLenFromRef(annuli_[0][0],sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	  << r <<  ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    double r = ptr->mapLenFromRef(annuli_[0][0],sys,ARCSEC);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] <<','
		<< r << '"' <<  ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;
	    double r = ptr->mapLenFromRef(annuli_[0][0],sys,ARCSEC);
	    str << type << '(' << ra << ',' << dec << ',' << r << '"' <<  ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	double r = ptr->mapLenFromRef(annuli_[0][0],sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] <<','
	    << r <<  ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Circle::listCiao(ostream& str, CoordSystem sys, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listCiaoPre(str);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,PHYSICAL);
      double r = ptr->mapLenFromRef(annuli_[0][0],PHYSICAL);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << r << ')';
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      double r = ptr->mapLenFromRef(annuli_[0][0],sys,ARCMIN);
      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
      string x(buf);
      istringstream wcs(x);
      wcs >> ra >> dec;
      str << type << '(' << ra << ',' << dec << ',' << r << '\'' << ')';
    }
  }

  listCiaoPost(str, conj, strip);
}

void Circle::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		     SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listSAOtngPre(str, strip);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {

      Vector v = ptr->mapFromRef(center,IMAGE);
      double r = ptr->mapLenFromRef(annuli_[0][0],IMAGE);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << r << ')';
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  double r = ptr->mapLenFromRef(annuli_[0][0],IMAGE);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	      << r << ')';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  double r = ptr->mapLenFromRef(annuli_[0][0],IMAGE);
	  char ra[16];
	  char dec[16];
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  str << type << '(' << ra << ',' << dec << ',' << r << ')';
	}
	break;
      }
    }
  }

  listSAOtngPost(str,strip);
}

void Circle::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      Vector v = ptr->mapFromRef(center,sys);
      double r = ptr->mapLenFromRef(annuli_[0][0],IMAGE);
      listProsCoordSystem(str,sys,sky);
      str << "; " << type << ' ' << setprecision(8) << v << r;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  double r = ptr->mapLenFromRef(annuli_[0][0],sys,ARCSEC);
	  listProsCoordSystem(str,sys,sky);
	  str << "; " << type << ' ' << setprecision(8) 
	      << v[0] << "d " << v[1] << "d "
	      << r << "\" ";
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char decc[16];
	  char *dec = decc;
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    dec++;

	  double r = ptr->mapLenFromRef(annuli_[0][0],sys,ARCSEC);
	  listProsCoordSystem(str,sys,sky);
	  str << "; " << type << ' ' << ra << ' ' << dec << ' '
	      << r << "\" ";
	}
	break;
      }
    }
  }

  listProsPost(str, strip);
}

void Circle::listSAOimage(ostream& str, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listSAOimagePre(str);

  Vector v = ptr->mapFromRef(center,IMAGE);
  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
      << annuli_[0][0] << ')';

  listSAOimagePost(str, strip);
}

// special composite funtionallity

void Circle::setComposite(const Matrix& mx, double aa)
{
  center *= mx;
  updateBBox();
}

