/*
 * Copyright (C) 2006 Sjoerd Simons <sjoerd@luon.net>
 * Copyright (C) 2003, 2004 Laurent Sansonetti <lrz@gnome.org>
 *
 * This file is part of Ruby/GStreamer.
 *
 * Ruby/GStreamer is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * Ruby/GStreamer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Ruby/GStreamer; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "rbgst.h"


VALUE klass;
/*
 * Since GstStructure is basically a hash it is more natural to
 * automatically convert it as a Ruby Hash object, and vice-versa.
, */

static VALUE
rb_gst_structure_new(int argc, VALUE *argv, VALUE self) {
  VALUE name, values;
  int ret;
  ret = rb_scan_args(argc, argv, "11", &name, &values);

  Check_Type(name, T_STRING);
  rb_iv_set(self, "@name", name);

  if (ret == 2) {
    Check_Type(values, T_HASH);
    rb_funcall(self, rb_intern("replace"), 1, values);
  }
  return self;
}

static VALUE
rb_gst_structure_inspect(VALUE self) {
  VALUE result = rb_str_new2("Gst::Structure ");
  result = rb_str_concat(result, rb_ivar_get(self, rb_intern("@name")));
  result = rb_str_cat(result, " ", 1);
  result = rb_str_concat(result, rb_call_super(0, NULL));
  return result;
}

static gboolean
foreach_pair (GQuark field_id, const GValue *value, gpointer user_data)
{
    VALUE hash;
    
    hash = *(VALUE *) user_data;
    rb_hash_aset (hash, CSTR2RVAL (g_quark_to_string (field_id)), 
                  GVAL2RVAL (value));
    
    return TRUE;
}

VALUE 
rbgststructure_structure2rval(GstStructure *structure)
{
    VALUE result;
    VALUE name;

    name = CSTR2RVAL(gst_structure_get_name(structure));
    
    result = rb_class_new_instance(1, &name, klass);
    gst_structure_foreach (structure, foreach_pair, &result);
    return result;
}

GstStructure *
rbgststructure_hash_to_gst_structure (VALUE hash, const char *name)
{
    GstStructure *gst_struct;
    VALUE ary;
    int i;
    
    Check_Type (hash, T_HASH);

    gst_struct = gst_structure_empty_new (name);
    ary = rb_funcall (hash, rb_intern ("to_a"), 0);
    for (i = 0; i < RARRAY (ary)->len; i++) {
        VALUE pair = RARRAY (ary)->ptr[i];
        GValue value = { 0, };
    
        g_value_init (&value, RVAL2GTYPE (RARRAY (pair)->ptr[1]));
        rbgobj_rvalue_to_gvalue (RARRAY (pair)->ptr[1], &value);
        gst_structure_set_value (gst_struct,
                                 RVAL2CSTR (RARRAY (pair)->ptr[0]),
                                 &value);
        g_value_unset (&value);
    }
    
    return gst_struct; 
}

GstStructure*
rbgststructure_rval2structure(VALUE structure) {
  VALUE name = rb_ivar_get(structure, rb_intern("@name"));
  char *cname = RVAL2CSTR(name);
  return rbgststructure_hash_to_gst_structure(structure, cname);
}

void
Init_gst_structure(void) {
  klass = rb_define_class_under(mGst, "Structure", rb_cHash);
  rb_funcall(klass, rb_intern("module_eval"), 1, rb_str_new2("attr_accessor :name"));
  rb_define_method(klass, "initialize", rb_gst_structure_new, -1);
  rb_define_method(klass, "inspect", rb_gst_structure_inspect, 0);
}
