//proto_simple.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2011
 *
 *  This file is part of RoarAudio PlayList Daemon,
 *  a playlist management daemon for RoarAudio.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "rpld.h"

// workarounds:
#ifdef ROAR_TARGET_WIN32
static int asprintf(char **strp, const char *fmt, ...) {
 static char buf[1024];
 int         ret;
 va_list     ap;

 // TODO: FIXME: this returns a static buffer, not a malloc()ed.

 *strp = buf;

 va_start(ap, fmt);
 ret = vsnprintf(buf, 1024, fmt, ap);
 va_end(ap);

 return ret;
}
#endif

static void remove_spaces(char ** str) {
 char * c;

 if ( str == NULL )
  return;

 c = *str;

 for (; *c == ' '; c++);

 *str = c;
}


#define HT_BOOL          "TRUE|FALSE|TOGGLE"
#define HT_PLI           "{\"Name\"|ID}"
#define HT_PLI_ANY       "{\"Name\"|ID|ANY}"

#ifdef HAVE_LIB_UUID
#define HT_EI            "{long:0xLongID|0xLongID|short:0xShortID|uuid:UUID|pointer:name|num:Index|likeness:Likeness}"
#else
#define HT_EI            "{long:0xLongID|0xLongID|short:0xShortID|pointer:name|num:Index|likeness:Likeness}"
#endif

#define HT_IOI           "{STDIN|STDOUT|\"Filename\"}"
#define HT_PLT           "{RPLD|PLS|M3U|VCLT|XSPF|PLAIN}"
#define HT_IMPORT_EXPORT "[" HT_PLI "] {TO|FROM} " HT_IOI " [AS " HT_PLT "]"
#define HT_STORE_RESTORE "{NONE|CONFIG|ALL|QUEUE|PLAYLIST " HT_PLI_ANY "}"
#define HT_PLE_BOPTS     HT_EI " [FROM " HT_PLI_ANY "]"
#define HT_PLE_POS       " [AT POSITION]"
#define HT_PLE_COPY_MOVE HT_PLE_BOPTS " TO " HT_PLI HT_PLE_POS
#define HT_PLE_SEARCH_TERM "{\"search string\"|0xdiscid|uuid}"
#define HT_PLE_SEARCH_F  "{ALL|TITLE|ARTIST|VERSION|FILENAME|DISCID|UUID|TAG:\"Tagname\"}"
#define HT_PLE_SEARCH_OP "{AS|IN}"
#define HT_PLE_SEARCH    HT_PLE_SEARCH_TERM " [" HT_PLE_SEARCH_OP " " HT_PLE_SEARCH_F "] [FROM " HT_PLI_ANY "]"
#define HT_PTRI          "{CURRENT|DEFAULT|STARTUP|TEMP}"
#define HT_POINTER_SET   HT_PTRI " [TO] " HT_PLE_BOPTS
#define HT_POINTER_SHOW  "[" HT_PTRI "]"
#define HT_POLICYS       "{DEFAULT|OLD|NEW}"
#define HT_POLICY        "{KEEP|DELETE}"

static char * proto_simple_help_text[] = {
 "Commands:\n",
 "  HELP\n",
 "  NOOP\n",
 "  PLAY\n",
 "  STOP\n",
 "  NEXT\n",
 "  PREV\n",
 "  ISPLAYING\n",
 "  SHOWIDENTIFIER\n",
 "  SHOWCUR\n",
 "  LISTQ [" HT_PLI "]\n",
 "  FLUSHQ\n",
 "  SETVOLUME new-volume\n",
 "  SHOWVOLUME\n",
 "  QUIT\n",
 "  LISTPLAYLISTS\n",
 "  SETPLAYLIST " HT_PLI "\n",
 "  SHOWLIST [" HT_PLI "]\n",
 "  ADDPLAYLIST \"Name\"\n",
 "  DELPLAYLIST [" HT_PLI "]\n",
 "  FLUSHPLAYLIST [" HT_PLI "]\n",
 "  LISTPLE [" HT_PLI "]\n",
 "  SHOWPLE " HT_PLE_BOPTS "\n",
 "  COPYPLE " HT_PLE_COPY_MOVE "\n",
 "  MOVEPLE " HT_PLE_COPY_MOVE "\n",
 "  DELPLE " HT_PLE_BOPTS "\n",
 "  QUEUEPLE " HT_PLE_BOPTS HT_PLE_POS "\n",
 "  IMPORT " HT_IMPORT_EXPORT "\n",
 "  EXPORT " HT_IMPORT_EXPORT "\n",
 "  SETPOINTER " HT_POINTER_SET "\n",
 "  SHOWPOINTER " HT_POINTER_SHOW "\n",
 "  PAUSE " HT_BOOL "\n",
 "  UNAUTH [ACCLEV] {BY n|TO {n|\"name\"}}\n",
 "  LIKE " HT_PLE_BOPTS " [+/- LIKENESS]\n",
 "  DISLIKE " HT_PLE_BOPTS " [+/- LIKENESS]\n",
 "Commands to fix:\n",
 "  SETPARENTLIST [OF " HT_PLI "] [TO] " HT_PLI "\n",
 "  SHOWPLAYING\n",
 "  STORE " HT_STORE_RESTORE "\n",
 "  RESTORE " HT_STORE_RESTORE "\n",
 // TODO: To implement
 "Commands to implement:\n",
 "  SEARCHPLE " HT_PLE_SEARCH "\n",
 "  UPDATE " HT_IMPORT_EXPORT " USING " HT_PLE_SEARCH_F " [AND ...]" " [POLICY " HT_POLICYS " " HT_POLICY "]" "\n",
 "  AUTH [TO AUTHLEV {n|\"name\"}] USING authtype...\n",
 "Commands to remove (obsoleted):\n",
 "  ADD2Q entry\n",
 "  DELFQ identifier\n",
 NULL
};

static int proto_simple_client_puts(int client, char * text) {
 struct roar_vio_calls * vio = client_get_vio(client);
 ssize_t len = strlen(text);

 if ( roar_vio_write(vio, text, (size_t)len) != len )
  return -1;

 return 0;
}

int proto_simple_client_handle(int id) {
 struct roar_vio_calls * vio = client_get_vio(id);
 int status;
 ssize_t len;
 char buffer[1024];
 const char * statustext;

 if ( vio == NULL )
  return -1;

 if ( (len = roar_vio_read(vio, buffer, 1023)) < 1 )
  return -1;

 if ( buffer[len-1] != '\n' )
  return -1;

 if ( len > 1 && buffer[len-2] == '\r' ) {
  len -= 2;
 } else {
  len--;
 }

 buffer[len] = 0;

 if ( len > 0 ) {
  status = proto_simple_parse(id, buffer, len);
 } else {
  status = PROTO_SIMPLE_STATUS_OK;
 }

 switch (status) {
  case PROTO_SIMPLE_STATUS_OK:      statustext = "OK";             break;
  case PROTO_SIMPLE_STATUS_ERROR:   statustext = "Error";          break;
  case PROTO_SIMPLE_STATUS_INPROC:  statustext = "In Process";     break;
  case PROTO_SIMPLE_STATUS_YES:     statustext = "Yes";            break;
  case PROTO_SIMPLE_STATUS_NO:      statustext = "No";             break;
  case PROTO_SIMPLE_STATUS_PROERR:  statustext = "Protocol Error"; break;
  case PROTO_SIMPLE_STATUS_GOODBYE: statustext = "GoodBye";        break;
  case PROTO_SIMPLE_STATUS_NOTPERM: statustext = "Not Permitted";  break;
  default:                          statustext = "(UNKNOWN)";      break;
 }

 snprintf(buffer, 1023, ".\n> %i %s\n", status, statustext);
 buffer[1023] = 0;

 if ( proto_simple_client_puts(id, buffer) == -1 )
  return -1;

 switch (status) {
  case PROTO_SIMPLE_STATUS_PROERR:
  case PROTO_SIMPLE_STATUS_GOODBYE:
    return -1;
   break;
 }

 return 0;
}

#define _CKPERM(p) if ( acclev < (p) ) return PROTO_SIMPLE_STATUS_NOTPERM

int proto_simple_parse(int client, char * line, size_t len) {
 enum   rpld_client_acclev    acclev = client_get_acclev(client);
 struct rpld_playlist_entry * plent;
 struct rpld_playlist       * cpl, * pl;
 char * cmd = line;
 char * args;
 char * tmp;
 float tmpf;
 size_t i;
 int mode;
 ssize_t pos = -2;

 if ( (args = strstr(cmd, " ")) != NULL ) {
  *args = 0;
   args++;
 }

 ROAR_DBG("proto_simple_parse(client=%i, ...): cmd='%s'", client, cmd);

 if ( !strcasecmp(cmd, "noop") ) {
  _CKPERM(ACCLEV_CONCTL);
  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "help") ) {
  _CKPERM(ACCLEV_CONCTL);
  for (i = 0; proto_simple_help_text[i] != NULL; i++) {
   if ( proto_simple_client_puts(client, proto_simple_help_text[i]) == -1 )
    return PROTO_SIMPLE_STATUS_PROERR;
  }

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "quit") ) {
  _CKPERM(ACCLEV_CONCTL);
  return PROTO_SIMPLE_STATUS_GOODBYE;


 } else if ( !strcasecmp(cmd, "auth") ) {
  _CKPERM(ACCLEV_CONCTL);
/* "  AUTH [TO AUTHLEV {n|\"name\"}] USING authtype...\n" */
  return PROTO_SIMPLE_STATUS_PROERR;
 } else if ( !strcasecmp(cmd, "unauth") ) {
  _CKPERM(ACCLEV_CONCTL);
/* "  UNAUTH [ACCLEV] {BY n|TO {n|\"name\"}}\n" */

  if ( !strncasecmp(args, "acclev ", 7) )
   args += 7;

  if ( !strncmp(args, "to ", 3) ) {
   args += 3;
   if ( *args == '"' ) {
    args++;
    for (i = 0; args[i] != 0; i++) {
     if ( args[i] == '"' ) {
      args[i] = 0;
      break;
     }
    }
    if ( client_set_acclev(client, client_str2accleacclev(args), client) == -1 ) {
     return PROTO_SIMPLE_STATUS_NOTPERM;
    } else {
     return PROTO_SIMPLE_STATUS_OK;
    }
   } else {
    if ( client_set_acclev(client, atoi(args), client) == -1 ) {
     return PROTO_SIMPLE_STATUS_NOTPERM;
    } else {
     return PROTO_SIMPLE_STATUS_OK;
    }
   }
  } else if ( !strncmp(args, "by ", 3) ) {
   args += 3;
   if ( client_inc_acclev(client, -1*atoi(args), client) == -1 ) {
    return PROTO_SIMPLE_STATUS_NOTPERM;
   } else {
    return PROTO_SIMPLE_STATUS_OK;
   }
  }

  return PROTO_SIMPLE_STATUS_PROERR;


 } else if ( !strcasecmp(cmd, "play") ) {
  _CKPERM(ACCLEV_USER);
  if ( playback_play() != -1 )
   return PROTO_SIMPLE_STATUS_OK;
  return PROTO_SIMPLE_STATUS_ERROR;
 } else if ( !strcasecmp(cmd, "stop") ) {
  _CKPERM(ACCLEV_USER);
  if ( playback_stop(0, 1) != -1 )
   return PROTO_SIMPLE_STATUS_OK;
  return PROTO_SIMPLE_STATUS_ERROR;
 } else if ( !strcasecmp(cmd, "next") ) {
  _CKPERM(ACCLEV_USER);
  if ( playback_next() != -1 )
   return PROTO_SIMPLE_STATUS_OK;
  return PROTO_SIMPLE_STATUS_ERROR;
 } else if ( !strcasecmp(cmd, "prev") ) {
  _CKPERM(ACCLEV_USER);
  if ( playback_prev() != -1 )
   return PROTO_SIMPLE_STATUS_OK;
  return PROTO_SIMPLE_STATUS_ERROR;

 } else if ( !strcasecmp(cmd, "pause") ) {
  _CKPERM(ACCLEV_USER);
  if ( args == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( !strcasecmp(args, "true") ) {
   mode = PLAYBACK_PAUSE_TRUE;
  } else if ( !strcasecmp(args, "false") ) {
   mode = PLAYBACK_PAUSE_FALSE;
  } else if ( !strcasecmp(args, "toggle") ) {
   mode = PLAYBACK_PAUSE_TOGGLE;
  } else {
   return PROTO_SIMPLE_STATUS_ERROR;
  }

  if ( playback_pause(mode) != -1 )
   return PROTO_SIMPLE_STATUS_OK;

  return PROTO_SIMPLE_STATUS_ERROR;

 } else if ( !strcasecmp(cmd, "isplaying") ) {
  _CKPERM(ACCLEV_GUEST);
  return playback_is_playing() ? PROTO_SIMPLE_STATUS_YES : PROTO_SIMPLE_STATUS_NO;
 } else if ( !strcasecmp(cmd, "showidentifier") ) {
  _CKPERM(ACCLEV_GUEST);
  if ( (args = playback_stream_identifier()) == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  // TODO: do some cleanup here:
  if ( proto_simple_client_puts(client, args) == -1 )
   return PROTO_SIMPLE_STATUS_PROERR;
  if ( proto_simple_client_puts(client, "\n") == 1 )
   return PROTO_SIMPLE_STATUS_ERROR;

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "showplaying") ) {
  _CKPERM(ACCLEV_GUEST);
  return proto_simple_showplaying(client);

 } else if ( !strcasecmp(cmd, "add2q") ) {
  _CKPERM(ACCLEV_USER);
  if ( proto_simple_parse_plent(&plent, args) == -1 )
   return PROTO_SIMPLE_STATUS_ERROR;

  rpld_pl_push(g_queue, plent);
  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "listq") || !strcasecmp(cmd, "listple") ) {
  _CKPERM(ACCLEV_GUEST);
  if ( args == NULL ) {
   if ( !strcasecmp(cmd, "listq") ) {
    pl = g_queue;
   } else {
    pl = rpld_pl_get_by_id(client_get_playlist(client));
   }
  } else {
   pl = proto_simple_parse_pli(args);
  }

  if ( pl == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  plent = rpld_pl_get_first(pl);

  while (plent != NULL) {
   proto_simple_print_plent(client, plent);
   plent = plent->list.next;
  }

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "showcur") || !strcasecmp(cmd, "showple") ) {
  _CKPERM(ACCLEV_GUEST);
  if ( args == NULL ) {
   if ( !strcasecmp(cmd, "showcur") ) {
    plent = rpld_pl_get_first(g_queue);
   } else {
    return PROTO_SIMPLE_STATUS_ERROR;
   }
  } else {
   plent = proto_simple_parse_ple_bopts(client, args, 0);
   if ( plent == NULL )
    return PROTO_SIMPLE_STATUS_ERROR;
  }

  if ( plent != NULL )
    proto_simple_print_plent(client, plent);

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "delfq") ) {
  _CKPERM(ACCLEV_USER);
  return rpld_pl_delent(g_queue, atoll(args)) == -1 ? PROTO_SIMPLE_STATUS_ERROR : PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "flushq") ) {
  _CKPERM(ACCLEV_USER);
  if ( playback_stop(0, 1) == -1 )
   return PROTO_SIMPLE_STATUS_ERROR;

  rpld_pl_flush(g_queue);
  return PROTO_SIMPLE_STATUS_OK;

 } else if ( !strcasecmp(cmd, "setvolume") ) {
  _CKPERM(ACCLEV_USER);
  if ( args == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( (len = strlen(args)) == 0 )
   return PROTO_SIMPLE_STATUS_ERROR;

  switch (args[len-1]) {
   case '%':
     args[len-1] = 0;
     if ( playback_set_volume_mpc(atoi(args)) == -1 )
      return PROTO_SIMPLE_STATUS_ERROR;
    break;
   default:
     if ( playback_set_volume_mu16(atoi(args)) == -1 )
      return PROTO_SIMPLE_STATUS_ERROR;
    break;
  }

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "showvolume") ) {
  _CKPERM(ACCLEV_GUEST);
  return proto_simple_showvolume(client);

 } else if ( !strcasecmp(cmd, "listplaylists") ) {
  _CKPERM(ACCLEV_GUEST);
  for (i = 0; i < MAX_PLAYLISTS; i++) {
   pl = rpld_pl_get_by_id(i);
   if ( pl != NULL )
    proto_simple_print_playlist(client, pl);
  }
  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "showlist") ) {
  _CKPERM(ACCLEV_GUEST);
  if ( args == NULL ) {
   pl = rpld_pl_get_by_id(client_get_playlist(client));
  } else {
   pl = proto_simple_parse_pli(args);
  }

  if ( pl == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  proto_simple_print_playlist(client, pl);
  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "delplaylist") ) {
  _CKPERM(ACCLEV_USER);
  if ( args == NULL ) {
   pl = rpld_pl_get_by_id(client_get_playlist(client));
  } else {
   pl = proto_simple_parse_pli(args);
  }

  if ( pl == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( pl == g_queue || pl == g_history )
   return PROTO_SIMPLE_STATUS_ERROR;

  rpld_pl_free(pl);
  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "addplaylist") ) {
  _CKPERM(ACCLEV_USER);
  if ( args == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( *args != '"' )
   return PROTO_SIMPLE_STATUS_ERROR;

  args++;
  i = strlen(args);
  if ( args[i-1] != '"' )
   return PROTO_SIMPLE_STATUS_ERROR;

  args[i-1] = 0;

  pl = rpld_pl_new();

  if ( pl == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  // TODO: check for errors:
  rpld_pl_set_name(pl, args);

  rpld_pl_register(pl);

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "flushplaylist") ) {
  _CKPERM(ACCLEV_USER);
  if ( args == NULL ) {
   pl = rpld_pl_get_by_id(client_get_playlist(client));
  } else {
   pl = proto_simple_parse_pli(args);
  }

  if ( pl == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  rpld_pl_flush(pl);

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "setplaylist") ) {
  _CKPERM(ACCLEV_GUEST);
  if ( args == NULL ) {
   pl = rpld_pl_get_by_id(client_get_playlist(client));
  } else {
   pl = proto_simple_parse_pli(args);
  }

  if ( pl == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( client_set_playlist(client, rpld_pl_get_id(pl)) == -1 )
   return PROTO_SIMPLE_STATUS_ERROR;

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "setparentlist") ) {
  _CKPERM(ACCLEV_USER);
  if ( args == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  i = 0;

  if ( *args == 'O' || *args == 'o' ) {
   i    = 1;
   args = strstr(args, " ");

   if ( args == NULL )
    return PROTO_SIMPLE_STATUS_ERROR;
  }

  if ( i ) {
   cpl = proto_simple_parse_pli(args);
   args = strstr(args, " ");
   if ( args == NULL )
    return PROTO_SIMPLE_STATUS_ERROR;
  } else {
   cpl = rpld_pl_get_by_id(client_get_playlist(client));
  }

  if ( cpl == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( *args == 'T' || *args == 't' ) {
   args = strstr(args, " "); 
   if ( args == NULL )
    return PROTO_SIMPLE_STATUS_ERROR;
  }

  pl = proto_simple_parse_pli(args);

  if ( pl == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  rpld_pl_set_parent(cpl, rpld_pl_get_id(pl));

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "copyple") || !strcasecmp(cmd, "queueple") ) {
  _CKPERM(ACCLEV_USER);
  if ( args == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( (plent = proto_simple_parse_ple_bopts(client, args, 0)) == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( !strcasecmp(cmd, "queueple") ) {
   pl = g_queue;
  } else {
   if ( (tmp = strstr(args, " TO ")) == NULL )
    if ( (tmp = strstr(args, " to ")) == NULL )
     return PROTO_SIMPLE_STATUS_ERROR;

   args = tmp + 4;

   pl = proto_simple_parse_pli(args);

   if ( pl == NULL )
    return PROTO_SIMPLE_STATUS_ERROR;
  }

  tmp = NULL;

  if ( (tmp = strstr(args, " AT ")) == NULL )
   tmp = strstr(args, " at ");

  if ( tmp != NULL ) {
   args = tmp + 4;
   pos = atoi(args);
  }

  if ( (plent = rpld_ple_copy(plent)) == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( pos == -2 ) {
   rpld_pl_push(pl, plent);
  } else {
   if ( pos == -1 && pl == g_queue && playback_is_playing() ) {
    playback_stop(0, 1);
    rpld_pl_add(pl, plent, pos);
    playback_play();
   } else {
    rpld_pl_add(pl, plent, pos);
   }
  }
  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "moveple") ) {
  _CKPERM(ACCLEV_USER);
  if ( args == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( (tmp = strstr(args, " TO ")) == NULL )
   if ( (tmp = strstr(args, " to ")) == NULL )
    return PROTO_SIMPLE_STATUS_ERROR;

  pl = proto_simple_parse_pli(tmp+4);

  if ( pl == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( (plent = proto_simple_parse_ple_bopts(client, args, 1)) == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( (plent = rpld_ple_cut_out_next(plent)) == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  tmp = NULL;

  if ( (tmp = strstr(args, " AT ")) == NULL )
   tmp = strstr(args, " at ");

  if ( tmp != NULL ) {
   args = tmp + 4;
   pos = atoi(args);
  }

  if ( pos == -2 ) {
   rpld_pl_push(pl, plent);
  } else {
   if ( pos == -1 && pl == g_queue && playback_is_playing() ) {
    playback_stop(0, 1);
    rpld_pl_add(pl, plent, pos);
    playback_play();
   } else {
    rpld_pl_add(pl, plent, pos);
   }
  }
  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "delple") ) {
  _CKPERM(ACCLEV_USER);
  if ( args == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  ROAR_DBG("proto_simple_parse(*) = ?");

  if ( (plent = proto_simple_parse_ple_bopts(client, args, 1)) == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  ROAR_DBG("proto_simple_parse(*) = ?");

  if ( (plent = rpld_ple_cut_out_next(plent)) == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  ROAR_DBG("proto_simple_parse(*) = ?");

  rpld_ple_free(plent);

  return PROTO_SIMPLE_STATUS_OK;

 } else if ( !strcasecmp(cmd, "like") || !strcasecmp(cmd, "dislike") ) {
  if ( args == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( (plent = proto_simple_parse_ple_bopts(client, args, 0)) == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  if ( (tmp = strrchr(args, ' ')) == NULL )
   return PROTO_SIMPLE_STATUS_ERROR;

  tmpf = atof(tmp+1);

  if ( !strcasecmp(cmd, "dislike") )
   tmpf *= -1;

  plent->likeness += tmpf;

  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "import") || !strcasecmp(cmd, "export") ) {
  _CKPERM(ACCLEV_USER);
  return proto_simple_import_export(client, args, !strcasecmp(cmd, "export"));

 } else if ( !strcasecmp(cmd, "store") ) {
  _CKPERM(ACCLEV_ALL);
  if ( store() == -1 )
   return PROTO_SIMPLE_STATUS_ERROR;
  return PROTO_SIMPLE_STATUS_OK;
 } else if ( !strcasecmp(cmd, "restore") ) {
  _CKPERM(ACCLEV_ALL);
  if ( restore() == -1 )
   return PROTO_SIMPLE_STATUS_ERROR;
  return PROTO_SIMPLE_STATUS_OK;

 } else if ( !strcasecmp(cmd, "setpointer") ) {
  _CKPERM(ACCLEV_USER);
  return proto_simple_setpointer(client, args);
 } else if ( !strcasecmp(cmd, "unsetpointer") ) {
  _CKPERM(ACCLEV_USER);
  return proto_simple_unsetpointer(client, args);
 } else if ( !strcasecmp(cmd, "showpointer") ) {
  _CKPERM(ACCLEV_GUEST);
  if ( args == NULL ) {
   for (i = 0; i < sizeof(g_pointer)/sizeof(*g_pointer); i++) {
    if ( proto_simple_showpointer(client, g_pointer[i].name, g_pointer[i].pointer) != PROTO_SIMPLE_STATUS_OK )
     return PROTO_SIMPLE_STATUS_ERROR;
   }
   return PROTO_SIMPLE_STATUS_OK;
  } else {
   for (i = 0; i < sizeof(g_pointer)/sizeof(*g_pointer); i++) {
    if ( !strcasecmp(args, g_pointer[i].name) ) {
     return proto_simple_showpointer(client, g_pointer[i].name, g_pointer[i].pointer);
    }
   }
  }
 }

 // unknown command
 return PROTO_SIMPLE_STATUS_PROERR;
}

int proto_simple_parse_plent(struct rpld_playlist_entry ** plent, char * txt) {
 struct rpld_playlist_entry * plent_new = rpld_ple_new();
 char * curtok;
//TYPE=LENGTH=ALBUM=TITLE=ARTIST=VERSION=FILE

 if ( plent_new == NULL )
  return -1;

 // TYPE aka CODEC:
 if ( (curtok = strtok(txt, "=")) == NULL ) {
  rpld_ple_free(plent_new);
  return -1;
 }

 plent_new->codec = roar_str2codec(curtok);

 // LENGTH, we ignore this
 if ( (curtok = strtok(NULL, "=")) == NULL ) {
  rpld_ple_free(plent_new);
  return -1;
 }

 if ( strstr(curtok, ":") == NULL ) {
  // parse SS
  plent_new->length = atol(curtok);
 } else {
  // parse HH:MM:SS
  rpld_ple_set_time_hr(plent_new, curtok);
 }

 // ALBUM
 if ( (curtok = strtok(NULL, "=")) == NULL ) {
  rpld_ple_free(plent_new);
  return -1;
 }
 strncpy(plent_new->meta.album, curtok, RPLD_MAX_PLF_LEN);
 plent_new->meta.album[RPLD_MAX_PLF_LEN-1] = 0;

 // TITLE
 if ( (curtok = strtok(NULL, "=")) == NULL ) {
  rpld_ple_free(plent_new);
  return -1;
 }
 strncpy(plent_new->meta.title, curtok, RPLD_MAX_PLF_LEN);
 plent_new->meta.title[RPLD_MAX_PLF_LEN-1] = 0;

 // ARTIST
 if ( (curtok = strtok(NULL, "=")) == NULL ) {
  rpld_ple_free(plent_new);
  return -1;
 }
 strncpy(plent_new->meta.artist, curtok, RPLD_MAX_PLF_LEN);
 plent_new->meta.artist[RPLD_MAX_PLF_LEN-1] = 0;

 // PERFORMER
 if ( (curtok = strtok(NULL, "=")) == NULL ) {
  rpld_ple_free(plent_new);
  return -1;
 }
 strncpy(plent_new->meta.performer, curtok, RPLD_MAX_PLF_LEN);
 plent_new->meta.performer[RPLD_MAX_PLF_LEN-1] = 0;

 // VERSION
 if ( (curtok = strtok(NULL, "=")) == NULL ) {
  rpld_ple_free(plent_new);
  return -1;
 }
 strncpy(plent_new->meta.version, curtok, RPLD_MAX_PLF_LEN);
 plent_new->meta.version[RPLD_MAX_PLF_LEN-1] = 0;


 // FILE
 if ( (curtok = strtok(NULL, "=")) == NULL ) {
  rpld_ple_free(plent_new);
  return -1;
 }
 strncpy(plent_new->io.filename, curtok, RPLD_MAX_PLF_LEN);
 plent_new->io.filename[RPLD_MAX_PLF_LEN-1] = 0;


 *plent = plent_new;

 return 0;
}

int proto_simple_print_plent(int client, const struct rpld_playlist_entry * plent) {
 char buffer[1024];
#ifdef HAVE_LIB_UUID
 char uuid[37];
#endif

#ifdef HAVE_LIB_UUID
 uuid_unparse(plent->uuid, uuid);
#endif

//TYPE=LENGTH=ALBUM=TITLE=ARTIST=VERSION=FILE
 snprintf(buffer, 1023,
#ifdef HAVE_LIB_UUID
          "%s=%s=%s=%s=%s=%s=%s=%s=long:0x%.16LX/short:0x%.8X/uuid:%s=0x%.8x/%i=%s(0x%x)=%f\n",
#else
          "%s=%s=%s=%s=%s=%s=%s=%s=long:0x%.16LX/short:0x%.8X=0x%.8x/%i=%s(0x%x)=%f\n",
#endif
                        roar_codec2str(plent->codec),
                        rpld_ple_time_hr(plent),
                        plent->meta.album,
                        plent->meta.title,
                        plent->meta.artist,
                        plent->meta.performer,
                        plent->meta.version,
                        plent->io.filename,
                        plent->global_tracknum,
                        plent->global_short_tracknum,
#ifdef HAVE_LIB_UUID
                        uuid,
#endif
                        plent->meta.discid,
                        plent->meta.tracknum,
                        roar_meta_strgenre(plent->meta.genre),
                        plent->meta.genre,
                        (double)plent->likeness
         );
 buffer[1023] = 0;

 return proto_simple_client_puts(client, buffer);
}

struct rpld_playlist * proto_simple_parse_pli(char * txt) {
 struct rpld_playlist * ret = NULL;
 char * end;
 char saved;

 if ( txt == NULL )
  return NULL;

 ROAR_DBG("proto_simple_parse_pli(txt='%s') = ?", txt);

 if ( *txt == '"' ) {
  if ( (end = strstr(txt+1, "\"")) == NULL )
   return NULL;

  saved = *end;
  *end  = 0;

  ret   = rpld_pl_get_by_name(txt+1);

  *end  = saved;
 } else {
  end = strstr(txt+1, " ");

  if ( end != NULL ) {
   saved = *end;
   *end  = 0;
  }

  ret   = rpld_pl_get_by_id(atoi(txt));

  if ( end != NULL )
   *end  = saved;
 }

 return ret;
}


int proto_simple_print_playlist (int client, struct rpld_playlist * pl) {
 char buf[1024];

 if ( pl == NULL )
  return -1;

 snprintf(buf, sizeof(buf), "%3i: [parent: %i] \"%s\"\n", rpld_pl_get_id(pl), rpld_pl_get_parent(pl), rpld_pl_get_name(pl));
 buf[sizeof(buf)-1] = 0;

 proto_simple_client_puts(client, buf);

 return 0;
}

int proto_simple_parse_ple_bopts_noeval(int client, char * txt, int need_prev, int * try_any, struct rpld_playlist ** pl, struct rpld_playlist_search * pls) {
 struct rpld_playlist_pointer * pointer = NULL;
 char * ei   = txt;
 char * from;
 char * pli  = NULL;
 char   from_save = 0;
 size_t i;

 if ( txt == NULL || client == -1 )
  return -1;

 if ( (from = strstr(ei, " ")) != NULL ) {
  if ( !strncasecmp(from, " from ", 6) ) {
   pli = strstr(from+1, " ");
  }
  from_save = *from;
  *from = 0;
 }

 if ( pli == NULL ) {
  *pl = rpld_pl_get_by_id(client_get_playlist(client));
 } else {
  pli++;
  ROAR_DBG("proto_simple_parse_ple_bopts(*): pli='%s'", pli);
  if ( !strcasecmp(pli, "any") || !strncasecmp(pli, "any ", 4) ) {
   *pl = NULL;
   *try_any = 1;
  } else {
   *pl = proto_simple_parse_pli(pli);
  }
 }

 if ( *pl == NULL && !*try_any ) {
  if ( from != NULL )
   *from = from_save;
  return -1;
 }

 ROAR_DBG("proto_simple_parse_ple_bopts(*): ei='%s', from='%s', pli='%s'", ei, from, pli);

 memset(pls, 0, sizeof(*pls));

 if ( need_prev )
  pls->options |= RPLD_PL_SO_PREVIOUS;

 if ( !strncmp(ei, "short:", 6) ) {
  if ( sscanf(ei+6, "0x%8X", &(pls->subject.short_tracknum)) != 1 )
   pls->subject.short_tracknum = 0;
  pls->type = RPLD_PL_ST_TRACKNUM_SHORT;
  ROAR_DBG("proto_simple_parse_ple_bopts(*): pls->subject.short_tracknum=0x%.8X", pls->subject.short_tracknum);
 } else if ( !strncmp(ei, "long:", 5) || *ei == '0' ) {
  if ( *ei == 'l' )
   ei += 5;

  if ( sscanf(ei, "0x%16llX", &(pls->subject.long_tracknum)) != 1 )
   pls->subject.long_tracknum = 0;

  pls->type = RPLD_PL_ST_TRACKNUM_LONG;
  ROAR_DBG("proto_simple_parse_ple_bopts(*): pls->subject.long_tracknum=0x%.16llX", pls->subject.long_tracknum);
 } else if ( !strncmp(ei, "pointer:", 8) ) {
  for (i = 0; i < sizeof(g_pointer)/sizeof(*g_pointer); i++) {
   if ( !strcasecmp(ei+8, g_pointer[i].name) ) {
    pointer = g_pointer[i].pointer;
   }
  }

  if ( pointer == NULL ) {
   if ( from != NULL )
    *from = from_save;
   return -1;
  } else {
   if ( pointer->pls.type == RPLD_PL_ST_RANDOM || pointer->pls.type == RPLD_PL_ST_RANDOM_LIKENESS ) {
    *pl = rpld_pl_get_by_id(pointer->pls.subject.pl);
   } else {
    *pl = pointer->hint.pl;
   }
   if ( *pl == NULL )
    *try_any = 1;
   memcpy(pls, &(pointer->pls), sizeof(*pls));
  }

#ifdef HAVE_LIB_UUID
 } else if ( !strncmp(ei, "uuid:", 5) ) {
  if ( uuid_parse(ei+5, pls->subject.uuid) != 0 )
   uuid_clear(pls->subject.uuid);

  pls->type = RPLD_PL_ST_UUID;
#endif
 } else if ( !strncmp(ei, "num:", 4) ) {
  pls->subject.num = (size_t)atoi(ei+4);
  pls->type = RPLD_PL_ST_NUM;
 } else if ( !strncmp(ei, "likeness:", 9) ) {
  pls->subject.likeness = (float)atof(ei+9);
  pls->type = RPLD_PL_ST_LIKENESS;
 } else if ( !strncmp(ei, "random:", 7) ) {
  pls->subject.pl = atoi(ei+7);
  if ( *pl != NULL && pls->subject.pl == 0 ) {
   pls->subject.pl = rpld_pl_get_id(*pl);
  }
  *pl = rpld_pl_get_by_id(pls->subject.pl);
  pls->type = RPLD_PL_ST_RANDOM;
 } else if ( !strncmp(ei, "randomlike:", 11) ) {
  pls->subject.pl = atoi(ei+11);
  if ( *pl != NULL && pls->subject.pl == 0 ) {
   pls->subject.pl = rpld_pl_get_id(*pl);
  }
  *pl = rpld_pl_get_by_id(pls->subject.pl);
  pls->type = RPLD_PL_ST_RANDOM_LIKENESS;
 } else {
  ROAR_WARN("proto_simple_parse_ple_bopts(*): Unknown PLE format: %s", ei);
  if ( from != NULL )
   *from = from_save;
  return -1;
 }

 if ( from != NULL )
  *from = from_save;

 return 0;
}

struct rpld_playlist_entry * proto_simple_parse_ple_bopts_eval(int try_any, struct rpld_playlist * pl, struct rpld_playlist_search * pls) {
 struct rpld_playlist_entry *   ret = NULL;
 size_t i;

 if ( try_any ) {
  for (i = 0; i < MAX_PLAYLISTS; i++) {
   if ( g_playlists[i] != NULL ) {
    if ( (ret = rpld_pl_search(g_playlists[i], pls, NULL)) != NULL )
     break;
   }
  }
 } else {
  ret = rpld_pl_search(pl, pls, NULL);
 }

 return ret;
}

struct rpld_playlist_entry * proto_simple_parse_ple_bopts(int client, char * txt, int need_prev) {
 struct rpld_playlist_search    pls;
 struct rpld_playlist         * pl  = NULL;
 int    try_any = 0;

 if ( proto_simple_parse_ple_bopts_noeval(client, txt, need_prev, &try_any, &pl, &pls) == -1 ) {
  return NULL;
 }

 return proto_simple_parse_ple_bopts_eval(try_any, pl, &pls);
}

int proto_simple_import_export (int client, char * args, int is_export) {
 struct roar_vio_defaults def;
 struct fformat_handle  * handle;
 struct roar_vio_calls    vio;
 struct rpld_playlist   * pl = NULL;
 int format;
 char * ftpos;
 char * tpos;
 const char * delm;
 int ret;

/*
  IMPORT [{"Name"|ID}] {TO|FROM} {STDIN|STDOUT|"Filename"} [AS {RPLD|PLS|M3U|VCLT|XSPF}]
  EXPORT [{"Name"|ID}] {TO|FROM} {STDIN|STDOUT|"Filename"} [AS {RPLD|PLS|M3U|VCLT|XSPF}]
*/

 ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

 if ( client == -1 || args == NULL )
  return PROTO_SIMPLE_STATUS_ERROR;

 if ( is_export ) {
  if ( roar_vio_dstr_init_defaults(&def, ROAR_VIO_DEF_TYPE_NONE, O_WRONLY|O_CREAT|O_TRUNC, 0644) == -1 )
   return PROTO_SIMPLE_STATUS_ERROR;
 } else {
  if ( roar_vio_dstr_init_defaults(&def, ROAR_VIO_DEF_TYPE_NONE, O_RDONLY, 0644) == -1 )
   return PROTO_SIMPLE_STATUS_ERROR;
 }

 ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

 if ( *args == 'f' || *args == 'F' ) {
  pl    = rpld_pl_get_by_id(client_get_playlist(client));
  args += 5;
 } else if ( *args == 't' || *args == 'T' ) {
  pl    = rpld_pl_get_by_id(client_get_playlist(client));
  args += 3;
 } else {
  pl = proto_simple_parse_pli(args);
  if ( (ftpos = strstr(args, " FROM ")) == NULL )
   if ( (ftpos = strstr(args, " from ")) == NULL )
    if ( (ftpos = strstr(args, " TO ")) == NULL )
     if ( (ftpos = strstr(args, " to ")) == NULL )
      return PROTO_SIMPLE_STATUS_ERROR;

  pl = proto_simple_parse_pli(args);
  args = ftpos + 1;
  if ( *ftpos == 'f' || *ftpos == 'F' ) {
   args += 5;
  } else {
   args += 3;
  }
 }

 if ( pl == NULL )
  return PROTO_SIMPLE_STATUS_ERROR;

 ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

 ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i): *args='%c'", client, args, is_export, *args);

 switch (*args) {
  case '\"': delm = "\""; break;
  case '\'': delm = "\'"; break;
  default:   delm = " ";  break;
 }

 ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i): delm='%s'", client, args, is_export, delm);

 if ( (tpos = strstr(*args == *delm ? args+1 : args, delm)) == NULL ) {
  format = RPLD_FF_PLF_DEFAULT;
 } else {
  *tpos = 0;
   tpos++;

  ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

  ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i): *tpos='%c'", client, args, is_export, *tpos);

  if ( *tpos == 0 ) {
   ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

   format = RPLD_FF_PLF_DEFAULT;
  } else {
   ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i): tpos='%s'", client, args, is_export, tpos);

   remove_spaces(&tpos);

   ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i): tpos='%s'", client, args, is_export, tpos);

   if ( strncasecmp(tpos, "as ", 3) )
    return PROTO_SIMPLE_STATUS_ERROR;

   tpos += 3;

   format = fformat_str2plf(tpos);
  }
 }

 if ( format == -1 )
  return PROTO_SIMPLE_STATUS_ERROR;

 ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

 if ( (handle = fformat_handle_new(format)) == NULL )
  return PROTO_SIMPLE_STATUS_ERROR;

 ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

 if ( !strcasecmp(args, "stdout") || !strcasecmp(args, "stdin") ) {
  if ( rpld_proto_simple_vio_open(&vio, client_get_vio(client)) == -1 ) {
   fformat_handle_close(handle);
   return PROTO_SIMPLE_STATUS_ERROR;
  }
 } else if ( *args == '"' ) {
  args++;

  ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

  if ( roar_vio_open_dstr(&vio, args, &def, 1) == -1 ) {
   fformat_handle_close(handle);
   return PROTO_SIMPLE_STATUS_ERROR;
  }

  ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

 } else {
  fformat_handle_close(handle);
  return PROTO_SIMPLE_STATUS_ERROR;
 }

 ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

 if ( is_export ) {
  ret = fformat_playlist_export(handle, &vio, pl);
 } else {
  ret = fformat_playlist_import(handle, &vio, pl);
 }

 roar_vio_close(&vio);

 fformat_handle_close(handle);

 ROAR_DBG("proto_simple_import_export(client=%i, args='%s', is_export=%i) = ?", client, args, is_export);

 return ret == -1 ? PROTO_SIMPLE_STATUS_ERROR : PROTO_SIMPLE_STATUS_OK;
}

int proto_simple_showvolume (int client) {
 char buf[80];

 snprintf(buf, 80, "VOLUME %i/65535 %i%%\n", playback_get_volume_mu16(), playback_get_volume_mpc());

 proto_simple_client_puts(client, buf);

 return PROTO_SIMPLE_STATUS_OK;
}

#define _BS  80
int proto_simple_showplaying (int client) {
 struct rpld_playlist_entry * plent;
 struct roar_stream         * s;
 char buf[_BS];
#ifdef HAVE_LIB_UUID
 char uuid[37];
#endif

 if ( !playback_is_playing() ) {
  proto_simple_client_puts(client, "STATE STOPPED\n");
  return PROTO_SIMPLE_STATUS_OK;
 }

 if ( playback_is_pause() == PLAYBACK_PAUSE_TRUE ) {
  proto_simple_client_puts(client, "STATE PAUSE\n");
 } else {
  proto_simple_client_puts(client, "STATE RUNNING\n");
 }

 plent = playback_get_ple();

 if ( plent == NULL )
  return PROTO_SIMPLE_STATUS_ERROR;

/*
 *           "%s=%s=%s=%s=%s=%s=%s=%s=long:0x%.16LX/short:0x%.8X/uuid:%s=0x%.8x/%i=%s(0x%x)\n",
 */
 snprintf(buf, _BS, "LONGGTN long:0x%.16LX\n", plent->global_tracknum);
 proto_simple_client_puts(client, buf);
 snprintf(buf, _BS, "SHORTGTN short:0x%.8X\n", plent->global_short_tracknum);
 proto_simple_client_puts(client, buf);

#ifdef HAVE_LIB_UUID
 uuid_unparse(plent->uuid, uuid);
 snprintf(buf, _BS, "UUID uuid:%s\n", uuid);
 proto_simple_client_puts(client, buf);
#endif

// send meta data:
 if ( *(plent->meta.album) != 0 ) {
  snprintf(buf, _BS, "META ALBUM \"%s\"\n", plent->meta.album);
  proto_simple_client_puts(client, buf);
 }

 if ( *(plent->meta.title) != 0 ) {
  snprintf(buf, _BS, "META TITLE \"%s\"\n", plent->meta.title);
  proto_simple_client_puts(client, buf);
 }

 if ( *(plent->meta.artist) != 0 ) {
  snprintf(buf, _BS, "META ARTIST \"%s\"\n", plent->meta.artist);
  proto_simple_client_puts(client, buf);
 }

 if ( *(plent->meta.performer) != 0 ) {
  snprintf(buf, _BS, "META PERFORMER \"%s\"\n", plent->meta.performer);
  proto_simple_client_puts(client, buf);
 }

 if ( *(plent->meta.version) != 0 ) {
  snprintf(buf, _BS, "META VERSION \"%s\"\n", plent->meta.version);
  proto_simple_client_puts(client, buf);
 }

 snprintf(buf, _BS, "MDUC %lu\n", (unsigned long int)playback_get_mduc());
 proto_simple_client_puts(client, buf);

// TODO: FIXME: add support for genre, tracknum and discid

 if ( (s = playback_get_stream()) == NULL )
  return PROTO_SIMPLE_STATUS_ERROR;

 if ( s->info.rate ) {
  snprintf(buf, _BS, "RATE %i\n", s->info.rate);
  proto_simple_client_puts(client, buf);
 }

 if ( s->info.channels ) {
  snprintf(buf, _BS, "CHANNELS %i\n", s->info.channels);
  proto_simple_client_puts(client, buf);
 }

 if ( s->info.bits ) {
  snprintf(buf, _BS, "BITS %i\n", s->info.bits);
  proto_simple_client_puts(client, buf);
 }

 if ( (int32_t) s->pos != -1 ) {
  if ( s->info.rate && s->info.channels ) {
   snprintf(buf, _BS, "TIME %lu S (%.3fs)\n", (unsigned long int) s->pos, (float)s->pos/(s->info.rate*s->info.channels));
  } else {
   snprintf(buf, _BS, "TIME %lu S\n", (unsigned long int) s->pos);
  }
  proto_simple_client_puts(client, buf);
 }

 return PROTO_SIMPLE_STATUS_OK;
}

int proto_simple_unsetpointer(int client, char * args) {
 struct rpld_playlist_pointer ** pointer = NULL;
 size_t i;

 (void)client;

 if ( args == NULL )
  return PROTO_SIMPLE_STATUS_ERROR;

 for (i = 0; i < sizeof(g_pointer)/sizeof(*g_pointer); i++) {
  if ( !strcasecmp(args, g_pointer[i].name) ) {
   pointer = &(g_pointer[i].pointer);
  }
 }

 if ( pointer == NULL ) {
  return PROTO_SIMPLE_STATUS_ERROR;
 }

 if ( *pointer != NULL ) {
  rpld_plp_uninit(*pointer);
  *pointer = NULL;
 }

 return PROTO_SIMPLE_STATUS_OK;
}

int proto_simple_setpointer(int client, char * args) {
 struct rpld_playlist         *  pl  = NULL;
 struct rpld_playlist_search     pls;
 struct rpld_playlist_entry   *  ple;
 struct rpld_playlist_pointer ** pointer = NULL;
 char * pointer_name;
 char * bopts;
 size_t i;
 int    try_any = 0;

//  SETPOINTER {CURRENT|DEFAULT} {long:0xLongID|0xLongID|short:0xShortID|uuid:UUID} [FROM {"Name"|ID}]

 if ( args == NULL )
  return PROTO_SIMPLE_STATUS_ERROR;

 pointer_name = args;

 if ( (bopts = strstr(args, " ")) == NULL )
  return PROTO_SIMPLE_STATUS_ERROR;

 *bopts = 0;
  bopts++;

 if ( !strncasecmp(bopts, "to ", 3) ) {
   bopts += 3;
 }

 if ( proto_simple_parse_ple_bopts_noeval(client, bopts, 0, &try_any, &pl, &pls) == -1 )
  return PROTO_SIMPLE_STATUS_ERROR;

 for (i = 0; i < sizeof(g_pointer)/sizeof(*g_pointer); i++) {
  if ( !strcasecmp(pointer_name, g_pointer[i].name) ) {
   pointer = &(g_pointer[i].pointer);
  }
 }

 if ( pointer == NULL ) {
  return PROTO_SIMPLE_STATUS_ERROR;
 }

 if ( *pointer != NULL ) {
  rpld_plp_uninit(*pointer);
  *pointer = NULL;
 }

 if ( !(pls.type == RPLD_PL_ST_RANDOM || pls.type == RPLD_PL_ST_RANDOM_LIKENESS) ) {
  ple = proto_simple_parse_ple_bopts_eval(try_any, pl, &pls);

  memset(&pls, 0, sizeof(pls));

#ifdef HAVE_LIB_UUID
  pls.type = RPLD_PL_ST_UUID;

  uuid_copy(pls.subject.uuid, ple->uuid);
#else
  pls.type = RPLD_PL_ST_TRACKNUM_LONG;

  pls.subject.long_tracknum = ple->global_tracknum;
#endif
 }

 *pointer = rpld_plp_init(NULL, &pls);

 if ( *pointer == NULL )
  return PROTO_SIMPLE_STATUS_ERROR;

 return PROTO_SIMPLE_STATUS_OK;
}

int proto_simple_showpointer(int client, const char * name, struct rpld_playlist_pointer * pointer) {
 char * text = NULL;
#ifdef HAVE_LIB_UUID
 char uuid[37];
#endif

 if ( client == -1 || name == NULL || *name == 0 )
  return PROTO_SIMPLE_STATUS_ERROR;

 if ( pointer == NULL ) {
  asprintf(&text, "POINTER %s NOT SET\n", name);
 } else {
  switch (pointer->pls.type) {
   case RPLD_PL_ST_TRACKNUM_LONG:
     asprintf(&text, "POINTER %s IS AT long:0x0x%.16LX\n", name, pointer->pls.subject.long_tracknum);
    break;
   case RPLD_PL_ST_TRACKNUM_SHORT:
     asprintf(&text, "POINTER %s IS AT short:0x0x%.8X\n", name, pointer->pls.subject.short_tracknum);
    break;
#ifdef HAVE_LIB_UUID
   case RPLD_PL_ST_UUID:
     uuid_unparse(pointer->pls.subject.uuid, uuid);
     asprintf(&text, "POINTER %s IS AT uuid:%s\n", name, uuid);
    break;
#endif
    case RPLD_PL_ST_RANDOM:
     asprintf(&text, "POINTER %s IS AT random:%i\n", name, pointer->pls.subject.pl);
    break;
    case RPLD_PL_ST_RANDOM_LIKENESS:
     asprintf(&text, "POINTER %s IS AT randomlike:%i\n", name, pointer->pls.subject.pl);
    break;
   default:
     return PROTO_SIMPLE_STATUS_ERROR;
  }
 }

 if ( proto_simple_client_puts(client, text) == -1 ) {
  free(text);
  return PROTO_SIMPLE_STATUS_ERROR;
 }

 free(text);

 return PROTO_SIMPLE_STATUS_OK;
}

//ll
