//playback.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2011
 *
 *  This file is part of RoarAudio PlayList Daemon,
 *  a playlist management daemon for RoarAudio.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "rpld.h"

static struct rpld_playback g_playback;
static int                  g_role = ROAR_ROLE_MUSIC;

int  playback_preinit(void) {
 static int inited = 0;
 int i;

 if ( inited )
  return 0;

 memset(&g_playback, 0, sizeof(g_playback));

 for (i = 0; i < PLAYBACK_MAX_CODECBL; i++) {
  g_playback.backend.roaraudio.codecbl[i] = -1;
 }

 inited = 1;
 return 0;
}

int  playback_init(const char * server) {
 ROAR_DBG("playback_init(server='%s') = ?", server);

 if ( playback_preinit() == -1 )
  return -1;

 ROAR_DBG("playback_init(server='%s') = ?", server);

 g_playback.playing = 0;

 g_playback.volume.mono = 65535;

 memset(&(g_playback.ple), 0, sizeof(g_playback.ple));
 g_playback.ple_mduc = 0;

 ROAR_DBG("playback_init(server='%s') = ?", server);

 if ( roar_simple_connect(&(g_playback.backend.roaraudio.con), server, "RoarAudio PlayList daemon") == -1 )
  return -1;

 ROAR_DBG("playback_init(server='%s') = 0", server);

 return 0;
}

void playback_uninit(void) {
 playback_stop(0, 1);

 roar_disconnect(&(g_playback.backend.roaraudio.con));
}

int  playback_blacklist_codec(int codec) {
 int i;

//   int                        codecbl[PLAYBACK_MAX_CODECBL];
 for (i = 0; i < PLAYBACK_MAX_CODECBL; i++) {
  if ( g_playback.backend.roaraudio.codecbl[i] == -1 ) {
   g_playback.backend.roaraudio.codecbl[i] = codec;
   return 0;
  }
 }

 return -1;
}

void playback_set_role(const int role) {
 g_role = role;
}

static int playback_blacklist_check_codec(int codec) {
 int i;

 if ( codec == -1 )
  return 1;

 for (i = 0; i < PLAYBACK_MAX_CODECBL; i++) {
  if ( g_playback.backend.roaraudio.codecbl[i] == codec ) {
   return 1;
  }
 }

 return 0;
}

void playback_check(int block) {
#ifndef ROAR_TARGET_WIN32
 struct timespec ts;
#endif
 struct roar_vio_select vioset[2];
 struct roar_vio_selecttv tv;
 ssize_t len;
 char buf[2*1024];
 int is_playing;

#ifdef ROAR_TARGET_WIN32
 if ( block == RPLD_YIELD_NONBLOCK ) {
  // we do not support non-blocking checks as win32 is totally broken. See below.
  return;
 }
#endif

 if ( block == RPLD_YIELD_NONBLOCK ) {
  block = 0;
 } else {
  block = 1;
 }

 if ( g_playback.playing ) {
  tv.sec  = 0;

  if ( block ) {
   tv.nsec = 10*1000*1000;
  } else {
   tv.nsec = 32*1000;
  }

  ROAR_VIO_SELECT_SETVIO(&(vioset[0]), &(g_playback.io.stream), ROAR_VIO_SELECT_WRITE);
  ROAR_VIO_SELECT_SETVIO(&(vioset[1]), g_playback.io.fp, ROAR_VIO_SELECT_READ);

  ROAR_DBG("playback_check(void) = (void)");

  // test if we can read and write stream data.
  // in blocking mode first just test for writing to
  // get sync with the server.
  if ( roar_vio_select(vioset, block ? 1 : 2, &tv, NULL) < (block ? 1 : 2) )
   return;

#ifndef ROAR_TARGET_WIN32
  // On win32 we can not do this check as we can not call select() on non-sockets.
  // This is because win32 has a very broken select().
  if ( block ) {
   // now retest if we can also read.
   tv.nsec = 32*1000;
   if ( roar_vio_select(vioset, 2, &tv, NULL) < 2 )
    return;
  }
#endif

  ROAR_DBG("playback_check(void) = (void)");

  if ( (len = roar_vio_read(g_playback.io.fp, buf, sizeof(buf))) < 1 ) {
   is_playing = playback_is_playing();

   playback_stop(0, 0);
   playback_next();

   if ( is_playing )
    playback_play();

   return;
  }

  if ( roar_vio_write(&(g_playback.io.stream), buf, len) != len ) {
   is_playing = playback_is_playing();

   playback_stop(0, 0);
   playback_next();

   if ( is_playing )
    playback_play();

   return;
  }
 } else {
#ifdef ROAR_TARGET_WIN32
  Sleep(10);
#else
  ts.tv_sec  = 0;
  ts.tv_nsec = 10L*1000000L;
  nanosleep(&ts, NULL);
#endif
 }
}

#if defined(HAVE_FUNC_UALARM) && defined(HAVE_FUNC_SIGINTERRUPT)
static void _sighandler_dummy(int sig) {
 (void)sig;
}
#endif

int  playback_play(void) {
 int mixer = -1; // TODO: FIXME: This should be configureable.
 struct rpld_playlist_entry * plent;
 int codec;
 ssize_t magic_len = 0;
 char magic[64];
#ifdef HAVE_LIB_UUID
 char uuid[37];
#endif
 const char * content_type;

 ROAR_DBG("playback_play(void) = ?");

 if ( g_playback.playing )
  return 0;

 ROAR_DBG("playback_play(void) = ?");

 if ( (plent = rpld_pl_get_first(g_queue)) == NULL ) {
  // retry with autoqueue
  if ( autoqueue() == -1 )
   return -1;
  if ( (plent = rpld_pl_get_first(g_queue)) == NULL )
   return -1;
 }

 ROAR_DBG("playback_play(void) = ?");

 if ( *(plent->io.filename) == 0 ) {
#ifdef HAVE_LIB_UUID
  uuid_unparse(plent->uuid, uuid);
  snprintf(plent->io.filename, sizeof(plent->io.filename), "tantalos://%s", uuid);
  plent->io.filename[sizeof(plent->io.filename)-1] = 0;
#else
  return -1;
#endif
 }

 ROAR_DBG("playback_play(void) = ?");

 yield_auto(RPLD_YIELD_PAUSE, 0, 0, 0);
#if defined(HAVE_FUNC_UALARM) && defined(HAVE_FUNC_SIGINTERRUPT)
 //signal(SIGALRM,  SIG_IGN);
 signal(SIGALRM,  _sighandler_dummy);
 siginterrupt(SIGALRM, 0);
 ualarm(1000*1000, 0);
#endif
 if ( roar_vio_open_dstr_simple(&(g_playback.io.file), plent->io.filename, ROAR_VIOF_READ|ROAR_VIOF_NONBLOCK) == -1 ) {
  ROAR_WARN("playback_play(void): Can not open file: %s: %s", plent->io.filename, roar_error2str(roar_error));
  yield_auto(RPLD_YIELD_UNPAUSE, 0, 0, 0);
  return -1;
 }
 yield_auto(RPLD_YIELD_UNPAUSE, 0, 0, 0);

 ROAR_DBG("playback_play(void) = ?");

 g_playback.io.fp = &(g_playback.io.file);

 /* TODO: FIXME: do a better check for streams than checking for :// ! */
 // try to guess the type by magic.
 if ( (codec = plent->codec) == -1 && strstr(plent->io.filename, "://") == NULL ) {
  if ( (magic_len = roar_vio_read(&(g_playback.io.file), magic, 64)) < 1 ) {
   roar_vio_close(&(g_playback.io.file));
   return -1;
  }

  codec = roar_file_codecdetect(magic, magic_len);

  if ( codec == -1 ) {
   if ( roar_vio_lseek(&(g_playback.io.file), 0, SEEK_SET) == (off_t)-1 ) {
    roar_vio_close(&(g_playback.io.file));
    return -1;
   }
  }
 }

 ROAR_DBG("playback_play(void): codec=%i", codec);

 // try to read the magic from filesystem or stream headers.
 if ( codec == -1 ) {
  if ( roar_vio_ctl(&(g_playback.io.file), ROAR_VIO_CTL_GET_MIMETYPE, &content_type) != -1 ) {
   codec = roar_mime2codec(content_type);
  }
 }

 ROAR_DBG("playback_play(void): codec=%i", codec);

 // test if we need the codec helper for this kind of stream.
 if ( playback_blacklist_check_codec(codec) ) {
  codec = -1;
 }

 // open stream...
 if ( codec != -1 ) {
  if ( roar_vio_simple_new_stream_obj(&(g_playback.io.stream),
                                      &(g_playback.backend.roaraudio.con),
                                      &(g_playback.backend.roaraudio.stream),
                                      ROAR_RATE_DEFAULT, ROAR_CHANNELS_DEFAULT, ROAR_BITS_DEFAULT,
                                      codec, ROAR_DIR_PLAY, mixer
                                      ) == -1 ) {
   ROAR_DBG("playback_play(void): codec=%i->-1", codec);
   codec = -1; // we try the helper in case roard does not know about this codec...
  }
 }

 ROAR_DBG("playback_play(void): codec=%i", codec);

 if ( codec == -1 ) {
  if ( roar_vio_simple_new_stream_obj(&(g_playback.io.stream),
                                      &(g_playback.backend.roaraudio.con),
                                      &(g_playback.backend.roaraudio.stream),
                                      ROAR_RATE_DEFAULT, ROAR_CHANNELS_DEFAULT, ROAR_BITS_DEFAULT,
                                      ROAR_CODEC_DEFAULT, ROAR_DIR_PLAY, mixer
                                      ) == -1 ) {
   roar_vio_close(&(g_playback.io.file));
   ROAR_DBG("playback_play(void) = -1");
   return -1;
  }
 }

 ROAR_DBG("playback_play(void) = ?");

 if ( codec == -1 ) {
  if ( roar_vio_open_cmd(&(g_playback.io.filter), &(g_playback.io.file),
                         "rpld-codec-helper", NULL,
                         0) == -1 ) {
   roar_vio_close(&(g_playback.io.file));
   roar_vio_close(&(g_playback.io.stream));
   ROAR_DBG("playback_play(void) = -1");
   return -1;
  }
  g_playback.io.fp = &(g_playback.io.filter);
 } else {
  if ( magic_len > 0 ) {
   if ( roar_vio_write(&(g_playback.io.stream), magic, magic_len) != magic_len ) {
    roar_vio_close(&(g_playback.io.file));
    roar_vio_close(&(g_playback.io.stream));
    ROAR_DBG("playback_play(void) = -1");
    return -1;
   }
  }
 }

 ROAR_DBG("playback_play(void) = ?");

 memcpy(&(g_playback.ple), plent, sizeof(g_playback.ple));
 g_playback.ple_mduc = 1;


 g_playback.playing = 1;

 // set role for new stream:
 if ( roar_stream_set_role(&(g_playback.backend.roaraudio.con),
                           &(g_playback.backend.roaraudio.stream),
                           g_role) == -1 ) {
  ROAR_WARN("playback_play(void): Can not set role for new stream to %s", roar_role2str(g_role));
 }

 // set volume for next track:
 playback_set_volume();

 ROAR_DBG("playback_play(void) = 0");
 return 0;
}

int  playback_next(void) {
 struct rpld_playlist_entry * old;
 int was_playing = g_playback.playing;
 // we implement this by stoping the current song,
 // hoping to the next and restart playing if we was playing.

 if ( playback_stop(0, 1) == -1 )
  return -1;

 old = rpld_pl_shift(g_queue);

 if ( old != NULL ) {
  rpld_pl_add(g_history, old, -1);

  while ( rpld_pl_num(g_history) > g_histsize ) {
   old = rpld_pl_pop(g_history);
   if ( old != NULL )
    rpld_ple_free(old);
  }
 }

 if ( was_playing )
  return playback_play();

 return 0;
}

int  playback_prev(void) {
 struct rpld_playlist_entry * old;
 int was_playing = g_playback.playing;

 old = rpld_pl_shift(g_history);

 if ( old == NULL )
  return -1;

 if ( playback_stop(0, 1) == -1 ) {
  rpld_pl_add(g_history, old, -1);
  return -1;
 }

 rpld_pl_add(g_queue, old, -1);

 if ( was_playing )
  return playback_play();

 return 0;
}

int  playback_stop(int offset, int kick) {
 if ( !g_playback.playing )
  return 0;

 roar_vio_close(&(g_playback.io.stream));
 roar_vio_close(g_playback.io.fp);

 roar_kick(&(g_playback.backend.roaraudio.con),
                    ROAR_OT_STREAM,
                    roar_stream_get_id(&(g_playback.backend.roaraudio.stream)));

 g_playback.playing = 0;

 memset(&(g_playback.ple), 0, sizeof(g_playback.ple));
 g_playback.ple_mduc = 0;

 ROAR_DBG("playback_stop(offset=%i, kick=%i) = 0", offset, kick);

 return 0;
}

int  playback_pause(int how) {
 struct roar_stream_info info;
 int reset = -1;

 if ( !g_playback.playing )
  return -1;

 switch (how) {
  case PLAYBACK_PAUSE_TRUE:
    reset = ROAR_SET_FLAG;
   break;
  case PLAYBACK_PAUSE_FALSE:
    reset = ROAR_RESET_FLAG;
   break;
  case PLAYBACK_PAUSE_TOGGLE:
#ifdef ROAR_FT_FUNC_SET_FLAGS2
#ifdef ROAR_FT_SONAME_LIBROAR2
#define _set_flags roar_stream_set_flags
#else
#define _set_flags roar_stream_set_flags2
#endif
    if ( _set_flags(&(g_playback.backend.roaraudio.con),
                    &(g_playback.backend.roaraudio.stream),
                    ROAR_FLAG_PAUSE, ROAR_TOGGLE_FLAG) == 0 ) {
     return 0;
    }
#endif

    if ( roar_stream_get_info(&(g_playback.backend.roaraudio.con),
                              &(g_playback.backend.roaraudio.stream),
                              &info) == -1 )
     return -1;
    if ( info.flags & ROAR_FLAG_PAUSE ) {
     reset = ROAR_RESET_FLAG;
    } else {
     reset = ROAR_SET_FLAG;
    }
   break;
 }

#ifdef ROAR_FT_FUNC_SET_FLAGS2
 return _set_flags(&(g_playback.backend.roaraudio.con),
                   &(g_playback.backend.roaraudio.stream),
                   ROAR_FLAG_PAUSE, reset);
#else
 return roar_stream_set_flags(&(g_playback.backend.roaraudio.con),
                              &(g_playback.backend.roaraudio.stream),
                              ROAR_FLAG_PAUSE, reset);
#endif
}

int  playback_is_playing(void) {
 return g_playback.playing;
}

char * playback_stream_identifier(void) {
 static char buf[32];

 if ( !g_playback.playing )
  return NULL;

 snprintf(buf, sizeof(buf), "%i", roar_stream_get_id(&(g_playback.backend.roaraudio.stream)));
 buf[sizeof(buf)-1] = 0;

 return buf;
}

int  playback_is_pause(void) {
 struct roar_stream_info info;

 if ( !g_playback.playing )
  return PLAYBACK_PAUSE_FALSE;

 if ( roar_stream_get_info(&(g_playback.backend.roaraudio.con),
                           &(g_playback.backend.roaraudio.stream),
                           &info) == -1 )
  return PLAYBACK_PAUSE_FALSE;

 if ( info.flags & ROAR_FLAG_PAUSE )
  return PLAYBACK_PAUSE_TRUE;

 return PLAYBACK_PAUSE_FALSE;
}

int  playback_set_volume(void) {
 int id;        // stream ID
 int channels;  // nummber of channels
 int i;

 // it's an error to enforce a volume in case we are not playing
 if ( !g_playback.playing )
  return -1;

 id = roar_stream_get_id(&(g_playback.backend.roaraudio.stream));

 if ( roar_get_vol(&(g_playback.backend.roaraudio.con), id, &(g_playback.backend.roaraudio.mixer), &channels) == -1 )
  return -1;

 g_playback.backend.roaraudio.mixer.scale   = 65535;
 g_playback.backend.roaraudio.mixer.rpg_mul = 1;
 g_playback.backend.roaraudio.mixer.rpg_div = 1;

 for (i = 0; i < channels; i++)
  g_playback.backend.roaraudio.mixer.mixer[i] = g_playback.volume.mono;

#ifdef ROAR_FT_SONAME_LIBROAR2
 if ( roar_set_vol(&(g_playback.backend.roaraudio.con), id,
                   &(g_playback.backend.roaraudio.mixer), channels, ROAR_SET_VOL_ALL) == -1 ) {
  channels = 1;
  if ( roar_set_vol(&(g_playback.backend.roaraudio.con), id,
                    &(g_playback.backend.roaraudio.mixer), channels, ROAR_SET_VOL_UNMAPPED) == -1 )
   return -1;
 }
#else
 if ( roar_set_vol(&(g_playback.backend.roaraudio.con), id, &(g_playback.backend.roaraudio.mixer), channels) == -1 )
  return -1;
#endif

 return 0;
}

int  playback_set_volume_mu16(uint16_t mono) {
 g_playback.volume.mono = mono;

 // if we are not playing we are done
 if ( !g_playback.playing )
  return 0;

 return playback_set_volume();
}

int  playback_set_volume_mpc(int pc) {
 uint16_t u16 = pc * 655;

 // correct maximum

 if ( u16 > 65480 )
  u16 = 65535;

 return playback_set_volume_mu16(u16);
}

uint16_t playback_get_volume_mu16(void) {
 uint32_t s = 0;
 int channels;  // nummber of channels
 int id;        // stream ID
 int i;

 if ( !g_playback.playing )
  return g_playback.volume.mono;

 id = roar_stream_get_id(&(g_playback.backend.roaraudio.stream));

 if ( roar_get_vol(&(g_playback.backend.roaraudio.con), id, &(g_playback.backend.roaraudio.mixer), &channels) == -1 )
  return g_playback.volume.mono;

 for (i = 0; i < channels; i++) {
  s += g_playback.backend.roaraudio.mixer.mixer[i];
 }

 if ( g_playback.backend.roaraudio.mixer.scale == 65535 ) {
  s = (float)s / (float)(g_playback.backend.roaraudio.mixer.scale * channels / 65535.);
 } else {
  s /= channels;
 }

 return g_playback.volume.mono = s;
}

int      playback_get_volume_mpc(void) {
 return playback_get_volume_mu16() / 655;
}

struct roar_stream * playback_get_stream (void) {
 static struct roar_stream stream;

 if ( roar_get_stream(&(g_playback.backend.roaraudio.con),
                      &stream,
                      roar_stream_get_id(&(g_playback.backend.roaraudio.stream))) == -1 )
  return NULL;

 return &stream;
}

struct rpld_playlist_entry * playback_get_ple (void) {
 struct rpld_playlist_entry * plent = &(g_playback.ple);
 struct roar_meta   meta;
 int types[ROAR_META_MAX_PER_STREAM];
 int i;
 int len;
 char * metadata;
 size_t metalen;

 if ( !g_playback.playing )
  return NULL;

 if ( (len = roar_stream_meta_list(&(g_playback.backend.roaraudio.con),
                                   &(g_playback.backend.roaraudio.stream),
                                   types, ROAR_META_MAX_PER_STREAM)) != -1 ) {
  for (i = 0; i < len; i++) {
   metalen   = RPLD_MAX_PLF_LEN;
   metadata  = NULL;
   switch (types[i]) {
    case ROAR_META_TYPE_ALBUM:
      metadata = plent->meta.album;
     break;
    case ROAR_META_TYPE_TITLE:
      metadata = plent->meta.title;
     break;
    case ROAR_META_TYPE_ARTIST:
      metadata = plent->meta.artist;
     break;
    case ROAR_META_TYPE_PERFORMER:
      metadata = plent->meta.performer;
     break;
    case ROAR_META_TYPE_VERSION:
      metadata = plent->meta.version;
     break;
/*
   TODO: FIXME: add support for this
   case ROAR_META_TYPE_TRACKNUMBER:
     plent->meta.tracknum = atoi(delm);
     continue;
    break;
   case ROAR_META_TYPE_GENRE:
     plent->meta.genre = roar_meta_intgenre(delm);
     continue;
    break;
   case ROAR_META_TYPE_DISCID:
     sscanf(delm, "%8x", &(plent->meta.discid));
     continue;
    break;
*/
    default:
      metalen = 0;
      continue;
     break;
   }

   if ( metalen != 0 ) {
    meta.type = types[i];
    if ( roar_stream_meta_get(&(g_playback.backend.roaraudio.con),
                              &(g_playback.backend.roaraudio.stream),
                              &meta) != -1 ) {
     g_playback.ple_mduc++;
     strncpy(metadata, meta.value, metalen);
     metadata[metalen-1] = 0;
     roar_meta_free(&meta);
    }
   }
  }
 }

 return plent;
}

size_t playback_get_mduc(void) {
 return g_playback.ple_mduc;
}
//ll
