//client.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2011
 *
 *  This file is part of RoarAudio PlayList Daemon,
 *  a playlist management daemon for RoarAudio.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "rpld.h"

static struct rpld_client {
 int proto;
 pli_t playlist;
 struct rpld_playlist_pointer * tmpptr;
 struct roar_vio_calls con;
 enum rpld_client_acclev acclev;
} g_clients[RPLD_CLIENTS_MAX];

void client_init(void) {
 int i;

 memset(g_clients, 0, sizeof(g_clients));

 for (i = 0; i < RPLD_CLIENTS_MAX; i++)
  g_clients[i].proto = RPLD_PROTO_UNUSED;
}

int  client_new(struct roar_vio_calls ** vio) {
 int i;

 if ( vio == NULL )
  return -1;

 for (i = 0; i < RPLD_CLIENTS_MAX; i++) {
  if ( g_clients[i].proto == RPLD_PROTO_UNUSED ) {
   g_clients[i].proto     =  RPLD_PROTO_SIMPLE;
   g_clients[i].playlist  =  0; /* 0=main queue */
   g_clients[i].tmpptr    =  NULL;
   g_clients[i].acclev    =  ACCLEV_ALL; // currently all clients have fill access rights
   roar_vio_clear_calls(&(g_clients[i].con));
   *vio = &(g_clients[i].con);
   return i;
  }
 }

 return -1;
}

int  client_delete(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( g_clients[id].proto == RPLD_PROTO_UNUSED )
  return 0;

 g_clients[id].proto = RPLD_PROTO_UNUSED;

 if ( g_clients[id].tmpptr != NULL ) {
  rpld_plp_uninit(g_clients[id].tmpptr);
  g_clients[id].tmpptr = NULL;
 }

 roar_vio_close(&(g_clients[id].con));

 return 0;
}

int  client_set_proto(int id, int proto) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 g_clients[id].proto = proto;

 switch (g_clients[id].proto) {
  case RPLD_PROTO_SIMPLE:
    return 0;
   break;
  case RPLD_PROTO_MPD:
    return proto_mpd_client_set_proto(id);
   break;
 }

 return -1;
}

struct roar_vio_calls * client_get_vio(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return NULL;

 if ( g_clients[id].proto == RPLD_PROTO_UNUSED )
  return NULL;

 return &(g_clients[id].con);
}

int  client_get_acclev(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 return g_clients[id].acclev;
}

int  client_set_acclev(int id, enum rpld_client_acclev acclev, int setter) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( setter != -1 ) {
  if ( setter != id )
   if ( setter < 0 || setter >= RPLD_CLIENTS_MAX )
    return -1;

  if ( acclev > g_clients[setter].acclev )
   return -1;
 }

 g_clients[id].acclev = acclev;

 return 0;
}

int  client_inc_acclev(int id, int inc, int setter) {
 int  newlev;
 enum rpld_client_acclev acclev;

 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( setter != id )
  if ( setter < 0 || setter >= RPLD_CLIENTS_MAX )
   return -1;

 newlev = (int)g_clients[id].acclev + inc;

 if ( newlev < 0 ) {
  acclev = ACCLEV_NONE;
 } else {
  acclev = newlev;
 }

 if ( acclev > g_clients[setter].acclev )
  return -1;

 g_clients[id].acclev = acclev;

 return 0;
}

int  client_set_playlist(int id, pli_t pl) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 g_clients[id].playlist = pl;

 return 0;
}

pli_t client_get_playlist(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 return g_clients[id].playlist;
}


int  client_handle(int id) {
 int ret = -1;

 ROAR_DBG("client_handle(id=%i) = ?", id);

 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( g_clients[id].proto == RPLD_PROTO_UNUSED )
  return -1;

 g_pointer[POINTER_TEMP].pointer = g_clients[id].tmpptr;

 switch (g_clients[id].proto) {
  case RPLD_PROTO_SIMPLE:
    ret = proto_simple_client_handle(id);
   break;
  case RPLD_PROTO_MPD:
    ret = proto_mpd_client_handle(id);
   break;
 }

 g_clients[id].tmpptr = g_pointer[POINTER_TEMP].pointer;
 g_pointer[POINTER_TEMP].pointer = NULL;

 ROAR_DBG("client_handle(id=%i) = %i", id, ret);
 return ret;
}

int  client_str2proto(char * str) {
 if ( str == NULL || *str == 0 )
  return -1;

 if ( !strcasecmp(str, "simple") )
  return RPLD_PROTO_SIMPLE;

 if ( !strcasecmp(str, "mpd") )
  return RPLD_PROTO_MPD;

 return -1;
}

int  client_str2accleacclev(char * str) {
 if ( !strcasecmp(str, "none") )
  return ACCLEV_NONE;

 if ( !strcasecmp(str, "conctl") )
  return ACCLEV_CONCTL;

 if ( !strcasecmp(str, "guest") )
  return ACCLEV_GUEST;

 if ( !strcasecmp(str, "user") )
  return ACCLEV_USER;

 if ( !strcasecmp(str, "all") )
  return ACCLEV_ALL;

 return -1;
}

//ll
