//base64.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2011
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

static char _base64_tab_fw[] =
 "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
 "abcdefghijklmnopqrstuvwxyz"
 "0123456789+/";

static char _base64_tab_bw[] = {
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3e, 0xff, 0xff, 0xff, 0x3f,
  0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
  0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12,
  0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24,
  0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
  0x31, 0x32, 0x33, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
};


int roar_base64_init(struct roar_base64 * state, int flags) {
 if ( flags == -1 )
  flags = 0;

 if ( state == NULL )
  return -1;

 memset(state, 0, sizeof(struct roar_base64));

 state->flags = flags;

 return 0;
}

static inline void _libroar_base64_encode_block(unsigned char out[4], const unsigned char in[3]) {
// printf("in={'%c', '%c', '%c'}\n", in[0], in[1], in[2]);
 out[0] = _base64_tab_fw[(in[0] >> 2) & 0x3F];
 out[1] = _base64_tab_fw[(((in[0] & 0x03) << 4) | (in[1] >> 4)) & 0x3F];
 out[2] = _base64_tab_fw[(((in[1] & 0x0F) << 2) | ((in[2] >> 6) & 0x03)) & 0x3F];
 out[3] = _base64_tab_fw[in[2] & 0x3F];
}

#define _setoff(x) do { if ( off != NULL ) { *off = (x); } } while (0)

ssize_t roar_base64_encode(struct roar_base64 * state, void * out, size_t outlen, const void * in, size_t inlen, size_t * off, int eof) {
 struct roar_base64 state_store;
 size_t offset = 0;
 size_t tlen;
 ssize_t ret = 0;
 unsigned char * outp;
 const unsigned char * inp;

 if ( inlen && in == NULL )
  return -1;

 if ( outlen && out == NULL )
  return -1;

 // this test should be removed as soon as those modes are supported.
 if ( in == NULL || out == NULL )
  return -1;

 // we can not write anything anyway...
 if ( outlen < 4 )
  return 0;

 if ( state == NULL ) {
  state = &state_store;
  roar_base64_init_encode(state, ROAR_BASE64_FLAG_NONE);
  eof = 1;
 }

 if ( state->buflen ) {
  tlen = 3 - state->buflen;
  ROAR_DBG("roar_base64_encode(state=%p,...): Still %llu bytes in state", state, (long long unsigned int)state->buflen);
  ROAR_DBG("roar_base64_encode(state=%p,...): Need %llu bytes", state, (long long unsigned int)tlen);
  if ( inlen >= tlen ) {
   memcpy(state->iobuf + state->buflen, in, tlen);
   offset += tlen;
   in += tlen;
   inlen -= tlen;
   _libroar_base64_encode_block(out, state->iobuf);
   out += 4;
   outlen -= 4;
   ret += 4;
   state->buflen = 0;
  } else {
   memcpy(state->iobuf + state->buflen, in, inlen);
   state->buflen += inlen;
   _setoff(inlen);
   return 0;
  }
 }

 for (; inlen >= 3 && outlen >= 4;) {
  _libroar_base64_encode_block(out, in);
  ret += 4;
  out += 4;
  in  += 3;
  offset += 3;
  inlen -= 3;
  outlen -= 4;
 }

 // something left?
 if ( inlen ) {
  if ( eof ) {
   if ( outlen >= 4 ) {
    outp = out;
    inp  = in;
    switch (inlen) {
     case 1:
       outp[0] = _base64_tab_fw[inp[0] >> 2];
       outp[1] = _base64_tab_fw[(inp[0] & 0x03) << 4];
       outp[2] = '=';
       outp[3] = '=';
       ret += 4;
       out += 4;
       in  += 1;
       offset += 1;
       inlen  -= 1;
       outlen -= 4;
      break;
     case 2:
       outp[0] = _base64_tab_fw[inp[0] >> 2];
       outp[1] = _base64_tab_fw[(((inp[0] & 0x03) << 4) | (inp[1] >> 4)) & 0x3F];
       outp[2] = _base64_tab_fw[(((inp[1] & 0x0F) << 2))                 & 0x3F];
       outp[3] = '=';
       ret += 4;
       out += 4;
       in  += 2;
       offset += 2;
       inlen  -= 2;
       outlen -= 4;
      break;
    }
   }
  } else {
   memcpy(state->iobuf, in, inlen);
   state->buflen = inlen;
   offset += inlen;
   in += inlen;
   inlen -= inlen;
  }
 }

 if ( outlen ) {
  *(char*)out = 0;
 }

 _setoff(offset);

 return 0;
}

ssize_t roar_base64_decode(struct roar_base64 * state, void * out, size_t outlen, const void * in, size_t inlen, size_t * off) {
 struct roar_base64 state_store;
 ssize_t ret = 0;
 unsigned char c;
 int bits;
 const unsigned char * inp = in;
 unsigned char * outp = out;
 size_t offset = 0;

 if ( inlen && in == NULL )
  return -1;

 if ( outlen && out == NULL )
  return -1;

 // this test should be removed as soon as those modes are supported.
 if ( in == NULL || out == NULL )
  return -1;

 if ( state == NULL ) {
  state = &state_store;
  roar_base64_init_encode(state, ROAR_BASE64_FLAG_NONE);
 }

 if ( state->reglen >= 8 ) {
  *outp  = (state->reg >> (state->reglen - 8)) & 0xFF;
  state->reglen -= 8;
  outp++;
  outlen--;
  ret++;
 }

 if ( state->flags & ROAR_BASE64_FLAG_EOF ) {
  if ( outlen ) {
   *outp = 0;
  }
  return ret;
 }

 //printf("state->reglen=%i\n", (int)state->reglen);

 for (; inlen && outlen; ) {
  c = *inp;
  inp++;
  inlen--;
  offset++;

  if ( c == '=' ) {
   bits = 0;
  } else {
   // hi-bit set chars need to be ignored.
   if ( (size_t)c > sizeof(_base64_tab_bw) )
    continue;

   bits = _base64_tab_bw[c];

   // ignore unknown chars.
   if ( bits == (char)0xFF )
    continue;
  }

  state->reg   <<= 6;
  state->reg    |= bits;
  state->reglen += 6;

  if ( state->reglen >= 8 ) {
   *outp  = (state->reg >> (state->reglen - 8)) & 0xFF;
   state->reglen -= 8;
   outp++;
   outlen--;
   ret++;
  }

  if ( c == '=' ) {
   state->flags |= ROAR_BASE64_FLAG_EOF;
   if ( state->reglen <= 6 ) {
    state->reglen = 0;
   }
   if ( inlen ) {
    if ( *inp == '=' ) {
     offset++;
    }
   }
   break;
  }
 }

 if ( off != NULL ) {
  *off = offset;
 }

 if ( outlen ) {
  *outp = 0;
 }

 return ret;
}

int roar_base64_is_eof(struct roar_base64 * state) {
 if ( state == NULL )
  return -1;

 if ( state->reglen >= 8 )
  return 0;

 return !!(state->flags & ROAR_BASE64_FLAG_EOF);
}

int roar_base64_uninit(struct roar_base64 * state) {
 if ( state == NULL )
  return -1;

 if ( state->buflen ) {
  ROAR_WARN("roar_base64_uninit(state=%p): State has %i bytes left in IO buffer. This is a application error.", state, state->buflen);
 }

 if ( state->reglen ) {
  ROAR_WARN("roar_base64_uninit(state=%p): State has %i bits left in decode register. This is a application error.", state, state->reglen);
 }

 return 0;
}

//ll
