///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// Maquette d'une classe geo avec type elements varies (polymorphe)
//  1-ere version
//
// MPI: on envoie les elements par groupes de type HPTqtep homogenes
// ce qui fait que ca reste performant du point de vue MPI
//
// limitation:
//  * Les elements ont supposes ici consecutifs.
//  * Pour s'etendre au cas des variantes curvilignes (Hc,...,qc,tc)
//    cela pose un probleme, car les elements touchant la frontiere 
//    ne sont pas consecutifs par numeros.
//    On pourrait renumeroter tous les elements apres l'identification
//    de la frontiere, mais on peut eviter cette renumerotation tardive.
//  * Pour faire sauter l'hypothese des consecutif,
//    on peut aussi envoyer une paire (index,element) puis ranger x(index) = element
//  => c'est l'etape suivante
//
// test: on construit sur le proc=0 et envoie au proc=1 qui depaquette
//
// Author: Pierre.Saramito@imag.fr
//
// Date: 14 dec 2010
//
#include "rheolef/config.h"
#ifndef _RHEOLEF_HAVE_MPI
int main() { return 0; }
#else // _RHEOLEF_HAVE_MPI

#include "rheolef/array.h"
#include <boost/serialization/export.hpp>
#include <boost/ptr_container/ptr_container.hpp>
#include <boost/ptr_container/serialize_ptr_vector.hpp>
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
using namespace rheolef;
using namespace std;

static size_t dummy = size_t(-1);

char uu [2] = {'a','b'};

class geo_element {
public:
    typedef std::vector<int>::size_type size_type;
    typedef enum {
	p = 0, 
	e = 1, 
	t = 2, 
	q = 3, 
	T = 4, 
	P = 5, 
	H = 6, 
	enum_type_max = 7 
    } enum_type; 

    static char tab_name [enum_type_max];

    virtual enum_type type() const = 0;
    virtual char      name() const = 0;
    virtual size_type size() const = 0;
    virtual size_type dimension() const = 0;
    virtual size_type  operator[] (size_type i) const = 0;
    virtual size_type& operator[] (size_type i)       = 0;
};
char geo_element::tab_name [geo_element::enum_type_max] = {'p', 'e', 't', 'q', 'T', 'P', 'H'};
inline
std::ostream&
operator<< (std::ostream& os, const geo_element& K) {
	os << K.name() << "\t";
        for (geo_element::size_type iloc = 0; iloc < K.size(); iloc++) {
	  os << K[iloc];
	  if (iloc < K.size() - 1) os << " ";
	}
	return os;
}
class geo_element_t : public geo_element {
public:
    geo_element_t () : geo_element() { 
        for (size_type iloc = 0; iloc < size(); iloc++) {
	    _vertex[iloc] = std::numeric_limits<size_type>::max();
        }
    }
    geo_element_t (const geo_element_t& K) : geo_element() { 	
        for (size_type iloc = 0; iloc < size(); iloc++) {
	    _vertex[iloc] = K[iloc];
        }
    }
    geo_element_t& operator= (const geo_element_t& K) {
        for (size_type iloc = 0; iloc < size(); iloc++) {
	    _vertex[iloc] = K[iloc];
        }
	return *this;
    }
    geo_element_t (size_type a, size_type b, size_type c) : geo_element() {
	_vertex[0] = a;
	_vertex[1] = b;
	_vertex[2] = c;
    } 
    enum_type type() const     { return  t; };
    char      name() const     { return 't'; };
    size_type size() const     { return  3; }
    size_type dimension() const { return 2; };
    size_type  operator[] (size_type i) const { return _vertex[i]; }
    size_type& operator[] (size_type i)       { return _vertex[i]; }
    template<class Archive>
    void serialize (Archive& ar, const unsigned int version) {
        ar & _vertex[0];
        ar & _vertex[1];
        ar & _vertex[2];
    }
// data:
protected:
    size_type _vertex[3];
};
class geo_element_q : public geo_element {
public:
    geo_element_q () : geo_element() { 
        for (size_type iloc = 0; iloc < size(); iloc++) {
	    _vertex[iloc] = std::numeric_limits<size_type>::max();
        }
    }
    geo_element_q (const geo_element_t& K) : geo_element() { 	
        for (size_type iloc = 0; iloc < size(); iloc++) {
	    _vertex[iloc] = K[iloc];
        }
    }
    geo_element_q& operator= (const geo_element_q& K) {
        for (size_type iloc = 0; iloc < size(); iloc++) {
	    _vertex[iloc] = K[iloc];
        }
	return *this;
    }
    geo_element_q (size_type a, size_type b, size_type c, size_type d) : geo_element() {
	_vertex[0] = a;
	_vertex[1] = b;
	_vertex[2] = c;
	_vertex[3] = d;
    } 
    enum_type type() const      { return  q; };
    char      name() const      { return 'q'; };
    size_type size() const      { return  4; }
    size_type dimension() const { return  2; };
    size_type  operator[] (size_type i) const { return _vertex[i]; }
    size_type& operator[] (size_type i)       { return _vertex[i]; }
    template<class Archive> 
    void serialize (Archive& ar, const unsigned int version) {
        ar & _vertex[0];
        ar & _vertex[1];
        ar & _vertex[2];
        ar & _vertex[3];
    }
// data:
protected:
    size_type _vertex[4];
};
struct my_polymorphic_geo {
  typedef boost::ptr_vector<geo_element>::size_type size_type;
  typedef boost::ptr_vector<geo_element>::iterator iterator;
  typedef boost::ptr_vector<geo_element>::const_iterator const_iterator;
  my_polymorphic_geo () : _x(), _comm() {
    fill (_count_by_type, _count_by_type+geo_element::enum_type_max, 0);
    if (_comm.rank() == 0) {
      _x.push_back (new geo_element_t (1,2,3));
      _x.push_back (new geo_element_t (4,5,6));
      _x.push_back (new geo_element_q (11,12,13,14));
      _x.push_back (new geo_element_q (15,16,17,18));
      _count_by_type [geo_element::t] = 2;
      _count_by_type [geo_element::q] = 2;
      reset_starters();
      for (size_type i = 0; i < size_type(geo_element::enum_type_max); i++) {
        warning_macro ("init count " << geo_element::tab_name[i] << " " << _count_by_type [i]);
      }
    }
  }
  void reset_starters () {
      _start_by_type [0] = 0;
      for (size_type i = 0; i < size_type(geo_element::enum_type_max); i++) {
        _start_by_type [i+1] = _start_by_type [i] + _count_by_type [i];
      }
  }
  iterator begin (geo_element::enum_type typ) { return _x.begin() + _start_by_type[size_type(typ)]; }
  iterator end   (geo_element::enum_type typ) { return _x.begin() + _start_by_type[size_type(typ)+1]; }
  const_iterator begin (geo_element::enum_type typ) const { return _x.begin() + _start_by_type[size_type(typ)]; }
  const_iterator end   (geo_element::enum_type typ) const { return _x.begin() + _start_by_type[size_type(typ)+1]; }
  void put () const {
    cout << "geo " << _x.size() << endl;
    for (size_t i = 0; i < _x.size(); i++) {
      const geo_element& K = _x[i];
      cout << K << endl;
    }
  }
  void test_mpi () {
    pair<size_type,size_type> count_tag_pair [size_type(geo_element::enum_type_max)];
    int tag_count = 0;
    if (_comm.rank() == 0) {
      for (size_type i = 0; i < size_type(geo_element::enum_type_max); i++) {
        size_type tag = 100+i;
        count_tag_pair [i] = make_pair(_count_by_type[i], tag);
      }
      for (size_type i = 0; i < size_type(geo_element::enum_type_max); i++) {
        warning_macro ("send count " << geo_element::tab_name[i] << " " << count_tag_pair [i].first);
      }
      _comm.send (1, tag_count, count_tag_pair, size_type(geo_element::enum_type_max));
      for (size_type i = 0; i < size_type(geo_element::enum_type_max); i++) {
        size_type i_msg_size = count_tag_pair [i].first;
        size_type i_msg_tag  = count_tag_pair [i].second;
        if (i_msg_size == 0) continue;
	// envoi
        switch (geo_element::enum_type(i)) {
          case geo_element::t: {
	    vector<geo_element_t> buffer (i_msg_size);
	    for (size_type j = 0; j < i_msg_size; j++) {
	      for (size_type k = 0; k < 3; k++) {
      	        buffer[j][k] = _x[_start_by_type[i]+j][k];
	      }
	    }
	    _comm.send (1, i_msg_tag, buffer.begin().operator->(), i_msg_size);
	    break;
	  }
          case geo_element::q: {
	    vector<geo_element_q> buffer (i_msg_size);
	    for (size_type j = 0; j < i_msg_size; j++) {
	      for (size_type k = 0; k < 4; k++) {
      	        buffer[j][k] = _x[_start_by_type[i]+j][k];
	      }
	    }
	    _comm.send (1, i_msg_tag, buffer.begin().operator->(), i_msg_size);
	    break;
	  }
	  default : {
	    error_macro ("unexpected element " << geo_element::tab_name[i]);
	  }
	}
      }
    } else if (_comm.rank() == 1) {
      _comm.recv (0, tag_count, count_tag_pair, size_type(geo_element::enum_type_max));
      for (size_type i = 0; i < size_type(geo_element::enum_type_max); i++) {
        _count_by_type[i] = count_tag_pair [i].first;
        warning_macro ("recv count " << geo_element::tab_name[i] << " " << count_tag_pair [i].first);
      }
      reset_starters();
      _x.reserve (_start_by_type [geo_element::enum_type_max]);
      for (size_type i = 0; i < size_type(geo_element::enum_type_max); i++) {
        size_type i_msg_size = count_tag_pair [i].first;
        size_type i_msg_tag  = count_tag_pair [i].second;
        if (i_msg_size == 0) continue;
	// reception
        switch (geo_element::enum_type(i)) {
          case geo_element::t: {
	    vector<geo_element_t> buffer (i_msg_size);
	    _comm.recv (0, i_msg_tag, buffer.begin().operator->(), i_msg_size);
	    for (size_type j = 0; j < i_msg_size; j++) {
      	      //_x[_start_by_type[i]+j] = buffer[j];
      	      //_x.push_back (buffer[j]);
      	      _x.push_back (new geo_element_t (buffer[j]));
	    }
	    break;
	  }
          case geo_element::q: {
	    vector<geo_element_q> buffer (i_msg_size);
	    _comm.recv (0, i_msg_tag, buffer.begin().operator->(), i_msg_size);
	    for (size_type j = 0; j < i_msg_size; j++) {
      	      //_x[_start_by_type[i]+j] = buffer[j];
      	      //_x.push_back (buffer[j]);
      	      _x.push_back (new geo_element_q (buffer[j]));
	    }
	    break;
	  }
	  default : {
	    error_macro ("unexpected element " << geo_element::tab_name[i]);
	  }
	}
      }
      put();
    }
  }
// data:
  boost::ptr_vector<geo_element> _x;
  mpi::communicator              _comm;
  size_type _count_by_type [geo_element::enum_type_max];
  size_type _start_by_type [geo_element::enum_type_max+1];
};
int main(int argc, char**argv) {
  environment distributed(argc, argv);
  mpi::communicator comm;
  if (comm.size() == 1) {
	cerr << "not sequential : use mpirun -np 2 !" << endl;
	return 0;
  }
  my_polymorphic_geo omega;
  omega.test_mpi();
}
#endif // _RHEOLEF_HAVE_MPI
