%{
// This file is part of Rheolef.
//
// Copyright (C) 2000-2009 Pierre Saramito 
//
// Rheolef is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Rheolef is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Rheolef; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// lexer for qmg mesh files
// =========================================================================

static vector<cad_domain>                   *p_entity_list;
static cad_domain                           *p_entity;
static cad_element         		    *p_geoentity;

static size_type			     entity_dim = size_type(-1);
static size_type                             i_cp = 0;
%}
%union  { 

        int                           empty;
        int 	                      int_value;
        size_type                     uint_value;
        double                        float_value;
        size_type                     string_value;
        cad_element::enum_type        geo_value;
}
%token<empty>			BREP2D
%token<empty>			BREP3D
%token<geo_value>          	VERTEX
%token<geo_value>          	BEZIER_CURVE
%token<geo_value>          	BEZIER_TRIANGLE
%token<geo_value>      		BEZIER_QUAD

%token<string_value>            IDENTIFIER
%token<int_value>               INT_NUMBER
%token<float_value>             FLOAT_NUMBER
%token<string_value>            STRING

%type<empty>                    controlPoints2d
%type<empty>                    controlPoints3d
%type<empty>                    value
%type<empty>                    value_list
%type<geo_value>                geoentity
%type<geo_value>                geom

%type<int_value>                uint
%type<int_value>                deg

%type<float_value>              float

%type<string_value>             name

%%
all : ascii_brep
		{ /* post-treatment at end of reading */
		    compute_bbox ( 
			p_data -> _x.begin(),
        		p_data -> _x.end(),
        		p_data -> _dim,
        		p_data -> _xmin,
        		p_data -> _xmax);
		}
    ;

ascii_brep : ascii_brep2d
	|    ascii_brep3d
	;

ascii_brep2d : BREP2D
	'(' propVal ')'
	'(' controlPoints2d ')'
	'(' vertexList ')'
	'(' edgeList ')'
	'(' surfaceList ')'

	/* BREP2D : brep_v2.0 2 2 */

			{ p_data -> _dim = 2; }
	;

ascii_brep3d : BREP3D
	'(' propVal ')'
	'(' controlPoints3d ')'
	'(' vertexList ')'
	'(' edgeList ')'
	'(' surfaceList ')'
	'(' chamberList ')'

			{ p_data -> _dim = 3; }

	/* BREP3D : brep_v2.0 3 3 */
	;

propVal : /* empty! */
	| propVal
	  IDENTIFIER value

	/* The property/value pairs are arbitrary strings. 
	 * If a property or value string contains spaces, 
	 * then it should be enclosed in an outer pair of parentheses.
     	 * Currently, the only global property value with significance to QMG 2.0
     	 * is geo_global_id.
	 */
	;

value   : IDENTIFIER            {$$=0;}
        | INT_NUMBER            {$$=0;}
        | FLOAT_NUMBER          {$$=0;}
        | '(' value_list ')'    {$$=0;}
        ;
value_list: /* empty */         {$$=0;}
        | value_list value
        ;

controlPoints2d : /* empty */
	 		{ $$ =0; }
	| controlPoints2d	
	  float float
	 		{ $$ =0;
			  p_data -> _x.push_back(point($2,$3));
			}
	;

controlPoints3d : /* empty */
	 		{ $$ =0; }
	| controlPoints3d	
	  float float float
	 		{ $$ =0;
			  p_data -> _x.push_back(point($2,$3,$4));
			}
     	/* This is a list of the n nodes in the mesh. Each entry consists of a
     	 * global ID, which must be a nonnegative integer, followed by either 2 or
     	 * 3 coordinates, which are real numbers. The number of coordinates is
     	 * equal to the embedded dimension (2 or 3). The global ID's do not have
     	 * to be in increasing order, but they must be distinct.]
	 */
	;
float : FLOAT_NUMBER | INT_NUMBER { $$ = $1; };
uint  : INT_NUMBER 
			{ 
			  if ($1 < 0) {
				warning_macro ("positive integer expected, "<<$1<<" founded"); 
			        cad_qmg_error ("positive integer expected"); 
			  }
			  $$ = $1;
			}
	;
vertexList:  { p_entity_list = &(p_data -> _p); entity_dim = 0; } entityList;
edgeList:    { p_entity_list = &(p_data -> _e); entity_dim = 1; } entityList;
surfaceList: { p_entity_list = &(p_data -> _s); entity_dim = 2; } entityList;
chamberList: { p_entity_list = &(p_data -> _v); entity_dim = 3; } entityList;

entityList: /* empty */
	| entityList
	  name 	
		{
		  trace_macro ("entityList [0]" << symbol($2));
		  p_entity_list -> push_back(cad_domain());
		  p_entity = &(p_entity_list -> back());
		  p_entity -> _name = symbol($2);
		  p_entity -> _dim = entity_dim;
		  trace_macro ("entityList [1]" << symbol($2));
		}
	  '(' propVal ')'
	  '(' boundary ')'
	  '(' lowdim ')'
	  '(' geom ')'

		  { trace_macro ("entityList [F]" << symbol($2)); }
	/* 
	 * In other words, each brep face has five entries in its face list: name, propVal,
     	 * child, lowdim, geom.
	 * The name field is an arbitary string.
	 * A name field with spaces should be enclosed in parentheses.
	 * The remaining fields are as follows.
	 */

	;

name : 	  IDENTIFIER 
	| uint ':' uint  {
			   char buf [256];
			   sprintf (buf, "d%d_%d", $1, $3);
			   $$ = insert (buf);
			 }
	;

boundary: /* empty */
	| boundary
	  name 

	/*
	 * These are names of bounding faces.
	 * That is, these names refer to the names appear in the 'name' field
	 * of the lower-dimensional faces.
	 * Names appearing in this list must be faces of exactly one lower dimension.
	 * Each childname can optionally be preceded by the symbol + or - to indicate orientation.
	 * Orientations are not used by QMG 2.0
	 */

		{ p_entity -> _boundary.push_back(symbol($2)); }
	;

lowdim: /* empty */
	| lowdim
	  name 

	/* These are the names of the low-dimensional bounding faces.
	 * Names appearing in this list must be faces of dimension lower by 2 or 3 from the
     	 * dimension of the face.
	 */
		{ p_entity -> _lowdim.push_back(symbol($2)); }
	;

geom: /* empty */
		{ $$ = reference_element::max_size; }
	| geom
		{
		  trace_macro ("geom [0]");
		  p_entity -> push_back(cad_element());
		  p_geoentity = &(p_entity -> back());
		  i_cp = 0;
		  trace_macro ("geom [1]");
		}
	  '(' geoentity ')'
		{ 
		  trace_macro ("geom [2]");
		  check_macro (i_cp == p_geoentity -> size(),
				"cad input:" << cad_qmg_line_no << ": "
					<< "control point list too short: " 
					<< p_geoentity -> name()
					<< ": "
					<< p_geoentity -> size()
					<< " indexes expected, "
					<< i_cp
					<< " founded");
		  trace_macro ("geom [f]");
		}
	;


geoentity : 
	  VERTEX 		{ p_geoentity -> set_type ($1); }
	  uint			{ p_geoentity -> operator[](i_cp++) = $3; }
	| BEZIER_CURVE deg 	{ p_geoentity -> set_type($1,$2); }
	  controlPointList
	| BEZIER_TRIANGLE deg   { p_geoentity -> set_type($1,$2); }
	  controlPointList
	| BEZIER_QUAD deg deg 	{ p_geoentity -> set_type($1,$2,$3); }
	  controlPointList

	/* The vertex entity is allowed only for topological vertices,
	 * and a topological vertex must have exactly one such entity.
	 * The bezier_curve entity is allowed only for edges.
	 * The bezier_triangle and bezier_quad entities are allowed only for surfaces.
	 * The degree entries are positive integers.
	 * The number of control points p is deg+1 for a curve,
	 * (deg+1)*(deg+2)/2 for a triangle, and
     	 * (deg1+1)*(deg2+1) for a quad.
	 * Each control point cpnum1, ..., cpnump is a nonnegative integer
	 * that is an index into the control point list.
	 */

	; 

deg: uint;

controlPointList: /* empty */
	| controlPointList
	  uint
			{
			  check_macro (i_cp < p_geoentity -> size(),
				"cad input:" << cad_qmg_line_no << ": "
					<< "control point list too long: " 
					<< p_geoentity -> name()
					<< ":"
					<< p_geoentity -> size()
					<< " indexes expected");
		  	  trace_macro ("point [" << i_cp << "] := " << $2);
			  p_geoentity -> operator[](i_cp++) = $2;
			}
	;
%%

