/*
 * help.c - Help functions
 *
 * refdbg - GObject refcount debugger
 * Copyright (C) 2004-2005 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 */
#include <stdio.h>
#include <string.h>
#include <glib.h>

#include "refdbg_priv.h"

static char *txt_topics =
    "Help Topics\n"
    "-----------\n"
    "commands              Command reference\n"
    "display               Additional information on `display' command\n"
    "examples              Examples of command usage\n"
    "objects               Additional information on `objects' command\n"
    "rules                 Rule format reference\n"
    "timer                 Information on the refcount timer\n";

static char *txt_commands =
    "Commands - To display the value of a `variable=value' just use `variable'\n"
    "* - Indicates the default value\n"
    "--------\n"
    "help [topic]          Get help\n"
    "exit/quit/q           Quit the RefDbg shell (ignored if not in the shell)\n"
    "display [DispRule]    Display event log with optional criteria\n"
    "objects [ObjRule]     Display object information with optional criteria\n"
    "clear                 Clear event log and refcount timer\n"
    "stats                 Get RefDbg statistics (event log size, mem usage, etc)\n"
    "time                  Get current time (offset from RefDbg start time)\n"
    "timer=<MSEC>          Refcount timer expire in milliseconds (0 = disable)\n"
    "rules                 Display current rule stack\n"
    "addrule <Rule>        Add an event rule to the event stack\n"
    "insrule <POS> <Rule>  Insert a rule at the given position (POS >= 0)\n"
    "delrule <N>           Delete a rule by rule number\n"
    "r0=<Rule>             Set the default event rule\n"
    "r<N>=<Rule>           Set a specific rule\n"
    "savelog               Saves the current event log and objects (if logobjs=1)\n"
    "logobjs=0|1*          Log active objects to end of log file on savelog\n"
    "logexit=0|1*          Enable or disable saving of event log on exit\n"
    "statsexit=0|1*        Enable or disable display of stats on exit\n"
    "logname='<filename>'  Set the log filename (default=refdbg.log)\n"
    "btnum=<N>             Set max callers in stack backtraces to N (init only)\n"
    "btpaths=0*|1          Enable or disable display of full paths in backtraces\n"
    "notimer               Disable glib timer callback (init only)\n"
    "dispmax=<N>           Set max items/events to display for display commands\n";

static char *txt_display =
    "Display Command\n"
    "---------------\n"
    "This command displays events from the event log. If no arguments are\n"
    "specified then all events are printed, or any of the following arguments\n"
    "can be supplied to set the criteria of displayed events:\n"
    "  [TIME-TIME]     Time range of events to display (see below for format)\n"
    "  btnum=N         Sets the number of backtrace callers to display per event\n"
    "  <GType>         GType to match (match objects of type or derived thereof)\n"
    "  <!GType>        Exclude GType (exclude objects of type or derived thereof)\n"
    "  0x76543210      Address of an object instance to match\n"
    "  !0xFEDCBA98     Address of an object instance to exclude\n"
    "  D:<Flags>       Same flags used in event rules (defaults to 'D:All')\n"
    "  limit=[OFS,]MAX Limit to MAX events from optional OFS (< 0: ofs from end)\n"
    "\n"
    "* Multiple GType and address match arguments can be specified.\n"
    "* <Flags> are the same as used in event rules ('help rules') except special\n"
    "  flags are not allowed.\n"
    "* TIME is in the format 'mm:ss.iiiiii', 'ss.iiiiii', 'mm:ss', or 'mm'\n"
    "  mm = minutes, ss = seconds, iiiiii = microseconds\n"
    "  If the first or second time value is omitted then it defaults to\n"
    "  the first or last event in the event log respectively. When omitting the\n"
    "  first time value the format should be '[-TIME]'.\n"
    "* The 'limit' argument can accept 1 or 2 arguments. Just one argument\n"
    "  sets the maximum result count, 2 arguments set the offset to begin at\n"
    "  (0 based) followed by the max result count.\n";

static char *txt_examples =
    "Command Examples\n"
    "----------------\n"
    "display\n"
    "Display all events in event log.\n"
    "\n"
    "display 0x83deb0f9 [00:10-00:20] btnum=0\n"
    "Display all events in event log that match a specific object, between\n"
    "10 seconds and 20 seconds from start time, and don't display any backtrace\n"
    "addresses.\n"
    "\n"
    "display limit=-10,10\n"
    "Display last 10 events.\n"
    "\n"
    "addrule <GtkObject> <!GtkAdjustment> D:Error B:None L:All\n"
    "  Add a rule to match GtkObject derived types except GtkAdjustment and\n"
    "  display errors, no break points will be executed and log all events.\n"
    "\n"
    "r0=<GtkWindow> D:All B:All L:All\n"
    "  Set the default rule to match only GtkWindow derived types and\n"
    "  display, break and log all events.\n"
    "\n"
    "addrule 0x12345678 D:Ref|Unref|Error B:Ref|Unref|Error L:All\n"
    "  Add a rule to match a specific object instance and display and set\n"
    "  break points for Ref, Unref and Error events and log all events.\n"
    "\n"
    "addrule 0x12345678 D:Error|Timer ; timer = 1000\n"
    "  Add a rule to match a specific object instance, display errors or\n"
    "  refcount timer events, no breakpoints (default) and log all (default).\n"
    "  The next command `timer = 1000' sets the timer expire time\n"
    "  to 1 second.\n" "\n" "delrule r1\n" "  Delete rule `r1'.\n";

static char *txt_objects =
    "Objects Command\n"
    "--------------\n"
    "This command displays information on objects such as address, type,\n"
    "refcount and whether it is still active or has been destroyed.\n"
    "If no arguments are specified then the state of all objects are printed\n"
    "(except for destroyed objects). Any of the following arguments can be\n"
    "supplied to set the criteria of displayed objects.\n"
    "  <GType>       GType to match (match objects of type or derived thereof)\n"
    "  <!GType>      GType to exclude (exclude objects of type or derived thereof)\n"
    "  0x76543210    Address of an object instance to match (implies O:All)\n"
    "  !0xFEDCBA98   Address of an object instance to exclude\n"
    "  O:<Flags>     Set what types of objects to display (see below)\n"
    "\n"
    "Multiple GType and address arguments can be specified.\n"
    "\n"
    "<Flags> can be a pipe '|' delimited list from 'Active|Destroyed'. A value\n"
    "of 'All' is equivalent to both flags specified. Defaults to 'Active' which\n"
    "displays active objects only.\n";

static char *txt_rules =
    "Event Rule Format\n"
    "-----------------\n"
    "<GType> <!GType> 0x76543210 !0xFEDCBA98 D:<Flags> B:<Flags> L:<Flags>\n"
    "\n"
    "  <GType>       GType to match (match objects of type or derived thereof)\n"
    "  <!GType>      GType to exclude (exclude objects of type or derived thereof)\n"
    "  0x76543210    Address of an object instance to match\n"
    "  !0xFEDCBA98   Address of an object instance to exclude\n"
    "  D:<Flags>     Display flags (what events/errors are displayed)\n"
    "  B:<Flags>     Break point flags (what events/errors cause a break point)\n"
    "  L:<Flags>     Event log flags (what events/errors get logged)\n"
    "\n"
    "<Flags> consists of one or more of the following separated by a pipe '|':\n"
    "---- Event flags ----\n"
    "  PreNew        Occurs just prior to a GObject being created\n"
    "  New           Occurs after a GObject has been created\n"
    "  Ref           Occurs before g_object_ref() is called on an object\n"
    "  Unref         Occurs before g_object_unref() is called on an object\n"
    "  PreFinalize   Occurs just before an object is destroyed (last unref)\n"
    "  Finalize      Occurs after an object has been destroyed\n"
    "---- Error flags ----\n"
    "  EUnknownObj   (Ref/Unref) Object address has never been seen before\n"
    "  EDestroyedObj (Ref/Unref) Object has already been destroyed\n"
    "  ENotObj       (Ref/Unref) Address is of a non-object (corrupted memory?)\n"
    "  EInitCount    (New) Initial reference count not equal to 1\n"
    "  EBadCount     (Ref/Unref) Unexpected reference count (corrupted memory?)\n"
    "---- Special flags ----\n"
    "  Timer         (D: only) Enable refcount timer\n"
    "  Paranoid      (D:/B: only) Process EUnknownObj/EDestroyedObj immediately\n"
    "---- Flag groups ----\n"
    "  All           All Event and Error flags\n"
    "  None          Clears all Event/Error flags\n"
    "  Event         All Event flags\n" "  Error         All Error flags\n";

static char *txt_timer =
    "Refcount Timer\n"
    "--------------\n"
    "The refcount timer detects object refcount changes over time. Since a\n"
    "lot of transient refcount activity can occur in a short period of\n"
    "time, it can be useful to only display long term refcount\n"
    "changes. This feature can help discover refcount leaks (postive or\n"
    "negative).\n"
    "To enable the refcount timer, set the \"timer\" variable to the desired\n"
    "number of expire milliseconds. You then need to add \"D:Timer\" to an\n"
    "event rule to enable display of refcount changes for objects matching\n"
    "that rule (L:New|Ref|Unref|Finalize is implied). If an object's refcount\n"
    "has not changed for the specified expire time, then the refcount is\n"
    "considered \"stable\" and is printed if it differs from the previous stable\n"
    "refcount value.\n";


/**
 * refdbg_help:
 * @topic: Topic name to get help on or %NULL to display topics
 * 
 * Display help on a given @topic, or display topics if %NULL.
 */
void
refdbg_help (const char *topic)
{
  if (!topic || !strlen (topic)) {
    fputs (txt_topics, stderr);
    fputc ('\n', stderr);
    fputs (txt_commands, stderr);
  } else if (casecmp (topic, "commands") == 0)
    fputs (txt_commands, stderr);
  else if (casecmp (topic, "display") == 0)
    fputs (txt_display, stderr);
  else if (casecmp (topic, "examples") == 0)
    fputs (txt_examples, stderr);
  else if (casecmp (topic, "objects") == 0)
    fputs (txt_objects, stderr);
  else if (casecmp (topic, "rules") == 0)
    fputs (txt_rules, stderr);
  else if (casecmp (topic, "timer") == 0)
    fputs (txt_timer, stderr);
  else
    fprintf (stderr, "Unknown help topic '%s'\n", topic);
}
