/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: zlink.c,v 1.17.2.2 2005/05/06 22:57:39 amcwilliam Exp $
 */

#include "setup.h"

#ifdef USE_ZLIB

#include <stdio.h>
#include <stdlib.h>
#include "zlink.h"
#include "memory.h"
#include "conf2.h"
#include "common.h"

extern void ircdlog(unsigned int, char *, ...);

extern ConfigItem_general GeneralConfig;

static char zip_in_buf[ZIP_IN_SIZE];
static char zip_out_buf[ZIP_OUT_SIZE];

static inline void zip_init_stream(z_stream *s)
{
	s->zalloc = NULL;
	s->zfree = NULL;
	s->data_type = Z_ASCII;
}

void *zip_create_input_session()
{
	ziplink_in *zin = (ziplink_in *)MyMalloc(sizeof(ziplink_in));

	zip_init_stream(&zin->stream);

	if (inflateInit(&zin->stream) != Z_OK) {
		return NULL;
	}

	return zin;
}

void *zip_create_output_session()
{
	ziplink_out *zout = (ziplink_out *)MyMalloc(sizeof(ziplink_out));

	zip_init_stream(&zout->stream);

	if (deflateInit(&zout->stream, GeneralConfig.compression_level) != Z_OK) {
		return NULL;
	}

	return zout;
}

char *zip_input(void *session, char *buf, int *len, int *err, char **nbuf, int *nbuflen)
{
	ziplink_in *zin = (ziplink_in *)session;
	z_stream *s;
	int ret = 0;

	ASSERT(zin != NULL);

	*nbuf = NULL;
	*err = 0;

	s = &zin->stream;
	s->next_in = buf;
	s->avail_in = *len;
	s->next_out = zip_in_buf;
	s->avail_out = ZIP_IN_SIZE;

	if ((ret = inflate(s, Z_SYNC_FLUSH)) == Z_OK) {
		if (s->avail_in) {
			if (s->avail_out) {
				*len = -1;
				return (s->msg != NULL) ? s->msg : "?";
			}

			*nbuf = s->next_in;
			*nbuflen = s->avail_in;
		}

		*len = (ZIP_IN_SIZE - s->avail_out);
		return zip_in_buf;
	}

	*len = -1;
	*err = ret;

	return (s->msg != NULL) ? s->msg : "?";
}

char *zip_output(void *session, char *buf, int *len, int force_flush, int *large_data)
{
	ziplink_out *zout = (ziplink_out *)session;
	z_stream *s;
	int ret;

	ASSERT(zout != NULL);

	s = &zout->stream;

	if (buf != NULL) {
		memcpy((zout->buf + zout->size), buf, *len);
		zout->size += *len;
	}
	if (!force_flush && ((zout->size < ZIP_MIN_BLOCK)
	  || (large_data != NULL && (zout->size < (ZIP_MAX_BLOCK - 512))))) {
		*len = 0;
		return NULL;
	}

	s->next_in = zout->buf;
	s->avail_in = zout->size;
	s->next_out = zip_out_buf;
	s->avail_out = ZIP_OUT_SIZE;

	if ((ret = deflate(s, Z_SYNC_FLUSH)) == Z_OK) {
		zout->size = 0;
		*len = (ZIP_OUT_SIZE - s->avail_out);
		return zip_out_buf;
	}

	*len = -1;
	*large_data = ret;
	return (s->msg != NULL) ? s->msg : "?";
}

void zip_in_get_stats(void *session, unsigned long *in, unsigned long *out, double *ratio)
{
	ziplink_in *zin = (ziplink_in *)session;
	ASSERT(zin != NULL);

	*in = zin->stream.total_in;
	*out = zin->stream.total_out;

	if (*out) {
		*ratio = ((100.0 * (double)zin->stream.total_in) / (double)zin->stream.total_out);
	}
}

void zip_out_get_stats(void *session, unsigned long *in, unsigned long *out, double *ratio)
{
	ziplink_out *zout = (ziplink_out *)session;
	ASSERT(zout != NULL);

	*in = zout->stream.total_in;
	*out = zout->stream.total_out;

	if (*in) {
		*ratio = ((100.0 * (double)zout->stream.total_out) / (double)zout->stream.total_in);
	}
}

void zip_destroy_input_session(void *session)
{
	ziplink_in *zin = (ziplink_in *)session;
	ASSERT(zin != NULL);
	inflateEnd(&zin->stream);
	MyFree(zin);
}

void zip_destroy_output_session(void *session)
{
	ziplink_out *zout = (ziplink_out *)session;
	ASSERT(zout != NULL);
	deflateEnd(&zout->stream);
	MyFree(zout);
}

#endif /* USE_ZLIB */
