/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: whowas.c,v 1.15.2.1 2004/12/07 03:05:42 pneumatus Exp $
 */

#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "h.h"
#include "memory.h"
#include "xmode.h"

extern unsigned int hash_whowas_name(char *);

static void add_whowas_to_clist(aWhowas **, aWhowas *);
static void del_whowas_from_clist(aWhowas **, aWhowas *);
static void add_whowas_to_list(aWhowas **, aWhowas *);
static void del_whowas_from_list(aWhowas **, aWhowas *);

aWhowas WHOWAS[NICKNAMEHISTORYLENGTH];
aWhowas *WHOWASHASH[WW_MAX];
int whowas_next = 0;

void add_history(aClient *cptr, int online)
{
	aWhowas *new;

	ASSERT(cptr != NULL);

	for (new = WHOWASHASH[hash_whowas_name(cptr->name)]; new; new = new->next) {
		if (!mycmp(cptr->name, new->name) &&
		  !mycmp(cptr->username, new->username) &&
		  !mycmp(cptr->host, new->hostname) &&
		  !mycmp(cptr->user->maskedhost, new->maskedhost) && 
		  !mycmp(cptr->info, new->realname)) {
			new->logoff = timeofday;
			return;
		}
	}

	new = &WHOWAS[whowas_next];
	if (new->hashv != -1) {
		if (new->online != NULL) {
			del_whowas_from_clist(&(new->online->user->whowas), new);
		}
		del_whowas_from_list(&WHOWASHASH[new->hashv], new);
	}

	new->hashv = hash_whowas_name(cptr->name);
	new->logoff = timeofday;
	strncpyzt(new->name, cptr->name, NICKLEN + 1);
	strncpyzt(new->username, cptr->username, USERLEN + 1);
	strncpyzt(new->hostname, cptr->host, HOSTLEN + 1);
	strncpyzt(new->maskedhost, MaskedHost(cptr), HOSTLEN + 1);
	strncpyzt(new->realname, cptr->info, REALLEN + 1);
	new->servername = cptr->user->server;

	if (online == 1) {
		new->online = cptr;
		add_whowas_to_clist(&(cptr->user->whowas), new);
	}
	else {
		new->online = NULL;
	}
	add_whowas_to_list(&WHOWASHASH[new->hashv], new);
	whowas_next++;
	if (whowas_next == NICKNAMEHISTORYLENGTH) {
		whowas_next = 0;
	}
	return;
}

void off_history(aClient *cptr)
{
	aWhowas *tmp, *next;

	for (tmp = cptr->user->whowas; tmp; tmp = next) {
		next = tmp->cnext;
		tmp->online = NULL;
		del_whowas_from_clist(&(cptr->user->whowas), tmp);
	}
}

aClient *get_history(char *nick, time_t timelimit)
{
	int blah = hash_whowas_name(nick);
	aWhowas *tmp = WHOWASHASH[blah];

	timelimit = timeofday - timelimit;
	for (; tmp; tmp = tmp->next) {
		if (mycmp(nick, tmp->name) || tmp->logoff < timelimit) {
			continue;
		}
		return tmp->online;
	}
	return NULL;
}

void count_whowas_memory(int *ww_cnt, unsigned long *ww_mem)
{
	unsigned long cnt = 0;
	int i;

	for (i = 0; i < NICKNAMEHISTORYLENGTH; i++) {
		if (WHOWAS[i].hashv == -1) {
			continue;
		}
		cnt++;
	}

	*ww_cnt = cnt;
	*ww_mem = (cnt * sizeof(aWhowas));
}

void init_whowas()
{
	int i;
	for (i = 0; i < NICKNAMEHISTORYLENGTH; i++) {
		memset((char *)&WHOWAS[i], '\0', sizeof(aWhowas));
		WHOWAS[i].hashv = -1;
	}
	for (i = 0; i < WW_MAX; i++) {
		WHOWASHASH[i] = NULL;
	}
}

static void add_whowas_to_clist(aWhowas **bucket, aWhowas *whowas)
{
	whowas->cprev = NULL;
	if ((whowas->cnext = *bucket) != NULL) {
		whowas->cnext->cprev = whowas;
	}
	*bucket = whowas;
}

static void del_whowas_from_clist(aWhowas **bucket, aWhowas *whowas)
{
	if (whowas->cprev) {
		whowas->cprev->cnext = whowas->cnext;
	}
	else {
		*bucket = whowas->cnext;
	}
	if (whowas->cnext) {
		whowas->cnext->cprev = whowas->cprev;
	}
}

static void add_whowas_to_list(aWhowas **bucket, aWhowas *whowas)
{
	whowas->prev = NULL;
	if ((whowas->next = *bucket) != NULL) {
		whowas->next->prev = whowas;
	}
	*bucket = whowas;
}

static void del_whowas_from_list(aWhowas **bucket, aWhowas *whowas)
{
	if (whowas->prev) {
		whowas->prev->next = whowas->next;
	}
	else {
		*bucket = whowas->next;
	}
	if (whowas->next) {
		whowas->next->prev = whowas->prev;
	}
}
