/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: struct.h,v 1.277.2.7 2005/07/08 23:29:50 amcwilliam Exp $
 */

#ifndef	__struct_include__
#define __struct_include__

#include "config.h"

#ifndef CONFIG_H_BLUEMOON
#error Incorrect config.h for this version of ircd.
#endif

#include "queue.h"
#include "setup.h"
#include <stdio.h>
#include <sys/types.h>
#include <time.h>
#include <netinet/in.h>
#include <netdb.h>
#ifdef HAVE_STDDEF_H
#include <stddef.h>
#endif
#include "hash.h"
#include "../adns/adns.h"

typedef struct Client aClient;
typedef struct Client_local LocalClient;
typedef struct User anUser;
typedef struct User_local LocalUser;
typedef struct Server aServer;
typedef struct Channel aChannel;
typedef struct slink SLink;
typedef struct channelban channelBan;
typedef struct channelMember chanMember;
typedef struct SMode Mode;
typedef struct Watch aWatch;
typedef struct ListOptions LOpts;
typedef struct Whowas aWhowas;
typedef struct iphash_entry IPEntry;
typedef struct module Module;
typedef struct event Event;
typedef struct scache_entry ServerCache;
#ifdef USE_THROTTLE
typedef struct hashent_t hashent;
typedef struct hashent_list_t hashent_list;
typedef struct hash_table_t hash_table;
typedef struct throttle_t throttle;
#endif
typedef struct listener Listener;
typedef struct dns_query DNSQuery;
typedef struct connauth ConnectAuth;
typedef struct message_file MessageFile;
typedef struct suid SUID;
typedef struct capab_t Capability;
typedef struct ircd_t IRCd;
typedef struct _stats_request StatsRequest;

#include "conf2.h"
#include "sbuf.h"
#include "ssl.h"
#include "dlink.h"

#define	HOSTLEN			63	/* Length of hostname.  Updated to comply with RFC1123 */
#define HOSTIPLEN		15	/* Length of dotted quad form of IPv4 */
#define	NICKLEN			30	/* Length of nickname */
#define MAX_DATE_STRING		32	/* Length of date string */
#define	USERLEN			10	/* Length of username */
#define	REALLEN			50	/* Length of realname */
#define	TOPICLEN		307	/* Length of topic */
#define	KILLLEN			400	/* Length of kill */
#define	CHANNELLEN		200	/* Length of channel name */
#define	PASSWDLEN		63	/* Length of passwords */
#define	KEYLEN			23	/* Length of channel key */
#define	BUFSIZE			512	/* Length of internal buffers */
#define IDLEN			8	/* Length of base64 SUID and prefix */
#define	MAXSILES		5
#define MAXSILELENGTH		128
#define HARD_TARGET_LIMIT	64	/* Maximum max_target limit for m_message() */

#define SCACHE_HASH_SIZE	257
#define IP_HASH_SIZE		0x1000
#define	USERHOST_REPLYLEN	(NICKLEN + HOSTLEN + USERLEN + 5)
#define ID_MAP_SIZE		4096
#define MAX_CLIENT_RECVQ	8192

#define MAX_SERVID              0xFFF	/* max serv id: 4,096 */
#define MIN_SERVID              0x1	/* min serv id: 1 */

#define MAX_USERID              0xFFFFE /* max user id: 1,048,573 */
#define MIN_USERID              0x1	/* min user id: 1 */

#define USERID_PART(x)		((x) & 0xFFFFF)
#define SERVID_PART(x)		(((x) >> 20) & 0xFFF)
#define MAKE_ID(s, u)		((((s) << 20) & 0xFFF00000) | ((u) & 0xFFFFF))

#ifndef offsetof
#define	offsetof(t, m)		(int)((&((t *)0L)->m))
#endif
#define timeofday		(Internal.ircd_time.tv_sec)

#define	STAT_CONNECTING		-4
#define	STAT_HANDSHAKE		-3
#define	STAT_ME			-2
#define	STAT_UNKNOWN		-1
#define	STAT_SERVER		0
#define	STAT_CLIENT		1

#define	IsRegistered(x)		((x)->status >= STAT_SERVER)
#define	IsConnecting(x)		((x)->status == STAT_CONNECTING)
#define	IsHandshake(x)		((x)->status == STAT_HANDSHAKE)
#define	IsMe(x)			((x)->status == STAT_ME)
#define	IsUnknown(x)		((x)->status == STAT_UNKNOWN)
#define	IsServer(x)		((x)->status == STAT_SERVER)
#define	IsClient(x)		((x)->status == STAT_CLIENT)

#define	IsPerson(x)		((x)->user && IsClient(x))
#define IsAnyServer(x)		(IsServer(x) || IsConnecting(x) || IsHandshake(x))

#define	SetConnecting(x)	((x)->status = STAT_CONNECTING)
#define	SetHandshake(x)		((x)->status = STAT_HANDSHAKE)
#define	SetMe(x)		((x)->status = STAT_ME)
#define	SetUnknown(x)		((x)->status = STAT_UNKNOWN)
#define	SetServer(x)		((x)->status = STAT_SERVER)
#define	SetClient(x)		((x)->status = STAT_CLIENT)

/* Client Flags */
#define	FLAGS_PINGSENT		0x00000001	/* Unreplied ping sent */
#define	FLAGS_DEADSOCKET 	0x00000002	/* Local socket is dead--exiting soon */
#define	FLAGS_KILLED		0x00000004	/* Prevents "QUIT" from being sent for this */
#define	FLAGS_CLOSING		0x00000008	/* set when closing to suppress errors */
#define	FLAGS_GOTID		0x00000010	/* identd check successful */
#define	FLAGS_NONL		0x00000020	/* No \n in buffer */
#define FLAGS_NORMALEX		0x00000040	/* Client exited normally */
#define FLAGS_SENDQEX		0x00000080	/* SendQ exceeded */
#define FLAGS_SUPER		0x00000100	/* client is super! */
#define FLAGS_SECURE		0x00000200	/* client is secure */
#define FLAGS_HASHEDIP		0x00000400	/* client is in the ip address hash table */
#define FLAGS_HASHED		0x00000800	/* client is in the client/server hash table */
#define FLAGS_SUIDPAIR		0x00001000	/* cptr->id.id is a valid servid/userid pair */
#define FLAGS_BLOCKED		0x00002000	/* socket is in blocking state */
#define FLAGS_RECVQ		0x00004000	/* client has data in recvQ to parse */
#define FLAGS_DCCNOTICE		0x00008000	/* seen notice regarding DCC SENDs */
#define FLAGS_HASVHOST		0x00010000	/* has a virtual host */
#define FLAGS_WANTDKEY		0x00020000	/* Link should be encrypted if possible */
#define FLAGS_DOINGDKEY		0x00040000	/* Link is performing DKEY exchange */
#define FLAGS_RC4IN		0x00080000	/* Link input is encrypted */
#define FLAGS_RC4OUT		0x00100000	/* Link output is encrypted */
#define FLAGS_WANTZIP		0x00200000	/* Link should be zipped */
#define FLAGS_ZIPIN		0x00400000	/* Link input is zipped */
#define FLAGS_ZIPOUT		0x00800000	/* Link output is zipped */
#define FLAGS_SPOOFED		0x01000000	/* User has spoofed host */
#define FLAGS_SPOOFEDUN		0x02000000	/* User has spoofed username */
#define FLAGS_KLINEEXEMPT	0x04000000	/* User is exempt from kline */
#ifdef USE_THROTTLE
#define FLAGS_THROTTLEEXEMPT	0x08000000	/* User is exempt from throttle */
#endif

#define PingSent(x)		((x)->flags & FLAGS_PINGSENT)
#define DeadSocket(x)		((x)->flags & FLAGS_DEADSOCKET)
#define IsKilled(x)		((x)->flags & FLAGS_KILLED)
#define IsClosing(x)		((x)->flags & FLAGS_CLOSING)
#define GotID(x)		((x)->flags & FLAGS_GOTID)
#define NoNewLine(x)		((x)->flags & FLAGS_NONL)
#define NormalExit(x)		((x)->flags & FLAGS_NORMALEX)
#define SendQEx(x)		((x)->flags & FLAGS_SENDQEX)
#define IsULine(x)		((x)->flags & FLAGS_SUPER)
#define IsSecure(x)		((x)->flags & FLAGS_SECURE)
#define HashedIP(x)		((x)->flags & FLAGS_HASHEDIP)
#define IsHashed(x)		((x)->flags & FLAGS_HASHED)
#define HasSUID(x)		((x)->flags & FLAGS_SUIDPAIR)
#define IsBlocked(x)		((x)->flags & FLAGS_BLOCKED)
#define HasRecvQ(x)		((x)->flags & FLAGS_RECVQ)
#define SeenDCCNotice(x)	((x)->flags & FLAGS_DCCNOTICE)
#define HasVhost(x)		((x)->flags & FLAGS_HASVHOST)
#define WantDKEY(x)		((x)->flags & FLAGS_WANTDKEY)
#define DoingDKEY(x)		((x)->flags & FLAGS_DOINGDKEY)
#define InputRC4(x)		((x)->flags & FLAGS_RC4IN)
#define OutputRC4(x)		((x)->flags & FLAGS_RC4OUT)
#define WantZIP(x)		((x)->flags & FLAGS_WANTZIP)
#define InputZIP(x)		((x)->flags & FLAGS_ZIPIN)
#define OutputZIP(x)		((x)->flags & FLAGS_ZIPOUT)
#define IsSpoofed(x)		((x)->flags & FLAGS_SPOOFED)
#define IsSpoofedUn(x)		((x)->flags & FLAGS_SPOOFEDUN)
#define IsKlineExempt(x)	((x)->flags & FLAGS_KLINEEXEMPT)
#ifdef USE_THROTTLE
#define IsThrottleExempt(x)	((x)->flags & FLAGS_THROTTLEEXEMPT)
#endif

#define SetPingSent(x)		((x)->flags |= FLAGS_PINGSENT)
#define SetDeadSocket(x)	((x)->flags |= FLAGS_DEADSOCKET)
#define SetKilled(x)		((x)->flags |= FLAGS_KILLED)
#define SetClosing(x)		((x)->flags |= FLAGS_CLOSING)
#define SetGotID(x)		((x)->flags |= FLAGS_GOTID)
#define SetNoNewLine(x)		((x)->flags |= FLAGS_NONL)
#define SetNormalExit(x)	((x)->flags |= FLAGS_NORMALEX)
#define SetSendQEx(x)		((x)->flags |= FLAGS_SENDQEX)
#define SetULine(x)		((x)->flags |= FLAGS_SUPER)
#define SetSecure(x)		((x)->flags |= FLAGS_SECURE)
#define SetHashedIP(x)		((x)->flags |= FLAGS_HASHEDIP)
#define SetHashed(x)		((x)->flags |= FLAGS_HASHED)
#define SetSUID(x)		((x)->flags |= FLAGS_SUIDPAIR)
#define SetBlocked(x)		((x)->flags |= FLAGS_BLOCKED)
#define SetRecvQ(x)		((x)->flags |= FLAGS_RECVQ)
#define SetDCCNotice(x)		((x)->flags |= FLAGS_DCCNOTICE)
#define SetVhost(x)		((x)->flags |= FLAGS_HASVHOST)
#define SetWantDKEY(x)		((x)->flags |= FLAGS_WANTDKEY)
#define SetDoingDKEY(x)		((x)->flags |= FLAGS_DOINGDKEY)
#define SetInputRC4(x)		((x)->flags |= FLAGS_RC4IN)
#define SetOutputRC4(x)		((x)->flags |= FLAGS_RC4OUT)
#define SetWantZIP(x)		((x)->flags |= FLAGS_WANTZIP)
#define SetInputZIP(x)		((x)->flags |= FLAGS_ZIPIN)
#define SetOutputZIP(x)		((x)->flags |= FLAGS_ZIPOUT)
#define SetSpoofed(x)		((x)->flags |= FLAGS_SPOOFED)
#define SetSpoofedUn(x)		((x)->flags |= FLAGS_SPOOFEDUN)
#define SetKlineExempt(x)	((x)->flags |= FLAGS_KLINEEXEMPT)
#ifdef USE_THROTTLE
#define SetThrottleExempt(x)	((x)->flags |= FLAGS_THROTTLEEXEMPT)
#endif

#define ClearPingSent(x)	((x)->flags &= ~FLAGS_PINGSENT)
#define ClearDeadSocket(x)	((x)->flags &= ~FLAGS_DEADSOCKET)
#define ClearKilled(x)		((x)->flags &= ~FLAGS_KILLED)
#define ClearClosing(x)		((x)->flags &= ~FLAGS_CLOSING)
#define ClearGotID(x)		((x)->flags &= ~FLAGS_GOTID)
#define ClearNoNewLine(x)	((x)->flags &= ~FLAGS_NONL)
#define ClearNormalExit(x)	((x)->flags &= ~FLAGS_NORMALEX)
#define ClearSendQEx(x)		((x)->flags &= ~FLAGS_SENDQEX)
#define ClearULine(x)		((x)->flags &= ~FLAGS_SUPER)
#define ClearSecure(x)		((x)->flags &= ~FLAGS_SECURE)
#define ClearHashedIP(x)	((x)->flags &= ~FLAGS_HASHEDIP)
#define ClearHashed(x)		((x)->flags &= ~FLAGS_HASHED)
#define ClearSUID(x)		((x)->flags &= ~FLAGS_SUIDPAIR)
#define ClearBlocked(x)		((x)->flags &= ~FLAGS_BLOCKED)
#define ClearRecvQ(x)		((x)->flags &= ~FLAGS_RECVQ)
#define ClearDCCNotice(x)	((x)->flags &= ~FLAGS_DCCNOTICE)
#define ClearVhost(x)		((x)->flags &= ~FLAGS_HASVHOST)
#define ClearWantDKEY(x)	((x)->flags &= ~FLAGS_WANTDKEY)
#define ClearDoingDKEY(x)	((x)->flags &= ~FLAGS_DOINGDKEY)
#define ClearInputRC4(x)	((x)->flags &= ~FLAGS_RC4IN)
#define ClearOutputRC4(x)	((x)->flags &= ~FLAGS_RC4OUT)
#define ClearWantZIP(x)		((x)->flags &= ~FLAGS_WANTZIP)
#define ClearInputZIP(x)	((x)->flags &= ~FLAGS_ZIPIN)
#define ClearOutputZIP(x)	((x)->flags &= ~FLAGS_ZIPOUT)
#define ClearSpoofed(x)		((x)->flags &= ~FLAGS_SPOOFED)
#define ClearSpoofedUn(x)	((x)->flags &= ~FLAGS_SPOOFEDUN)
#define ClearKlineExempt(x)	((x)->flags &= ~FLAGS_KLINEEXEMPT)
#ifdef USE_THROTTLE
#define ClearThrottleExempt(x)	((x)->flags &= ~FLAGS_THROTTLEEXEMPT)
#endif

/* SERVER-only flags */
#define SERVFLAG_USERBURST	0x0001	/* Server in nick/server netburst */
#define SERVFLAG_TOPICBURST	0x0002	/* Server in topic netburst */
#define SERVFLAG_SENTSOB	0x0004	/* We've sent SOB, just need to send EOB */
#define SERVFLAG_RECVEOB	0x0008	/* We're waiting for EOB */
#define SERVFLAG_DKEYIN		0x0010	/* Got DKEY PUB I */
#define SERVFLAG_DKEYOUT	0x0020	/* Got DKEY PUB O */
#define SERVFLAG_MAPPED		0x0040	/* Already mapped in /MAP */
#define SERVFLAG_NBURST		0x0080	/* Got Network Burst */

#define UserBurst(x)		((x)->serv->flags & SERVFLAG_USERBURST)
#define TopicBurst(x)		((x)->serv->flags & SERVFLAG_TOPICBURST)
#define SendingBurst(x)		((x)->serv->flags & (SERVFLAG_USERBURST|SERVFLAG_TOPICBURST))
#define SentSOB(x)		((x)->serv->flags & SERVFLAG_SENTSOB)
#define RecvEOB(x)		((x)->serv->flags & SERVFLAG_RECVEOB)
#define GotDKEYin(x)		((x)->serv->flags & SERVFLAG_DKEYIN)
#define GotDKEYout(x)		((x)->serv->flags & SERVFLAG_DKEYOUT)
#define IsMapped(x)		((x)->serv->flags & SERVFLAG_MAPPED)
#define GotNBurst(x)		((x)->serv->flags & SERVFLAG_NBURST)

#define SetUserBurst(x)		((x)->serv->flags |= SERVFLAG_USERBURST)
#define SetTopicBurst(x)	((x)->serv->flags |= SERVFLAG_TOPICBURST)
#define SetSentSOB(x)		((x)->serv->flags |= SERVFLAG_SENTSOB)
#define SetRecvEOB(x)		((x)->serv->flags |= SERVFLAG_RECVEOB)
#define SetDKEYin(x)		((x)->serv->flags |= SERVFLAG_DKEYIN)
#define SetDKEYout(x)		((x)->serv->flags |= SERVFLAG_DKEYOUT)
#define SetMapped(x)		((x)->serv->flags |= SERVFLAG_MAPPED)
#define SetGotNBurst(x)		((x)->serv->flags |= SERVFLAG_NBURST)

#define ClearUserBurst(x)	((x)->serv->flags &= ~SERVFLAG_USERBURST)
#define ClearTopicBurst(x)	((x)->serv->flags &= ~SERVFLAG_TOPICBURST)
#define ClearSentSOB(x)		((x)->serv->flags &= ~SERVFLAG_SENTSOB)
#define ClearRecvEOB(x)		((x)->serv->flags &= ~SERVFLAG_RECVEOB)
#define ClearDKEYin(x)		((x)->serv->flags &= ~SERVFLAG_DKEYIN)
#define ClearDKEYout(x)		((x)->serv->flags &= ~SERVFLAG_DKEYOUT)
#define ClearMapped(x)		((x)->serv->flags &= ~SERVFLAG_MAPPED)
#define ClearGotNBurst(x)	((x)->serv->flags &= ~SERVFLAG_NBURST)

/* SERVER capabilities */
#define CAP_TS			0x00000001	/* Supports TS protocol */
#define CAP_BURST		0x00000002	/* Supports BURST protocol */
#define CAP_UNCONN		0x00000004	/* Supports UNCONNECT protocol */
#define CAP_SUPER		0x00000008	/* Supports U: lines (kludge) */
#define CAP_ZIP			0x00000010	/* Supports ZIP (server<->server compression) */
#define CAP_DKEY		0x00000020	/* Supports DKEY protocol */
#define CAP_SSJ3		0x00000040	/* Supports SSJ3 protocol */
#define CAP_SN2			0x00000080	/* Supports SN2 protocol */
#define CAP_VHOST		0x00000100	/* Supports VHOST protocol */
#define CAP_SUID		0x00000200	/* Supports SUID protocol */
#define CAP_TOK1		0x00000400	/* Supports TOK1 protocol */
#define CAP_NOQUIT		0x00000800	/* Supports NOQUIT protocol */
#define CAP_TSMODE		0x00001000	/* Supports TS in MODE */
#define CAP_NBURST		0x00002000	/* Supports NBURST */

#define DEFAULT_CAPABS		(CAP_BURST|CAP_UNCONN|CAP_ZIP|CAP_SSJ3|CAP_SN2|\
				CAP_VHOST|CAP_SUID|CAP_TOK1|CAP_TSMODE|CAP_NBURST|\
				CAP_NOQUIT)
#define ID_CAPS			(CAP_SUID|CAP_TOK1)
#define NO_CAPS			0

#define CapTS(x)		((x)->localClient->capabs & CAP_TS)
#define CapBURST(x)		((x)->localClient->capabs & CAP_BURST)
#define CapUNCONNECT(x)		((x)->localClient->capabs & CAP_UNCONN)
#define CapZIP(x)		((x)->localClient->capabs & CAP_ZIP)
#define CapDKEY(x)		((x)->localClient->capabs & CAP_DKEY)
#define CapSSJ3(x)		((x)->localClient->capabs & CAP_SSJ3)
#define CapSN2(x)		((x)->localClient->capabs & CAP_SN2)
#define CapVHOST(x)		((x)->localClient->capabs & CAP_VHOST)
#define CapSUID(x)		((x)->localClient->capabs & CAP_SUID)
#define CapTOK1(x)		((x)->localClient->capabs & CAP_TOK1)
#define CapID(x)		(((x)->localClient->capabs & ID_CAPS) == ID_CAPS)
#define CapNOQUIT(x)		((x)->localClient->capabs & CAP_NOQUIT)
#define CapTSMODE(x)		((x)->localClient->capabs & CAP_TSMODE)
#define CapNBURST(x)		((x)->localClient->capabs & CAP_NBURST)

#define HasMode(x, y)		((x)->umode & (y))
#define AddMode(x, y)		((x)->umode |= (y))
#define DelMode(x, y)		((x)->umode &= ~(y))

#define OFLAG_DIE		0x00000001	/* oper can use DIE command */
#define OFLAG_RESTART		0x00000002	/* oper can use RESTART command */
#define OFLAG_REHASH		0x00000004	/* oper can use REHASH command */
#define OFLAG_KLINE		0x00000008	/* oper can use KLINE command */
#define OFLAG_UNKLINE		0x00000010	/* oper can use UNKLINE command */
#define OFLAG_LROUTE		0x00000020	/* oper can do local routing */
#define OFLAG_GROUTE		0x00000040	/* oper can do global routing */
#define OFLAG_LKILL		0x00000080	/* oper can do local kills */
#define OFLAG_GKILL		0x00000100	/* oper can do global kills */
#define OFLAG_LOCOPS		0x00000200	/* oper can send LOCOPS notices */
#define OFLAG_GLOBOPS		0x00000400	/* oper can send GLOBOPS notices */
#define OFLAG_WALLOPS		0x00000800	/* oper can send WALLOPS notices */
#define OFLAG_LNOTICE		0x00001000	/* oper can send local notices */
#define OFLAG_GNOTICE		0x00002000	/* oper can send global notices */
#define OFLAG_FNOTICE		0x00004000	/* oper can receive flood notices */
#define OFLAG_LCLICONN		0x00008000	/* oper can receive local cliconnect notices */
#define OFLAG_GCLICONN		0x00010000	/* oper can receive global cliconnect notices */
#define OFLAG_NETADMIN		0x00020000	/* oper is a Network Administrator */
#define OFLAG_ADMIN		0x00040000	/* oper is a Server Administrator */
#define OFLAG_COADMIN		0x00080000	/* oper is a Server Co-Administrator */
#define OFLAG_SADMIN		0x00100000	/* oper is a Services Administrator */
#define OFLAG_RSTAFF		0x00200000	/* oper is routing staff (can see routing notices and server ip's) */
#define OFLAG_OPERMOTD		0x00400000	/* oper receives OPERMOTD on oper-up */
#define OFLAG_NORECVQTHROTTLE	0x00800000	/* oper is except from recvQ throttling */

#define OPHasFlag(x, y)		((x)->localUser->oflags & (y))
#define OPAddFlag(x, y)		((x)->localUser->oflags |= (y))
#define OPDelFlag(x, y)		((x)->localUser->oflags &= ~(y))

enum {
	CCONN_LEV = 0,		/* Cli-connect notices */
	GCCONN_LEV,		/* Global cli-connect notices */
	REJ_LEV,		/* Reject notices */
	SKILL_LEV,		/* Server kill notices */
	SPY_LEV,		/* Spying notices */
	DEBUG_LEV,		/* Debug notices */
	FLOOD_LEV,		/* Flood notices */
	SPAM_LEV,		/* Spambot notices */
	DCCSEND_LEV		/* DCCAllow notices */
};

enum {
	DEBUG_FATAL = 1,	/* Fatal errors */
	DEBUG_ERROR,		/* Errors */
	DEBUG_NOTICE,		/* Notices */
	DEBUG_DNS,		/* DNS debug */
	DEBUG_INFO,		/* General info debug */
	DEBUG_NUM,		/* Numeric debug */
	DEBUG_SEND,		/* Sending debug */
	DEBUG_DEBUG,		/* Debugging (unimportant) debug */
	DEBUG_MALLOC,		/* Memory debug */
	DEBUG_LIST		/* List debug */
};

enum {
	MFTYPE_MOTD,		/* MOTD */
	MFTYPE_OPERMOTD,	/* OPERMOTD */
	MFTYPE_RULES		/* RULES */
};

enum {
	CLIENTAUTH_MATCHED = 0,	/* access granted, allow/class block attached */
	CLIENTAUTH_NOMATCH,	/* no matched allow block */
	CLIENTAUTH_CLASSFULL,	/* class block full */
	CLIENTAUTH_INVALIDPW,	/* invalid password */
	CLIENTAUTH_TOOMANYIPS	/* too many clients from IP */
};

struct Counter {
	int server;		/* servers */
	int myserver;		/* my servers */
	int mysuper;		/* my super servers */
	int oper;		/* opers */
	int chan;		/* channels */
	int local;		/* local clients */
	int total;		/* total clients */
	int invisi;		/* invisible clients */
	int unknown;		/* unknown connections */
	int max_loc;		/* max local clients */
	int max_tot;		/* max global clients */
	int cli_restart;	/* clients since (re)start */
	time_t start;		/* when we started collecting info */
	u_long today;		/* client connections today */
	time_t day;		/* when today started */
	u_long weekly;		/* connections this week */
	time_t week;		/* when this week started */
	u_long monthly;		/* connections this month */
	time_t month;		/* when this month started */
	u_long yearly;		/* connections this year (this is gunna be be big!) */
	time_t year;		/* when this year started */
};

struct Whowas {
	int hashv;
	char name[NICKLEN + 1];
	char username[USERLEN + 1];
	char hostname[HOSTLEN + 1];
	char maskedhost[HOSTLEN + 1];
	char realname[REALLEN + 1];
	char *servername;
	time_t logoff;
	aClient *online;
	aWhowas *next, *prev;
	aWhowas *cnext, *cprev;
};

/* iphash entry structure */
struct iphash_entry {
	struct in_addr ip;
	int count;
	IPEntry *next;
};

/* module structure */
struct module {
	char *name;		/* Used directly: do not move! */
	char *info;		/* Used directly: do not move! */
	int version;		/* Used directly: do not move! */
	char *revision;		/* Used directly: do not move! */
	char *filename;
	void *ptr;
	int (*unloadfunc)();
	unsigned short flags;
	dlink_list commands;
	dlink_list events;
	dlink_list hookevents;
	dlink_list hooks;
	dlink_list usermodes;
};

struct event {
	char *name;
	void (*func)();
	void *arg;
	time_t freq;
	time_t when;
	unsigned loop : 1;
	Module *owner;
};

/* ServerCache structure */
struct scache_entry {
	char name[HOSTLEN + 1];
	ServerCache *next;
};

#ifdef USE_THROTTLE
struct hashent_t {
	void *ent;
	SLIST_ENTRY(hashent_t) lp;
};

struct hash_table_t {
	int size;
	hashent_list *table;
	size_t keyoffset;
	size_t keylen;
	struct {
		unsigned nocase : 1;
		unsigned string : 1;
	} flag;
	int (*cmpfunc)(void *, void *);
};

struct throttle_t {
	char addr[HOSTIPLEN + 1];
	int conns;
	time_t first;
	time_t last;
	time_t zline_start;
	int stage;
	int re_zlines;
	LIST_ENTRY(throttle_t) lp;
};
#endif

struct listener {
	ConfigItem item;
	char ipaddr[HOSTIPLEN + 1];
	int port;
	unsigned short flags;
	int fd;
	unsigned long count;
	int clients;
	struct in_addr ip;
	time_t lasttime;
	long sendM;
	long receiveM;
	long sendK;
	long receiveK;
	unsigned short sendB;
	unsigned short receiveB;
	ConfigItem_listen *conf;
};

struct dns_query {
	void *ptr;
	adns_query query;
	adns_answer answer;
	void (*callback)();
};

struct connauth {
	aClient *cptr;
	int fd;
	time_t timeout;

	dlink_node self;
	DNSQuery dns_query;

	unsigned dns_pending : 1;
	unsigned ident_connect : 1;
	unsigned ident_pending : 1;
};

struct message_file {
	char *filename;
	short type;
	dlink_list data;
	char changed_date[MAX_DATE_STRING + 1];
};

struct suid {
	unsigned int id;
	char string[IDLEN + 1];
};

struct capab_t {
	unsigned int flag;
	char *token;
	unsigned link_state : 1;
	unsigned synch_state : 1;
	char *pretty_token;
};

struct ircd_t {
	char *conf_file;
#if defined(USE_OPENSSL) && defined(OS_CYGWIN)
	char ssl_privkey_passwd[512];
#endif
	int conf_errors;
	unsigned valid_conf : 1;
	char *debug_mode;
	int debug_level;
	unsigned run_in_terminal : 1;
	unsigned do_rehash : 1;
	unsigned do_module_rehash : 1;
	time_t next_event;
	int max_con_count;
	int max_cli_count;
	float curr_sendK;
	float curr_recvK;
	unsigned rehashing : 1;
	struct timeval ircd_time;
	unsigned int default_capabs;
	char masking_keys[256];
#ifdef ALWAYS_SEND_DURING_SPLIT
	unsigned net_split : 1;
#endif
	unsigned verbose : 1;
};

struct Server {
	char *up;
	char by_nick[NICKLEN + 1];
	char by_user[USERLEN + 1];
	char by_host[HOSTLEN + 1];
#ifdef USE_OPENSSL
	void *sessioninfo_in;		/* pointer to opaque sessioninfo structure */
	void *sessioninfo_out;		/* pointer to opaque sessioninfo structure */
	void *rc4_in;			/* etc */
	void *rc4_out;			/* etc */
#endif
#ifdef USE_ZLIB
	void *zip_out;			/* inbound zipped data */
	void *zip_in;			/* outbound zipped data */
#endif
	ConfigItem_link *conf;		/* config */
	unsigned short flags;		/* Server-only flags */
	aClient *userid[ID_MAP_SIZE];
};

struct User_local {
	time_t last;
	unsigned long oflags;
	ConfigItem_oper *oper;
	ConfigItem_allow *allow;
	SLink *invited;
	LOpts *lopt;
	int nick_changes;
	time_t last_nick_change;
	int away_changes;
	time_t last_away_change;
	time_t last_join_time;
	time_t last_part_time;
	int join_part_count;
	int oper_warn_countdown;
#ifdef FLUD
	time_t fludblock;
	struct fludbot *fluders;
#endif
};

struct User {
	char maskedhost[HOSTLEN + 1];
	chanMember *channel;
	char *away;
	int joined;
	char *server;
	unsigned int servicestamp;
	SLink *silence;
	dlink_list dccallow_list;
	dlink_list on_dccallow_list;
	dlink_list accept_list;
	dlink_list on_accept_list;
	time_t last_accept_notice;
	SLink *watch;
	int watches;
	aWhowas *whowas;
	char nicksent;
#ifdef FLUD
	SLink *fludees;
#endif
};

struct Client_local {
	dlink_node self;
	SBuf sendQ;
	SBuf recvQ;
	short last_sendq;
	char buffer[BUFSIZE];
	unsigned int sendM;
	unsigned int receiveM;
	unsigned int sendK;
	unsigned int receiveK;
	unsigned short sendB;
	unsigned short receiveB;
	int fd;
#ifdef USE_OPENSSL
	SSL *ssl;
#endif
	int count;
	char sockhost[HOSTLEN + 1];
	char passwd[PASSWDLEN + 1];
	unsigned int capabs;
	int sockerr;
	Listener *listener;
	ConfigItem_class *class;
	ConnectAuth *auth;
};

struct Client {
	aClient *next, *prev, *hnext, *idhnext;
	LocalClient *localClient;
	anUser *user;
	LocalUser *localUser;
	aServer *serv;
	time_t lasttime;
	time_t firsttime;
	time_t since;
	time_t tsinfo;
	unsigned long flags;
	unsigned long umode;
	aClient *from;
	aClient *uplink;
  	int hopcount;
	short status;
	char name[HOSTLEN + 1];
	char username[USERLEN + 1];
	char host[HOSTLEN + 1];
	char info[REALLEN + 1];
	struct in_addr ip;
	SUID id;
	char hostip[HOSTIPLEN + 1];
	unsigned int serial;
};

/* statistics structures */
struct stats {
	unsigned int is_cl;     /* number of client connections */
	unsigned int is_sv;     /* number of server connections */
	unsigned int is_ni;     /* connection but no idea who it was */
	unsigned short is_cbs;  /* bytes sent to clients */
	unsigned short is_cbr;  /* bytes received to clients */
	unsigned short is_sbs;	/* bytes sent to servers */
	unsigned short is_sbr;  /* bytes received to servers */
	unsigned long is_cks;	  /* k-bytes sent to clients */
	unsigned long is_ckr;	  /* k-bytes received to clients */
	unsigned long is_sks;	  /* k-bytes sent to servers */
	unsigned long is_skr;	  /* k-bytes received to servers */
	time_t      is_cti;		  /* time spent connected by clients */
	time_t      is_sti;		  /* time spent connected by servers */
	unsigned int is_ac;		  /* connections accepted */
	unsigned int is_ref;	  /* accepts refused */
	unsigned int is_unco;	  /* unknown commands */
	unsigned int is_wrdi;	  /* command going in wrong direction */
	unsigned int is_unpf;	  /* unknown prefix */
	unsigned int is_empt;	  /* empty message */
	unsigned int is_num;	  /* numeric message */
	unsigned int is_kill;  	/* number of kills generated on collisions */
	unsigned int is_fake;	  /* MODE 'fakes' */
	unsigned int is_asuc;	  /* successful auth requests */
	unsigned int is_abad;	  /* bad auth requests */
	unsigned int is_udp;		/* packets recv'd on udp port */
#ifdef FLUD
	unsigned int is_flud;	  /* users/channels flood protected */
#endif
};

/* mode structure for channels */
struct SMode {
	unsigned int mode;
	int limit;
	char key[KEYLEN + 1];
};

/* channel member link structure, used for chanmember chains */
struct channelMember {
	chanMember *nextuser;
	chanMember *nextchan;
	aClient *cptr;
	aChannel *chptr;
	int flags;
	unsigned int banserial;
};

/* channel ban structure */
struct channelban {
	char *id;
	char *set_by;
	time_t when;
};

/* general link structure used for chains */
struct slink {
	SLink *next;
	union {
		aClient *cptr;
		aChannel *chptr;
		channelBan *banptr;
		aWatch *wptr;
		char *cp;
	} value;
	int flags;
};

/* channel structure */

struct Channel {
	struct Channel *nextch, *prevch, *hnextch;
	int hashv;
	Mode mode;
	char topic[TOPICLEN + 1];
        char topic_nick[NICKLEN + 1];
	time_t topic_time;
	int users;
	chanMember *members;
	SLink *invites;
	dlink_list banlist;
	dlink_list exceptlist;
	dlink_list invexlist;
	time_t channelts;
#ifdef FLUD
	time_t fludblock;
	struct fludbot *fluders;
#endif
	char chname[CHANNELLEN+1];
	time_t last_knock;
	unsigned int banserial;
};

#define STATS_NONE		0x00
#define STATS_OPERONLY		0x01
#define STATS_ADMINONLY		0x02
#define STATS_NETADMINONLY	0x04
#define STATS_PACESIMPLE	0x08
#define STATS_PACEINTENSE	0x10

struct _stats_request {
	char higher;
	char lower;
	void (*func)(aClient *, char *);
	unsigned short options;
	char *info;
};

#define TS_CURRENT	5		/* Current TS protocol version */
#define TS_MIN		3		/* Minimum TS protocol version */

#define	SecretChannel(x)	((x) && ((x)->mode.mode & CMODE_SECRET))
#define	ShowChannel(v,c)	(PubChannel(c) || IsMember((v),(c)))
#define	PubChannel(x)		((!x) || ((x)->mode.mode & CMODE_SECRET) == 0)
#define IsMember(blah, chan)	((blah && blah->user && find_user_member((blah->user)->channel, chan)) ? 1 : 0)

/* Misc macros */

#define	BadPtr(x)	((x) == NULL || (*(x) == '\0'))
#define MyConnect(x)	((x)->localClient != NULL)
#define MyClient(x)	(MyConnect(x) && IsClient(x))

#define IsSendable(x)	(!((x)->flags & FLAGS_BLOCKED) && SBufLength(&x->localClient->sendQ) < 16384)
#define DoList(x)	(((x)->localUser != NULL) && ((x)->localUser->lopt != NULL))

#define strncpyzt(x, y, N)	do { strncpy(x, y, N); x[N - 1] = '\0'; } while (0)

#define parse_netmask(x, y, z)	(in_str(x, '.') ? parse_ipv4_netmask((const char *)x, y, z) : 0)
#define match_ipv4(a, m, b)	((ntohl((a).s_addr) & ~((1 << (32 - (b))) - 1)) == ntohl((m).s_addr))

#define MaskedHost(x)	(((x)->umode & UMODE_MASKED) ? (x)->user->maskedhost : (x)->host)

#define get_ts(x)	(*(x) == '!') ? base64dec(x) : strtoul(x, NULL, 0)
#define is_id(x)	(*(x) == '!' && *(x + 1) != '\0')
#define valid_servid(x)	(USERID_PART(base64dec(x)) == 0)

/* Get id/tok or name/msg */
#define get_id(x)	((HasSUID(x)) ? (x)->id.string : (x)->name)
#define get_msg(x)	(((x)->tok_str != NULL) ? (x)->tok_str : (x)->msg_str)

/* Get id/tok or name/msg of x for use with t */
#define use_id(x,t)	((CapID(t) && HasSUID(x)) ? (x)->id.string : (x)->name)
#define use_msg(x,t)	((CapID(t) && ((x)->tok_str != NULL)) ? (x)->tok_str : (x)->msg_str)

/* Zipped session has data waiting */
#ifdef USE_ZLIB
#define zip_out(x)	(OutputZIP(x) && ((ziplink_out *)(x)->serv->zip_out)->size)
#else
#define zip_out(x)	(0)
#endif

/* return values for hunt_server() */

#define	HUNTED_NOSUCH	(-1)	/* if the hunted server is not found */
#define	HUNTED_ISME	0	/* if this server should execute the command */
#define	HUNTED_PASS	1	/* if message passed onwards successfully */

/* used when sending to #mask or $mask */

#define	MATCH_SERVER  1
#define	MATCH_HOST    2

/* misc variable externs */

extern char ircd_version[128], *revision;
extern char *infotext[], *dalinfotext[];
extern char *compile_number, *creation;

/* misc defines */

#define ZIP_NEXT_BUFFER -4
#define RC4_NEXT_BUFFER -3
#define	FLUSH_BUFFER	-2

#ifdef FLUD
struct fludbot {
	aClient *fluder;
	int count;
	time_t first_msg;
	time_t last_msg;
	struct fludbot *next;
};
#endif

struct Watch {
	aWatch *hnext;
	time_t lasttime;
	SLink *watch;
	char nick[1];
};

struct ListOptions {
	LOpts *next;
	SLink *yeslist, *nolist;
	int starthash;
	short int showall;
	unsigned short usermin;
	int usermax;
	time_t currenttime;
	time_t chantimemin;
	time_t chantimemax;
	time_t topictimemin;
	time_t topictimemax;
};

/* internal defines for cptr->sockerr */
#define IRCERR_BUFALLOC		-11
#define IRCERR_ZIP		-12
#define IRCERR_SSL		-13

#endif /* __struct_include__ */
