\name{gausscop}
\title{Multivariate Gaussian Copula with Arbitrary Marginals}
\alias{gausscop}
\usage{
gausscop(response=NULL, distribution="gamma", mu=NULL, shape=NULL,
	autocorr="exponential", pmu=NULL, pshape=NULL, par=NULL,
	pre=NULL, delta=NULL, shfn=FALSE, common=FALSE, envir=parent.frame(),
	print.level=0, ndigit=10, gradtol=0.00001, steptol=0.00001,
	iterlim=100, fscale=1, stepmax=10*sqrt(theta\%*\%theta),
	typsiz=abs(c(theta)))
fitted.gausscop(z)
residuals.gausscop(z)
}
\alias{fitted.gausscop}
\alias{residuals.gausscop}
\description{
\code{gausscop} fits multivariate repeated measurements models based
on the Gaussian copula with a choice of marginal distributions.
Dependence among responses is provided by the correlation matrix
containing random effects and/or autoregression.

With two levels of nesting, the first is the individual and the second
will consist of clusters within individuals.

For clustered (non-longitudinal) data, where only random effects will
be fitted, \code{times} are not necessary.

This function is designed to fit linear and nonlinear models with
time-varying covariates observed at arbitrary time points. A
continuous-time AR(1) and zero, one, or two levels of nesting can be
handled.

Nonlinear regression models can be supplied as formulae where
parameters are unknowns in which case factor variables cannot be used
and parameters must be scalars. (See \code{\link[rmutil]{finterp}}.)
}
\arguments{
\item{response}{A list of two or three column matrices with response
values, times, and possibly nesting categories, for each individual,
one matrix or dataframe of response values, or an object of class,
\code{response} (created by \code{\link[rmutil]{restovec}}) or \code{repeated}
(created by \code{\link[rmutil]{rmna}} or \code{\link[rmutil]{lvna}}). If the
\code{repeated} data object contains more than one response variable,
give that object in \code{envir} and give the name of the response
variable to be used here.}
\item{distribution}{The marginal distribution: exponential, gamma,
Weibull, Pareto, inverse Gauss, logistic, Cauchy, Laplace, or Levy.}
\item{mu}{The linear or nonlinear regression model to be fitted for
the location parameter. For marginal distributions requiring positive
response values, a log link is used. This model can be a function of
the parameters or a formula beginning with ~, specifying either a
linear regression function for the location parameter in the Wilkinson
and Rogers notation or a general function with named unknown
parameters that describes the location, returning a vector the same
length as the number of observations.}
\item{shape}{The linear or nonlinear regression model to be fitted for
the log shape parameter. This can be a function of the parameters
or a formula beginning with ~, specifying either a linear regression
function for the location parameter in the Wilkinson and Rogers
notation or a general function with named unknown parameters that
describes the location. If it contains unknown parameters, the keyword
\code{mu} may be used to specify a function of the location parameter.}
\item{autocorr}{The form of the autocorrelation function:
\code{exponential} is the usual \eqn{\rho^{|t_i-t_j|}}{rho^|t_i-t_j|};
\code{gaussian} is \eqn{\rho^{(t_i-t_j)^2}}{rho^((t_i-t_j)^2)};
\code{cauchy} is \eqn{1/(1+\rho(t_i-t_j)^2)}{1/(1+rho(t_i-t_j)^2)};
\code{spherical} is
\eqn{((|t_i-t_j|\rho)^3-3|t_i-t_j|\rho+2)/2}{((|t_i-t_j|rho)^3-3|t_i-t_j|rho+2)/2}
for \eqn{|t_i-t_j|\leq1/\rho}{|t_i-t_j|<=1/rho} and zero otherwise.}
\item{pmu}{Initial parameter estimates for the location regression model.}
\item{pshape}{Initial parameter estimate for the shape regression model.}
\item{par}{If supplied, an initial estimate for the autocorrelation
parameter.}
\item{pre}{Zero, one or two parameter estimates for the variance
components, depending on the number of levels of nesting.}
\item{delta}{Scalar or vector giving the unit of measurement
for each response value, set to unity by default. For example, if a
response is measured to two decimals, \code{delta=0.01}. Ignored if
response has class, \code{response} or \code{repeated}.}
\item{shfn}{If TRUE, the supplied shape function depends on the
location function. The name of this location function must be the last
argument of the shape function.}
\item{common}{If TRUE, \code{mu} and \code{shape} must both be
functions with, as argument, a vector of parameters having some or all
elements in common between them so that indexing is in common
between them; all parameter estimates must be supplied in \code{pmu}.
If FALSE, parameters are distinct between the two functions and
indexing starts at one in each function.}
\item{envir}{Environment in which model formulae are to be
interpreted or a data object of class, \code{repeated}, \code{tccov},
or \code{tvcov}; the name of the response variable should be given in
\code{response}. If \code{response} has class \code{repeated}, it is
used as the environment.}
\item{others}{Arguments controlling \code{\link{nlm}}.}
\item{z}{An object of class, \code{gausscop}.}
}
\references{
Song, P.X.K. (2000) Multivariate dispersion models generated from
Gaussian copula. Scandinavian Journal of Statistics 27, 305-320.
}
\value{
A list of class \code{gausscop} is returned that contains all of the
relevant information calculated, including error codes.
}
\author{J.K. Lindsey}
\seealso{
\code{\link[growth]{carma}}, \code{\link[growth]{elliptic}},
\code{\link[rmutil]{finterp}}, \code{\link[repeated]{gar}},
\code{\link[rmutil]{gettvc}}, \code{\link[repeated]{glmm}},
\code{\link[repeated]{gnlmm}}, \code{\link[gnlm]{gnlr}},
\code{\link[rmutil]{iprofile}}, \code{\link[repeated]{kalseries}},
\code{\link[rmutil]{mprofile}}, \code{\link[growth]{potthoff}},
\code{\link[rmutil]{read.list}}, \code{\link[rmutil]{restovec}},
\code{\link[rmutil]{rmna}}, \code{\link[rmutil]{tcctomat}},
\code{\link[rmutil]{tvctomat}}.}
\examples{
library(gnlm)
# linear models
y <- matrix(rgamma(40,1,1),ncol=5)+rep(rgamma(8,0.5,1),5)
x1 <- c(rep(0,4),rep(1,4))
reps <- rmna(restovec(y),ccov=tcctomat(x1))
# independence with default gamma marginals
gnlr(y, pmu=1, psh=0, dist="gamma", env=reps)
gausscop(y, pmu=1, pshape=0, env=reps)
gausscop(y, mu=~x1, pmu=c(1,0), pshape=0, env=reps)
# AR(1)
gausscop(y, pmu=1, pshape=0, par=0.1, env=reps)
# random effect
gausscop(y, pmu=1, pshape=0, pre=0.1, env=reps)
# try other marginal distributions
gausscop(y, pmu=1, pshape=0, pre=0.1, env=reps, dist="Weibull")
gausscop(y, pmu=1, pshape=0, pre=0.1, env=reps, dist="inverse Gauss",
	stepmax=1)
gausscop(y, pmu=1, pshape=0, pre=0.1, env=reps, dist="Cauchy")
#
# first-order one-compartment model
# create data objects for formulae
dose <- c(2,5)
dd <- tcctomat(dose)
times <- matrix(rep(1:20,2), nrow=2, byrow=TRUE)
tt <- tvctomat(times)
# vector covariates for functions
dose <- c(rep(2,20),rep(5,20))
times <- rep(1:20,2)
# functions
mu <- function(p) exp(p[1]-p[3])*(dose/(exp(p[1])-exp(p[2]))*
	(exp(-exp(p[2])*times)-exp(-exp(p[1])*times)))
shape <- function(p) exp(p[1]-p[2])*times*dose*exp(-exp(p[1])*times)
lmu <- function(p) p[1]-p[3]+log(dose/(exp(p[1])-exp(p[2]))*
	(exp(-exp(p[2])*times)-exp(-exp(p[1])*times)))
lshape <- function(p) p[1]-p[2]+log(times*dose)-exp(p[1])*times
# response
#conc <- matrix(rgamma(40,shape(log(c(0.1,0.4))),
#	scale=mu(log(c(1,0.3,0.2))))/shape(log(c(0.1,0.4))),ncol=20,byrow=TRUE)
#conc[,2:20] <- conc[,2:20]+0.5*(conc[,1:19]-matrix(mu(log(c(1,0.3,0.2))),
#	ncol=20,byrow=TRUE)[,1:19])
#conc <- restovec(ifelse(conc>0,conc,0.01),name="conc")
conc <- matrix(c(3.65586845,0.01000000,0.01000000,0.01731192,1.68707608,
	0.01000000,4.67338974,4.79679942,1.86429851,1.82886732,1.54708795,
	0.57592054,0.08014232,0.09436425,0.26106139,0.11125534,0.22685364,
	0.22896015,0.04886441,0.01000000,33.59011263,16.89115866,19.99638316,
	16.94021361,9.95440037,7.10473948,2.97769676,1.53785279,2.13059515,
	0.72562344,1.27832563,1.33917155,0.99811111,0.23437424,0.42751355,
	0.65702300,0.41126684,0.15406463,0.03092312,0.14672610),
	ncol=20,byrow=TRUE)
conc <- restovec(conc)
reps <- rmna(conc, ccov=dd, tvcov=tt)
# constant shape parameter
gausscop(conc, mu=lmu, pmu=log(c(1,0.4,0.1)), par=0.5, pshape=0, envir=reps)
# or
gausscop(conc, mu=~absorption-volume+
	log(dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*times)-exp(-exp(absorption)*times))),
	pmu=list(absorption=0,elimination=log(0.4),volume=log(0.1)),
	par=0.5, pshape=0, envir=reps)
# compare to gar autoregression
gar(conc, dist="gamma", times=1:20, mu=mu,
	preg=log(c(1,0.4,0.1)), pdepend=0.5, pshape=1)
#
# time dependent shape parameter
gausscop(conc, mu=lmu, shape=lshape,
	pmu=log(c(1,0.4,0.1)), par=0.5, pshape=c(-0.1,-0.1))
# or
gausscop(conc, mu=~absorption-volume+
	log(dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*times)-exp(-exp(absorption)*times))),
	shape=~b1-b2+log(times*dose)-exp(b1)*times,
	pmu=list(absorption=0,elimination=log(0.4),volume=log(0.1)),
	par=0.5, pshape=list(b1=-0.1,b2=-0.1), envir=reps)
#
# shape depends on location
lshape <- function(p, mu) p[1]*log(abs(mu))
gausscop(conc, mu=lmu, shape=lshape, shfn=TRUE, pmu=log(c(1,0.4,0.1)),
	par=0.5, pshape=1)
# or
gausscop(conc, mu=~absorption-volume+
	log(dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*times)-exp(-exp(absorption)*times))),
	shape=~d*log(abs(mu)), shfn=TRUE,
	pmu=list(absorption=0,elimination=log(0.4),volume=log(0.1)),
	par=0.5, pshape=list(d=1), envir=reps)
}
\keyword{models}
