\name{gar}
\title{Generalized Autoregression Models}
\usage{
gar(response=NULL, distribution="normal", times=NULL, totals=NULL,
	censor=NULL, delta=NULL, mu=NULL, shape=NULL, depend=NULL,
	shfn=FALSE, common=FALSE, preg=NULL, pshape=NULL, pdepend=NULL,
	parch=NULL, arch="additive", transform="identity",
	link="identity", autocorr="exponential", order=1,
	envir=parent.frame(), print.level=0, ndigit=10,
	gradtol=0.00001, steptol=0.00001, fscale=1, iterlim=100,
	typsiz=abs(p), stepmax=10*sqrt(p\%*\%p))
volatility(z, nind=NULL)
}
\alias{gar}
\alias{volatility.gar}
\description{
\code{gar} fits a first- or second-order generalized autoregression,
possibly with Kalman update over time (first-order only).

Nonlinear regression models can be supplied as formulae where
parameters are unknowns in which case factor variables cannot be used and
parameters must be scalars. (See \code{\link[rmutil]{finterp}}.)

Marginal and individual profiles can be plotted using
\code{\link[rmutil]{mprofile}} and \code{\link[rmutil]{iprofile}} and
residuals with \code{\link[rmutil]{plot.residuals}}.

When the dispersion parameter is not constant over time,
\code{volatility} extracts the square root of the dispersion parameter
for a fitted model.
}
\arguments{
\item{response}{A list of two or three column matrices with responses,
corresponding times, and possibly a censor indicator, for each
individual, one matrix or dataframe of responses, or an object of
class, \code{response} (created by \code{\link[rmutil]{restovec}}) or
\code{repeated} (created by \code{\link[rmutil]{rmna}}). If the
\code{repeated} data object contains more than one response variable,
give that object in \code{envir} and give the name of the response
variable to be used here.}
\item{distribution}{The distribution to be fitted: binomial, Poisson,
exponential, negative binomial, mult Poisson, double Poisson, Consul
generalized Poisson, beta binomial, mult binomial, double binomial,
normal, inverse Gauss, logistic, gamma, Weibull, Cauchy, Laplace,
Levy, Pareto, beta, simplex, two-sided power, gen(eralized) gamma,
gen(eralized) logistic, Hjorth, Burr, gen(eralized) Weibull,
gen(eralized) extreme value, gen(eralized) inverse Gauss, power
exponential, power variance function Poisson, skew Laplace, or Student
t. (For definitions of distributions, see the
corresponding [dpqr]distribution help.)}
\item{times}{When response is a matrix, a vector of possibly unequally
spaced times when they are the same for all individuals or a matrix of
times. Not necessary if equally spaced. Ignored if response has class,
\code{response} or \code{repeated}.}
\item{totals}{An appropriate scalar, vector, or matrix of binomial
totals (only applicable for binomial, beta binomial, mult binomial,
double binomial). Ignored if response has class, \code{response} or
\code{repeated}.}
\item{censor}{If response is a matrix, a matrix of the same size
containing the censor indicator: 1=uncensored, 0=right-censored,
-1=left-censored. Ignored if response has class, \code{response} or
\code{repeated}.}
\item{delta}{Scalar or vector giving the unit of measurement for each
response value, set to unity by default. For example, if a response is
measured to two decimals, \code{delta=0.01}. If the response has been
pretransformed, this must be multiplied by the Jacobian. This
transformation cannot contain unknown parameters. For example, with a
log transformation, \code{delta=1/y}. (The delta values for the
censored response are ignored.) The jacobian is calculated
automatically for the transform option. Ignored if response has class,
\code{response} or \code{repeated}.}
\item{mu}{A user-specified function of \code{pmu} giving the
regression equation for the location. It may also be a formula
beginning with ~, specifying either a linear regression function for
the location parameter in the Wilkinson and Rogers notation or a
general function with named unknown parameters. It must yield a  value
for each observation on each individual.}
\item{shape}{An optional user-specified shape regression function;
this may depend on the location (function) through its second
argument, in which case, \code{shfn} must be set to TRUE. It may also
be a formula beginning with ~, specifying either a linear regression
function for the shape parameter in the Wilkinson and Rogers notation
or a general function with named unknown parameters. If it contains
unknown parameters, the keyword \code{mu} may be used to specify a
function of the location parameter.}
\item{depend}{An optional user-specified regression function for the
log dependence parameter. It may also be a formula beginning with ~,
specifying either a linear regression function for the dependence
parameter in the Wilkinson and Rogers notation or a general function
with named unknown parameters. If used, \code{order} must be one.}
\item{shfn}{If TRUE, the supplied shape function depends on the
location function. The name of this location function must be the
last argument of the shape function.}
\item{common}{If TRUE, \code{mu} and \code{shape} must both be either
functions with, as argument, a vector of parameters having some or all
elements in common between them so that indexing is in common
between them or formulae with unknowns. All parameter estimates must
be supplied in \code{preg}. If FALSE, parameters are distinct between
the two functions and indexing starts at one in each function.}
\item{preg}{The initial parameter estimates for the location
regression function. If \code{mu} is a formula with unknown
parameters, their estimates must be supplied either in their order of
appearance in the expression or in a named list.}
\item{pdepend}{One or two estimates of the dependence parameters for
the Kalman update. With one, it is Markovian and, with two, it is
nonstationary. For the latter, the \code{order} must be one. If
\code{depend} is a function or formula, the corresponding number of
estimates must be supplied. Either pdepend or parch or both must be
supplied.}
\item{parch}{Estimate for an ARCH model where the shape parameter
depends on the square of the previous residual. Either pdepend or
parch or both must be supplied.}
\item{arch}{If \code{square}, then \code{shape+parch^diff*residual^2};
if \code{absolute value}, then \code{shape+parch^diff*|residual|};
if \code{exponential}, then \code{shape*exp(parch*residual^2*diff)},
where \code{diff} is the length of time since the previous observation
and \code{residual} is the previous residual or innovation.}
\item{pshape}{Zero to two estimates for the shape parameters,
depending on the distribution, if \code{shape} is not a function;
otherwise, estimates for the parameters in this function, with one
extra at the end for three-parameter distributions. If \code{shape} is
a formula with unknown parameters, their estimates must be supplied
either in their order of appearance in the expression or in a named
list (only for two-parameter distributions).}
\item{transform}{Transformation of the response variable: \code{identity},
\code{exp}, \code{square}, \code{sqrt}, or \code{log}.}
\item{link}{Link function for the mean: \code{identity}, \code{exp},
\code{square}, \code{sqrt}, \code{log}, \code{logit}, \code{cloglog} or
\code{loglog} (last three only for binary/binomial-type data).}
\item{autocorr}{The form of the (second if two) dependence function:
\code{exponential} is the usual \eqn{\rho^{|t_i-t_j|}}{rho^|t_i-t_j|};
\code{gaussian} is \eqn{\rho^{(t_i-t_j)^2}}{rho^((t_i-t_j)^2)};
\code{cauchy} is \eqn{1/(1+\rho(t_i-t_j)^2)}{1/(1+rho(t_i-t_j)^2)};
\code{spherical} is
\eqn{((|t_i-t_j|\rho)^3-3|t_i-t_j|\rho+2)/2}{((|t_i-t_j|rho)^3-3|t_i-t_j|rho+2)/2}
for \eqn{|t_i-t_j|\leq1/\rho}{|t_i-t_j|<=1/rho} and zero otherwise;
\code{IOU} is the integrated
Ornstein-Uhlenbeck process, \eqn{(2\rho \min(t_i,t_j)+\exp(-\rho t_i)
+\exp(-\rho t_j)-1 -\exp(\rho|ti-t_j|))/2\rho^3}{(2rho min(t_i,t_j)+exp(-rho t_i)
+exp(-rho t_j)-1 -exp(rho|ti-t_j|))/2rho^3}.}
\item{order}{First- or second-order stationary autoregression.}
\item{envir}{Environment in which model formulae are to be
interpreted or a data object of class, \code{repeated}, \code{tccov},
or \code{tvcov}; the name of the response variable should be given in
\code{response}.
If \code{response} has class \code{repeated}, it is used as the
environment.}
\item{others}{Arguments controlling \code{\link{nlm}}.}
}
\references{
Lindsey, J.K. (1997) Applying Generalized Linear Models. Springer,
pp.\ 93--101

Lambert, P. (1996) Statistics in Medicine 15, 1695-1708
}
\value{
A list of classes \code{gar} and \code{recursive} is returned
that contains all of the relevant information calculated, including
error codes.

The volatility vector for models with a shape regression function and
ARCH models contains the square root of the dispersion parameter at
each time point.
}
\author{J.K. Lindsey}
\seealso{
\code{\link[growth]{carma}}, \code{\link[growth]{elliptic}},
\code{\link[rmutil]{finterp}}, \code{\link[repeated]{gnlmm}},
\code{\link[gnlm]{gnlr}}, \code{\link[rmutil]{iprofile}},
\code{\link[repeated]{kalcount}}, \code{\link[repeated]{kalseries}},
\code{\link[event]{kalsurv}}, \code{\link[rmutil]{mprofile}},
\code{\link[rmutil]{plot.residuals}}, \code{\link[rmutil]{read.list}},
\code{\link[rmutil]{restovec}}, \code{\link[rmutil]{rmna}},
\code{\link[rmutil]{tcctomat}}, \code{\link[rmutil]{tvctomat}}.
}
\examples{
# first-order one-compartment model
# data objects for formulae
dose <- c(2,5)
dd <- tcctomat(dose)
times <- matrix(rep(1:20,2), nrow=2, byrow=TRUE)
tt <- tvctomat(times)
# vector covariates for functions
dose <- c(rep(2,20),rep(5,20))
times <- rep(1:20,2)
# functions
mu <- function(p) exp(p[1]-p[3])*(dose/(exp(p[1])-exp(p[2]))*
	(exp(-exp(p[2])*times)-exp(-exp(p[1])*times)))
shape <- function(p) exp(p[1]-p[2])*times*dose*exp(-exp(p[1])*times)
# response
conc <- matrix(rgamma(40,shape(log(c(0.1,0.4))),
	scale=mu(log(c(1,0.3,0.2))))/shape(log(c(0.1,0.4))),ncol=20,byrow=TRUE)
conc[,2:20] <- conc[,2:20]+0.5*(conc[,1:19]-matrix(mu(log(c(1,0.3,0.2))),
	ncol=20,byrow=TRUE)[,1:19])
conc <- restovec(ifelse(conc>0,conc,0.01),name="conc")
reps <- rmna(conc, ccov=dd, tvcov=tt)
# constant shape parameter
gar(conc, dist="gamma", times=1:20, mu=mu,
	preg=log(c(1,0.4,0.1)), pdepend=0.5, pshape=1)
# or
gar(conc, dist="gamma", times=1:20, mu=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*times)-exp(-exp(absorption)*times)),
	preg=list(absorption=0,elimination=log(0.4),volume=log(0.1)),
	pdepend=0.5, pshape=1, envir=reps)
# generalized gamma distribution
gar(conc, dist="gen gamma", times=1:20, mu=mu,
	preg=log(c(1,0.4,0.1)), pdepend=0.5, pshape=c(1,1))
# (if the covariates contained NAs, reps would have to be used as
# response instead of conc)
#
# time dependent shape parameter
gar(conc, dist="gamma", times=1:20, mu=mu, shape=shape,
	preg=log(c(1,0.4,0.1)), pdepend=0.5, pshape=log(c(1,0.2)))
# or
gar(conc, dist="gamma", times=1:20, mu=~exp(absorption-volume)*
	dose/(exp(absorption)-exp(elimination))*
	(exp(-exp(elimination)*times)-exp(-exp(absorption)*times)),
	shape=~exp(b1-b2)*times*dose*exp(-exp(b1)*times),
	preg=list(absorption=0,elimination=log(0.4),volume=log(0.1)),
	pdepend=0.5, pshape=list(b1=0,b2=log(0.2)), envir=reps)
# generalized gamma distribution
gar(conc, dist="gen gamma", times=1:20, mu=mu, shape=shape,
	preg=log(c(1,0.4,0.1)), pdepend=0.5,
	pshape=c(log(c(1,0.2)),2))
#
# shape function depends on location parameter
shape <- function(p, mu) p[1]+p[2]*mu
gar(conc, dist="gamma", times=1:20, mu=mu, shape=shape, shfn=TRUE,
	preg=log(c(1,0.4,0.1)), pdepend=0.5, pshape=c(1,0))
# or
gar(conc, dist="gamma", times=1:20, mu=mu, shape=~a+d*mu, shfn=TRUE,
	preg=log(c(1,0.4,0.1)), pdepend=0.5, pshape=c(1,0))
}
\keyword{models}
