\name{lcr}
\title{Linear categorical regression}
\usage{
lcr(response=NULL,frequencies=NULL,mu=NULL,ccov=NULL,tvcov=NULL,
    times=F,pcoef=NULL,plink=NULL,distribution=NULL,
    direction=NULL,transformation=NULL,link=NULL,
    envir=parent.frame(),steptol=1e-04,iterlim=100)
}
\alias{lcr}
\alias{print.lcr}
\alias{extractAIC.lcr}
\alias{deviance.lcr}
\alias{fitted.lcr}
\alias{names.lcr}
\description{
  \code{lcr} is used to fit a linear categorical regression possibly
  including time-constant and/or time-varying covariates.

  Marginal and individual profiles can be plotted using
  \code{\link{moprofile}} and \code{\link{ioprofile}}.
}
\arguments{
  \item{response}{A response is a vector with cell indecees representing
    entries of a contengency table. A vector or an object of class
    \code{response} (created by \code{\link{restovec}}) or
    \code{repeated} (created by \code{\link{rmna}} or
    \code{\link{lvna}}) containing the response variable information. If
    the \code{repeated} data object contains more than one response
    variable, give that object in \code{envir} and give the name of the
    response variable to be used here.}
  \item{frequencies}{A vector containing the frequencies respectively
    corresponding to the cell entries of the contengency table described
    by the response. If the response is an object of class
    \code{response} or \code{repeated}, the frequencies should be
    included in this object as weights. If the frequencies are omited,
    it is then assumed that each response entry corresponds to an event
    (the frequencies will be by default a vector of 1's).}
  \item{mu}{A formula beginning with ~ specifying a linear regression
    function in the Wilkinson and Rogers notation. Give the initial
    estimates in \code{pcoef}.}
  \item{ccov}{A matrix or an object of class \code{tccov} (created by
    \code{\link{tcctomat}}) containing the time-constant baseline
    covariates.}
  \item{tvcov}{A matrix or an object of class \code{tvcov} (created by
    \code{\link{tvctomat}}) containing the time-varying covariate
    values.}
  \item{times}{A vector containing the times, if the \code{response} is
    not an object of class \code{repeated} or \code{response}.}
  \item{pcoef}{Initial parameter estimates for the regression model: If
    "binary", a vector containing an intercept plus one parameter for
    each covariate. If "mutinomial", a vector containing, for each
    category except the reference one, an intercept plus one parameter
    for each covariate. If "simplified mutinomial" or "continuation
    ratio" (upwards or downwards) or "proportional odds" or "adjacent
    categories", a vector containing an intercept for each category
    except the reference one plus one parameter for each covariate.}
  \item{plink}{Initial parameter estimates for the link function. These
    parameters are fixed at the given value.}
  \item{distribution}{A string containing one of: "binary" (default if
    two categories), "multinomial" (default if more than two
    categories), "simplified multinomial", "continuation-ratio",
    "proportional odds", or "adjacent categories".}
  \item{reference}{Only used if the model is "continuation ratio" then
    reference must be a string corresponding either to "upwards"
    (default) or "downwards".}
  \item{transformation}{A string containing one of: "identity"
    (default), "sqrt", "log", "logit", "cloglog", "square", "exp", or
    "loglog".}
  \item{link}{If "binary" or "proportional odds", a string containing
    one of: "logit" (default), "probit", "loglog", "cloglog", "uniform",
    "log-normal", "exponential", "Pareto", "Cauchy", "Laplace", "Levy",
    "simplex", "gamma", "Weibull", "inverse Gauss", "t", "chi-square",
    "gen logistic", "gen extreme value", "Box-Cox", "power exponential",
    "Burr", "Hjorth", "beta", "stable", "gen gamma", "gen Weibull", "gen
    inverse Gauss", "F", "nc t", "nc chi-square", "Tukey", "nc beta", or
    "nc F".}
  \item{envir}{Environment in which model formulae are to be interpreted
    or a data object of class \code{repeated}, \code{tccov}, or
    \code{tvcov}. The name of the response variable should be given in
    \code{response}.
    If \code{response} has class \code{repeated}, it is used as the
    environment.}
  \item{steptol}{A positive scalar providing the minimum allowable
    relative step length.}
  \item{iterlim}{A positive integer specifying the maximum number of
    iterations to be performed.}
}
\details{
  There are seven possible types of models that can be fitted. These are
  a binary, multinomial, simplified multinomial, continuation-ratio
  (upwards and downwards), proportional-odds, or adjacent categories.
  For each of these models, covariates can be included. All explanatory
  variables are taken as continuous. One must therefore create dummy
  binary indicator variables (for example using \code{\link{wr}}), if a
  factor variable is to be included in a model.

  To obtain the regression coefficients, the linear system of equations
  is solved by using the first and second derivatives. This guarantees
  fast and stable convergence to the maximum likelihood estimates.
}
\value{
  A list of classes \code{lcr} and \code{repeated} is returned. This
  list is composed of: the function call (call), an object of class
  \code{response} (response), an object of class \code{tvcov} (tvcov)
  containing all covariates (even the time-constant ones), the
  frequencies (frequencies), whether the response is individual data or
  a contengency table (individual), the distribution used
  (distribution), the direction for "continuation ratio" models
  (direction), the transformation used (transformation), the link used
  (link), the length of the response (n), the number of levels of the
  response (nc), the number of covariates (ncv), the total number of
  individuals or events (events), the log-likelihood (likelihood), the
  linear regression estimates (coefficients), the standard errors of
  beta (se), the covariance matrix (covariance), the correlation matrix
  (correlations), the fitted probabilities (fitted), the categories
  containing the highest probabilities (pred), the cumulative
  probabilities (cpred), the number of iterations performed
  (iterations), the iteration limit (iterlim), the QR decomposition
  exact singularity indicator (info), and the second derivative
  singularity indicator (rank).
}
\author{P.J. Lindsey}
\seealso{
  \code{\link{ioprofile}}, \code{\link{lvna}}, \code{\link{moprofile}},
  \code{\link{rmna}}, \code{\link{restovec}}, \code{\link{tcctomat}},
  \code{\link{tvctomat}}.
}
\examples{
library(ordinal)

#
# Binary example:
#
data(cardiac)

resp <- cardiac[,1:4]
freq <- cardiac[,11]

age <- cardiac[,5]
sex <- cardiac[,6]
pmi <- cardiac[,7]
dia <- cardiac[,8]
ren <- cardiac[,9]
cop <- cardiac[,10]

rm(cardiac)

y <- restovec(resp,times=T,weights=freq,type="ordinal")

tcc <- tcctomat(age,name="age")
tcc <- tcctomat(sex,name="sex",oldccov=tcc)
tcc <- tcctomat(pmi,name="pmi",oldccov=tcc)
tcc <- tcctomat(dia,name="dia",oldccov=tcc)
tcc <- tcctomat(ren,name="ren",oldccov=tcc)
tcc <- tcctomat(cop,name="cop",oldccov=tcc)

tvc <- tvctomat(matrix(times(y)^2,ncol=4,byrow=T),name="times2")

w <- rmna(y,ccov=tcc,tvcov=tvc)

rm(resp,freq,age,sex,pmi,dia,ren,cop,y,tcc,tvc)

lcr(w)
lcr(w,mu=~times)
lcr(w,mu=~age+sex+times+times2)
lcr(w,mu=~age+sex+pmi+dia+ren+cop+pmi:dia+pmi:cop+dia:cop+times)
lcr(w,mu=~age+sex+pmi+dia+ren+cop+pmi:dia+pmi:cop+dia:cop+times+times2)

rm(w)

#
# Multinomial example:
#
data(criminal)

resp <- criminal[,2]-1
freq <- criminal[,5]

race <- criminal[,3]-1
county <- criminal[,4]-1
trial <- as.factor(criminal[,1])

rm(criminal)

y <- restovec(resp,times=F,weights=freq,type="ordinal")

tcc <- tcctomat(race,name="race")
tcc <- tcctomat(county,name="county",oldccov=tcc)
tcc <- tcctomat(trial,name="trial",oldccov=tcc)

w <- rmna(y,ccov=tcc)

rm(resp,freq,race,county,trial,y,tcc)

lcr(w)
lcr(w,mu=~race)
lcr(w,mu=~race+county)
lcr(w,mu=~race+county+trial)

#
# Simplified multinomial example:
#
lcr(w,dist="simpl")
lcr(w,dist="simpl",mu=~race)
lcr(w,dist="simpl",mu=~race+county)
lcr(w,dist="simpl",mu=~race+county+trial)

rm(w)

#
# Continuation ratio (upwards) example:
#
data(tmi)

resp <- tmi[,1:4]-1
freq <- tmi[,6]

cov <- matrix(tmi[,5],ncol=1,dimnames=list(NULL,"distance"))

rm(tmi)

y <- restovec(resp,times=T,weights=freq,type="ordinal")

tcc <- tcctomat(cov,name="distance")

w <- rmna(y,ccov=tcc)

rm(resp,freq,cov,y,tcc)

lcr(w,distribution="cont")
lcr(w,mu=~distance,distribution="cont")
lcr(w,mu=~distance+times,distribution="cont")

#
# Continuation ratio (downwards) example:
#
lcr(w,distribution="cont",direc="down")
lcr(w,distribution="cont",direc="down",mu=~distance)
lcr(w,distribution="cont",direc="down",mu=~distance+times)

#
# Proportional odds example:
#
lcr(w,distribution="prop")
lcr(w,distribution="prop",mu=~distance)
lcr(w,distribution="prop",mu=~distance+times)

#
# Adjacent categories example:
#
lcr(w,distribution="adj")
lcr(w,distribution="adj",mu=~distance)
lcr(w,distribution="adj",mu=~distance+times)

#
# Example using links:
#
lcr(w,distribution="prop",mu=~distance)
lcr(w,distribution="prop",mu=~distance,link="probit",pcoef=c(-1.3634,0.589,0.1709))
lcr(w,distribution="prop",mu=~distance,link="log-normal",transform="log",pcoef=c(-1.3634,0.589,0.1709))
lcr(w,distribution="prop",mu=~distance,link="Cauchy",pcoef=c(-2.9979,0.6297,0.5164))
lcr(w,distribution="prop",mu=~distance,link="Laplace",pcoef=c(-1.6946,0.5448,0.2531))
lcr(w,distribution="prop",mu=~distance,link="gen logistic",pcoef=c(-1.691,3.2053,0.4578),plink=2)
lcr(w,distribution="prop",mu=~distance,link="gen extreme value",transform="logit",pcoef=c(-2.9815,2.3526,0.6966),plink=1)
lcr(w,distribution="prop",mu=~distance,link="gen extreme value",transform="logit",pcoef=c(-5.8473,-1.6886,0.3214),plink=1/2)
lcr(w,distribution="prop",mu=~distance,link="power exponential",pcoef=c(-3.3892,1.0895,0.5061),plink=1/2)
lcr(w,distribution="prop",mu=~distance,link="stable",pcoef=c(-1.9281,0.8331,0.2417),plink=c(0,2))
lcr(w,distribution="prop",mu=~distance,link="nc F",transform="log",pcoef=c(-2.8999,2.2461,0.5254),plink=c(1,1,1))
lcr(w,distribution="prop",mu=~distance,link="nc F",transform="log",pcoef=c(-3.318,2.091,0.382),plink=c(1/2,3,2))

rm(w)
}
\keyword{models}
