/***************************************************************************
 *   Copyright (C) 2008-2009 by fra74                                           *
 *   francesco.b74@gmail.com                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "schedastazione.h"
#include "qviaggiatreno.h"
//
SchedaStazione::SchedaStazione(QViaggiaTreno* parent, const QString& nome, const unsigned int intervalloStandard)
	: SchedaViaggiaTreno(parent, tsStazione, intervalloStandard)
{
	m_stazione = nome;
	m_codice = "";
	m_stato = statoNuovaScheda;
	m_documentoDom = 0L;
	m_idTabella = m_idNomeSbagliato = m_idNomeAmbiguo = -1;
	
	//crea il widget della scheda
	m_idTabella = addWidget(creaWidgetStazione());
	setCurrentIndex(m_idTabella);

	//connessioni
	connect(this, SIGNAL(statoCambiato(quint32)), parent, SLOT(aggiornaStatoScheda(quint32)));
	connect(this, SIGNAL(nomeSchedaCambiato(quint32)), parent, SLOT(aggiornaNomeScheda(quint32)));
	connect(this, SIGNAL(apriSchedaStazione(const QString&)), parent, SLOT(nuovaStazione(const QString&)));
	connect(this, SIGNAL(apriSchedaTreno(const QString&)), parent, SLOT(nuovoTreno(const QString&)));
	connect(this, SIGNAL(messaggioStatus(const QString&)), parent, SLOT(mostraMessaggioStatusBar(const QString&)));
}

//restituisce il nome della stazione
//seguendo la convenzione di ViaggiaTreno il nome della stazione viene restituito in maiuscolo
QString SchedaStazione::nomeStazione() const
{
	return m_stazione.toUpper();
}

//restituisce una stringa con il "titolo" di questa scheda, che viene ad esempio usato come etichetta della TabBar
// se titoloItem == true restituisce un titolo per l'item dell'albero
//seguendo la convenzione di Viaggiatreno il nome della stazione verrà scritto tutto in maiuscolo
QString SchedaStazione::titolo( bool titoloItem ) const 
{
	if (titoloItem)
		return nomeStazione();
	
	
	return QString::fromUtf8("Stazione di %1").arg(nomeStazione());
}

//questo slot viene richiamato quando scatta il timer
void SchedaStazione::aggiorna()
{
	cambiaStato(statoInAggiornamento);
	
	//cancella dallo StackedWidget i widget creati per gestire gli errori
	if (m_idNomeSbagliato != -1)
	{
		QWidget *widgetDaRimuovere = widget(m_idNomeSbagliato);
		widgetDaRimuovere->deleteLater();
		m_idNomeSbagliato = -1;
	}

	if (m_idNomeAmbiguo != -1)
	{
		QWidget *widgetDaRimuovere = widget(m_idNomeAmbiguo);
		widgetDaRimuovere->deleteLater();
		m_idNomeAmbiguo = -1;
	}


	QNetworkRequest request;
	QString dati;
	//costruisce i dati da inviare al server sotto forma di richiesta POST
	if (m_codice.isEmpty())	
		dati = QString("stazione=%1").arg(nomeStazione());
	else
		dati = QString("codiceStazione=%1").arg(codiceStazione());
	dati.replace(" ", "+");
	request.setUrl(QUrl("http://mobile.viaggiatreno.it/viaggiatreno/mobile/stazione?lang=IT"));
	request.setRawHeader("Content-type", "application/x-www-form-urlencoded");
	networkAM()->post(request, dati.toUtf8());
}

//questo slot viene richiamato quando è finito il download della risposta dal sito di viaggiatreno
void SchedaStazione::downloadFinito(QNetworkReply *risposta)
{
	QString messaggioErrore;
	int riga, colonna;
	
	QString rispostaVT = QString::fromUtf8(risposta->readAll());
	rispostaVT = correggiOutputVT(rispostaVT);
	//cerca nel testo la stringa "non trovata", visto che quando VT non trova una stazione visualizza
	//una casella di testo con di fianco la scritta "località non trovata"
	//se viene trovata visualizza il widget con il messaggio di errore e cambia lo stato
	if (rispostaVT.contains("non trovata", Qt::CaseInsensitive))
	{
		ferma();
		m_idNomeSbagliato = addWidget(creaWidgetErroreNome());
		setCurrentIndex(m_idNomeSbagliato);
		cambiaStato(statoErrore);
		return;
	}
	
	//cerca nel tempo la stringa "option", se è presente vuol dire che viaggiatreno ha preparato un form
	//per la selezione del nome della stazione da una lista, in quanto il nome introdotto è ambiguo
	//in tale caso prepara un widget per la selezione
	if (rispostaVT.contains("option", Qt::CaseInsensitive))
	{
		ferma();
		m_idNomeAmbiguo = addWidget(new WidgetDisambiguaNome(rispostaVT, this));
		setCurrentIndex(m_idNomeAmbiguo);
		cambiaStato(statoErrore);
		return;
	}

	//rispostaVT = correggiErroriVT(rispostaVT);
	if (m_documentoDom)
		delete m_documentoDom;

	m_documentoDom = new QDomDocument;
	
	if (!m_documentoDom->setContent(rispostaVT, &messaggioErrore, &riga, &colonna ))
	{
		//errore nel parsing
		cambiaStato(statoErrore);
		QMessageBox msgBox;
		msgBox.setIcon(QMessageBox::Warning);
		msgBox.setText(QString::fromUtf8("Errore durante l'analisi della risposta di ViaggiaTreno. Se il problema persiste contattare l'autore"));
		msgBox.setDetailedText(QString::fromUtf8("Ricevuto messaggio di errore %1 alla riga %2, colonna %3.\nTesto:\n %4").arg(messaggioErrore).arg(riga).arg(colonna).arg(rispostaVT));
		msgBox.exec();
		return;
	}
	
	analizzaDocumentoDom();
	
	
	m_ultimoAgg = QDateTime::currentDateTime();
	m_labelAggiornamento->setText(QString::fromUtf8("Ultimo aggiornamento: %1").arg(m_ultimoAgg.toString(Qt::DefaultLocaleShortDate)));
	cambiaStato(statoMonitoraggioAttivo);
	emit messaggioStatus(QString::fromUtf8("Aggiornati dati stazione di %1").arg(titolo(true)));

	risposta->deleteLater();
	
}

QWidget* SchedaStazione::creaWidgetStazione()
{
	QWidget *widget = new QWidget;
	//layout principale
	QVBoxLayout *vbox = new QVBoxLayout;

	//costruisce l'etichetta per il titolo	
	m_labelTitolo = new QLabel(widget);
	//l'etichetta sarà centrata
	m_labelTitolo->setAlignment(Qt::AlignCenter);
	m_labelTitolo->setText(titolo());
	//imposta font con dimensioni del 50% maggiori di quella standard e in grassetto
	QFont font = m_labelTitolo->font();
	font.setBold(true);
        font.setPointSize(font.pointSize()*1.5);
	m_labelTitolo->setFont(font);
	//aggiunge l'etichetta al layout
	vbox->addWidget(m_labelTitolo, 0);
	
	//costruisce l'etichetta con ora/data di aggiornamento
	m_labelAggiornamento = new QLabel(widget);
	//l'etichetta sarà centrata
	m_labelAggiornamento->setAlignment(Qt::AlignCenter);
	m_labelAggiornamento->setText(QString::fromUtf8("Ultimo aggiornamento: aggiornamento in corso"));
	vbox->addWidget(m_labelAggiornamento, 0);
	
	QSplitter *splitter = new QSplitter(Qt::Vertical, widget);

	QWidget *widgetArrivi = new QWidget(widget);
	QVBoxLayout *vboxArrivi = new QVBoxLayout;
	QLabel *labelArrivi = new QLabel(widgetArrivi);
	labelArrivi->setAlignment(Qt::AlignCenter);
	labelArrivi->setText(QString::fromUtf8("<b>Arrivi</b>"));
	vboxArrivi->addWidget(labelArrivi);
	m_tabellaArrivi = new TabellaStazione(widgetArrivi);
	QStringList etichetteArrivi;
	m_tabellaArrivi->setColumnCount(8);
	etichetteArrivi<<QString::fromUtf8("Categoria")<<QString::fromUtf8("Numero")<<QString::fromUtf8("Origine")
		<<QString::fromUtf8("Bin. Programmato")<<QString::fromUtf8("Bin. reale")<<QString::fromUtf8("Orario teorico")<<QString::fromUtf8("Ritardo")<<"";
	m_tabellaArrivi->setHorizontalHeaderLabels(etichetteArrivi);
	

	vboxArrivi->addWidget(m_tabellaArrivi, 1);
	widgetArrivi->setLayout(vboxArrivi);
	splitter->addWidget(widgetArrivi);
	
	QWidget *widgetPartenze = new QWidget(widget);
	QVBoxLayout *vboxPartenze = new QVBoxLayout;
	QLabel *labelPartenze = new QLabel(widgetArrivi);
	labelPartenze->setAlignment(Qt::AlignCenter);
	labelPartenze->setText(QString::fromUtf8("<b>Partenze</b>"));
	vboxPartenze->addWidget(labelPartenze);
	m_tabellaPartenze = new TabellaStazione(widgetPartenze);
	QStringList etichettePartenze;
	m_tabellaPartenze->setColumnCount(8);
	etichettePartenze<<QString::fromUtf8("Categoria")<<QString::fromUtf8("Numero")<<QString::fromUtf8("Destinazione")
		<<QString::fromUtf8("Bin. Programmato")<<QString::fromUtf8("Bin. reale")<<QString::fromUtf8("Orario teorico")<<QString::fromUtf8("Ritardo")<<"";
	m_tabellaPartenze->setHorizontalHeaderLabels(etichettePartenze);
	QPalette paletta = QApplication::palette();
	paletta.setColor(QPalette::Base, QColor(255, 244, 160));
	paletta.setColor(QPalette::AlternateBase, QColor(255, 244, 100));
	m_tabellaPartenze->setPalette(paletta);

	vboxPartenze->addWidget(m_tabellaPartenze, 1);
	widgetPartenze->setLayout(vboxPartenze);
	splitter->addWidget(widgetPartenze);
	
	vbox->addWidget(splitter, 1);
	widget->setLayout(vbox);

	//connect(m_tabellaArrivi, SIGNAL(itemActivated(QTableWidgetItem *)), this, SLOT(itemAttivato(QTableWidgetItem*)));
	//connect(m_tabellaPartenze, SIGNAL(itemActivated(QTableWidgetItem *)), this, SLOT(itemAttivato(QTableWidgetItem*)));

	connect(m_tabellaArrivi, SIGNAL(itemClicked(QTableWidgetItem *)), this, SLOT(itemAttivato(QTableWidgetItem*)));
	connect(m_tabellaPartenze, SIGNAL(itemClicked(QTableWidgetItem *)), this, SLOT(itemAttivato(QTableWidgetItem*)));

	return widget;
}

QWidget* SchedaStazione::creaWidgetErroreNome()
{
	QWidget* widget = new QWidget(this);
	
	QVBoxLayout * vbox = new QVBoxLayout;
	widget->setLayout(vbox);
	QLabel *labelErrore = new QLabel(this);
	labelErrore->setWordWrap(true);
	labelErrore->setText(QString::fromUtf8("Il nome della stazione ('<b>%1</b>') non è stato riconosciuto"
		" da ViaggiaTreno. Premete il pulsante sottostante per modificare il nome della stazione, "
		"oppure il pulsante di chiusura se desiderate semplicemente chiudere questa scheda").arg(nomeStazione()));
	QPushButton *btnCambiaNome = new QPushButton(QString::fromUtf8("Cambia nome"), this);
	connect(btnCambiaNome, SIGNAL(clicked()), this, SLOT(modificaNomeStazione()));
	vbox->addWidget(labelErrore);
	vbox->addWidget(btnCambiaNome, 0, Qt::AlignCenter);
	vbox->addStretch();
	
	return widget;
}

void SchedaStazione::modificaNomeStazione()
{
	bool ok;
	QString nuovoNome = QInputDialog::getText(this, QString::fromUtf8("Modifica nome stazione"), QString::fromUtf8("Inserire il nuovo nome della stazione:"),
		QLineEdit::Normal, nomeStazione(), &ok);
		
	if (ok & !nuovoNome.isEmpty())
	{
		m_stazione = nuovoNome;
		m_codice = "";
		m_labelTitolo->setText(titolo());
		//assicurarsi che venga visualizzato il widget della stazione e non il widget di errore
		setCurrentIndex(m_idTabella);
		emit(nomeSchedaCambiato(idScheda()));
		aggiorna();
		avvia();
	}
}

void SchedaStazione::analizzaDocumentoDom()
{
	if (!m_documentoDom)
		return;
	
	//recupera l'elemento body
	QDomElement body = m_documentoDom->documentElement().firstChildElement("body");
	
	//visto che il parser non ha rilevato problemi si fa l'assunzione che la pagina inviata da viaggiatreno sia corretta
	//quindi si suppone che l'elemento body esista
	//in future versioni si implementerà probabilmente un algoritmo più robusto
	
	//ottiene la lista di tutti i nodi figli
	QDomNodeList nodi = body.childNodes();
	//debugNodo(body);
	
	//recupera nome della stazione dal documento DOM
	QDomElement h1 = body.firstChildElement("h1");
	m_stazione = h1.text().section(' ', 2);
	emit(nomeSchedaCambiato(idScheda()));
	m_labelTitolo->setText(titolo());
	
	
	uint inizioArrivi, fineArrivi, inizioPartenze, finePartenze;
	inizioArrivi = fineArrivi = inizioPartenze = finePartenze = -1;
	//scorre la lista alla ricerca di tutti i commenti	
	for (uint idx = 0; idx < nodi.length(); idx++)
	{
		if (nodi.at(idx).isComment())
		{
			if (nodi.at(idx).toComment().data().contains("partenze", Qt::CaseInsensitive))
			{
				inizioPartenze = idx+1;
				//individua il commento successivo che identifica la fine del blocco di arrivi o partenze
				for (uint j = idx +1; j < nodi.length(); j++)
				{
					if (nodi.at(j).isComment())
					{
						
						finePartenze = j-1;
						idx = j;
						break;
					}
				}
			}
						
			if (nodi.at(idx).toComment().data().contains("arrivi", Qt::CaseInsensitive))
			{
				inizioArrivi = idx+1;
				//individua il commento successivo che identifica la fine del blocco di arrivi o partenze
				for (uint j = idx +1; j < nodi.length(); j++)
				{
					if (nodi.at(j).isComment())
					{
						fineArrivi = j-1;
						idx = j;
						break;
					}
				}
			}
		}
	}
	
	//pulisci le liste
	m_arrivi.clear();
	m_partenze.clear();
	
	//itera lungo la lista dei nodi, partendo dal primo elemento che rappresenta un treno in partenza
	//e finendo con l'ultimo elemento. I rispettivi indici sono memorizzati in idxInizioPartenze, idxFinePartenze
	for (uint j=inizioPartenze, i=0 ; j <= finePartenze; j++,i++)
	{
		Stazione::DatiTreno treno;
		//recupera l'elemento dom corrente
		QDomElement elemento = nodi.at(j).toElement();
				
		//categoria e numero treno sono memorizzati in un elemento "h2", recuperiamolo
		QDomElement elementoh2 = elemento.firstChildElement("h2").toElement();
		
		//recupera dall'elemento h2 il testo, e lo separa in due stringhe... la prima conterrà la categoria
		//...la seconda il numero 
		QStringList catNumeroTreno = elementoh2.text().split(" ");
			
		//copia le rispettive stringhe nella variabile che memorizza i dati del treno
		treno.impostaCategoria(catNumeroTreno[0]);
		treno.impostaNumero(catNumeroTreno[1]);
		
		//ricava dai due elementi "<strong>" nome della stazione di origine/destinazione 
		//e l'orario teorico di arrivo/partenza		
		QDomElement bloccotreno = elemento.firstChildElement("div");
		QDomElement stazione = bloccotreno.firstChildElement("strong");
		
		QDomElement orario = stazione.nextSiblingElement("strong");
		treno.impostaStazione(stazione.text());
		treno.impostaOrario(orario.text());
		
		QString datitreno = bloccotreno.text();
		
		QStringList lista = datitreno.split(" ", QString::SkipEmptyParts);
		
		QStringListIterator it(lista);
		
		//cerca la stringa "Previsto:"
		it.findNext("Previsto:");
		QString binario = it.next();
		if (binario == "--")
			binario = QString::fromUtf8("Sconosciuto");
		else
		{
			//la string successiva o è "Binario Reale:" o fa ancora parte del numero binario
			while (true)
			{
				QString next = it.next();
				if (next != "Binario")
					binario += QString(" %1").arg(next);
				else
					break;
			}
			
		}
		
		treno.impostaBinarioProgrammato(binario);
				
		//riporta l'iteratore in cima alla lista
		it.toFront();
		//cerca la stringa "Reale:"
		it.findNext("Reale:");
		binario = it.next();
		if (binario == "--")
			binario = QString::fromUtf8("Sconosciuto");
		else
		{
			//la stringa successiva o è "in" ("in orario") o è "ritardo" (ritardo x minuti)
			//oppure fa ancora parte del nome del binario
			while (true)
			{
				QString next = it.next();
				if ((next != "in") && (next != "ritardo"))
					binario += QString(" %1").arg(next);
				else
					break;
			}
		}
		
		treno.impostaBinarioReale(binario);
				
		//individua se il treno è in orario o in ritardo
		it.toFront();
		
		if (it.findNext("orario"))
			treno.impostaRitardo(QString::fromUtf8("In orario"));
		else
		{
			//il treno non è indicato in orario (l'indicazione vale anche per treni in anticipo
			//quindi è in ritardo
			//il numero di minuti di ritardo è nel penultimo elemento della lista
			it.toBack();
			it.previous();
			treno.impostaRitardo(QString::fromUtf8("ritardo %1 minuti").arg(it.previous()));
		}
	m_partenze.append(treno);
	}
	
	//ripete la procedura anche per gli arrivi
	for (uint j=inizioArrivi, i=0 ; j <= fineArrivi; j++,i++)
	{
		Stazione::DatiTreno treno;
		//recupera l'elemento dom corrente
		QDomElement elemento = nodi.at(j).toElement();
				
		//categoria e numero treno sono memorizzati in un elemento "h2", recuperiamolo
		QDomElement elementoh2 = elemento.firstChildElement("h2").toElement();
		
		//recupera dall'elemento h2 il testo, e lo separa in due stringhe... la prima conterrà la categoria
		//...la seconda il numero 
		QStringList catNumeroTreno = elementoh2.text().split(" ");
			
		//copia le rispettive stringhe nella variabile che memorizza i dati del treno
		treno.impostaCategoria(catNumeroTreno[0]);
		treno.impostaNumero(catNumeroTreno[1]);
		
		//ricava dai due elementi "<strong>" nome della stazione di origine/destinazione 
		//e l'orario teorico di arrivo/partenza		
		QDomElement bloccotreno = elemento.firstChildElement("div");
		QDomElement stazione = bloccotreno.firstChildElement("strong");
		
		QDomElement orario = stazione.nextSiblingElement("strong");
		treno.impostaStazione(stazione.text());
		treno.impostaOrario(orario.text());
		
		QString datitreno = bloccotreno.text();
		
		QStringList lista = datitreno.split(" ", QString::SkipEmptyParts);
		
		QStringListIterator it(lista);
		
		//cerca la stringa "Previsto:"
		it.findNext("Previsto:");
		QString binario = it.next();
		if (binario == "--")
			binario = QString::fromUtf8("Sconosciuto");
		else
		{
			//la string successiva o è "Binario Reale:" o fa ancora parte del numero binario
			while (true)
			{
				QString next = it.next();
				if (next != "Binario")
					binario += QString(" %1").arg(next);
				else
					break;
			}
			
		}
		
		treno.impostaBinarioProgrammato(binario);
				
		//riporta l'iteratore in cima alla lista
		it.toFront();
		//cerca la stringa "Reale:"
		it.findNext("Reale:");
		binario = it.next();
		if (binario == "--")
			binario = QString::fromUtf8("Sconosciuto");
		else
		{
			//la stringa successiva o è "in" ("in orario") o è "ritardo" (ritardo x minuti)
			//oppure fa ancora parte del nome del binario
			while (true)
			{
				QString next = it.next();
				if ((next != "in") && (next != "ritardo"))
					binario += QString(" %1").arg(next);
				else
					break;
			}
		}
		
		treno.impostaBinarioReale(binario);
				
		//individua se il treno è in orario o in ritardo
		it.toFront();
		
		if (it.findNext("orario"))
			treno.impostaRitardo(QString::fromUtf8("In orario"));
		else
		{
			//il treno non è indicato in orario (l'indicazione vale anche per treni in anticipo
			//quindi è in ritardo
			//il numero di minuti di ritardo è nell'ultimo elemento della lista
			it.toBack();
			treno.impostaRitardo(QString::fromUtf8("ritardo %1 minuti").arg(it.previous()));
		}
	m_arrivi.append(treno);
	}

	riempiTabelle();	
}



void SchedaStazione::riempiTabelle()
{

	int riga = -1;
	QVector<int> larghezzaTabArrivi(Stazione::colUltima+1), larghezzaTabPartenze(Stazione::colUltima+1);
	
	//pulisci le tabelle
	m_tabellaArrivi->clearContents();
	m_tabellaPartenze->clearContents();
	
	//imposta il numero corretto di righe
	m_tabellaArrivi->setRowCount(m_arrivi.count());
	m_tabellaPartenze->setRowCount(m_partenze.count());
	
	//inizializza il vettore con la larghezza delle varie colonne
	//con i valori della larghezza degli header
	for (int i = 0; i <= Stazione::colUltima ; i++)
		larghezzaTabPartenze[i] = m_tabellaPartenze->horizontalHeader()->sectionSizeHint(i);


	//scorre la lista dei treni in partenza
	QListIterator<Stazione::DatiTreno> it(m_partenze);
	while (it.hasNext())
	{
		QTableWidgetItem* item;
		riga++;

		Stazione::DatiTreno treno = it.next();
		// imposta le varie colonne nella tabella
		//le funzioni per ridimensionare la larghezza delle colonne in QTableWidget
		//ad esempio ResizeColumnsToContent o quelle di QHeaderView hanno effetto
		//solo sulla parte visible della tabella.
		//Per ridimensionare correttamente la colonna è necessario procedere "a mano"
		//Di conseguenza mentre si riempie la tabella teniamo conto della larghezza delle varie colonne,
		//in particolare del massimo valore di larghezza per ogni singola colonna
		item = new QTableWidgetItem(treno.categoria());
		m_tabellaPartenze->setItem(riga, Stazione::colCategoria, item);
		larghezzaTabPartenze[Stazione::colCategoria] = qMax(larghezzaTabPartenze[Stazione::colCategoria], larghezzaItem(item));
		//nel caso delle colonne numero treno e stazione crea un item con un font sottolineato e con il colore dei link, a simulare la presenza
		//di un link, per far capire che cliccando si apre la scheda corrispondente
		item = new QTableWidgetItem(treno.numero());
		itemSottolineato(item);
		m_tabellaPartenze->setItem(riga, Stazione::colNumero, item);
		larghezzaTabPartenze[Stazione::colNumero] = qMax(larghezzaTabPartenze[Stazione::colNumero], larghezzaItem(item));
		item = new QTableWidgetItem(treno.stazione());
		itemSottolineato(item);
		m_tabellaPartenze->setItem(riga, Stazione::colStazione, item);
		larghezzaTabPartenze[Stazione::colStazione] = qMax(larghezzaTabPartenze[Stazione::colStazione], larghezzaItem(item));
		item = new QTableWidgetItem(treno.binarioProgrammato());
		//allinea il testo centrandolo nella cologna
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaPartenze->setItem(riga, Stazione::colBinProgrammato, item);
		larghezzaTabPartenze[Stazione::colBinProgrammato] = qMax(larghezzaTabPartenze[Stazione::colBinProgrammato], larghezzaItem(item));
		item = new QTableWidgetItem(treno.binarioReale());
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaPartenze->setItem(riga, Stazione::colBinReale, item);
		larghezzaTabPartenze[Stazione::colBinReale] = qMax(larghezzaTabPartenze[Stazione::colBinReale], larghezzaItem(item));
		item = new QTableWidgetItem(treno.orario());
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaPartenze->setItem(riga, Stazione::colOrario, item);
		larghezzaTabPartenze[Stazione::colOrario] = qMax(larghezzaTabPartenze[Stazione::colOrario], larghezzaItem(item));
		item = new QTableWidgetItem(treno.ritardo());
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaPartenze->setItem(riga, Stazione::colRitardo, item);
		larghezzaTabPartenze[Stazione::colRitardo] = qMax(larghezzaTabPartenze[Stazione::colRitardo], larghezzaItem(item));
	}

	//imposta la larghezza delle colonne della tabella con i valori appena memorizzati
	//aggiungendo un margine di 10 punti
	for (int i = 0; i <= Stazione::colUltima; i++)
		m_tabellaPartenze->setColumnWidth(i, larghezzaTabPartenze[i]+10);

	//scorre la lista dei treni in arrivo
	it = QListIterator<Stazione::DatiTreno>(m_arrivi);
	riga = -1;

	for (int i = 0; i <= Stazione::colUltima ; i++)
		larghezzaTabArrivi[i] = m_tabellaArrivi->horizontalHeader()->sectionSizeHint(i);

	while (it.hasNext())
	{
		QTableWidgetItem* item;
		riga++;
		Stazione::DatiTreno treno = it.next();

		item = new QTableWidgetItem(treno.categoria());
		m_tabellaArrivi->setItem(riga, Stazione::colCategoria, item);
		larghezzaTabArrivi[Stazione::colCategoria] = qMax(larghezzaTabArrivi[Stazione::colCategoria], larghezzaItem(item));
		item = new QTableWidgetItem(treno.numero());
		itemSottolineato(item);
		m_tabellaArrivi->setItem(riga, Stazione::colNumero, item);
		larghezzaTabArrivi[Stazione::colNumero] = qMax(larghezzaTabArrivi[Stazione::colNumero], larghezzaItem(item));
		item = new QTableWidgetItem(treno.stazione());
		itemSottolineato(item);
		m_tabellaArrivi->setItem(riga, Stazione::colStazione, item);
		larghezzaTabArrivi[Stazione::colStazione] = qMax(larghezzaTabArrivi[Stazione::colStazione], larghezzaItem(item));
		item = new QTableWidgetItem(treno.binarioProgrammato());
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaArrivi->setItem(riga, Stazione::colBinProgrammato, item);
		larghezzaTabArrivi[Stazione::colBinProgrammato] = qMax(larghezzaTabArrivi[Stazione::colBinProgrammato], larghezzaItem(item));
		item = new QTableWidgetItem(treno.binarioReale());
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaArrivi->setItem(riga, Stazione::colBinReale, item);
		larghezzaTabArrivi[Stazione::colBinReale] = qMax(larghezzaTabArrivi[Stazione::colBinReale], larghezzaItem(item));
		item = new QTableWidgetItem(treno.orario());
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaArrivi->setItem(riga, Stazione::colOrario, item);
		larghezzaTabArrivi[Stazione::colOrario] = qMax(larghezzaTabArrivi[Stazione::colOrario], larghezzaItem(item));
		item = new QTableWidgetItem(treno.ritardo());
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaArrivi->setItem(riga, Stazione::colRitardo, item);
		larghezzaTabArrivi[Stazione::colRitardo] = qMax(larghezzaTabArrivi[Stazione::colRitardo], larghezzaItem(item));
	}

	for (int i = 0; i <= Stazione::colUltima; i++)
		m_tabellaArrivi->setColumnWidth(i, larghezzaTabArrivi[i]+10);
	
}

void SchedaStazione::itemAttivato(QTableWidgetItem* item)
{
	int col = item->column();
	
	if (col == Stazione::colNumero)
		emit(apriSchedaTreno(item->text()));
	
	if (col == Stazione::colStazione)
		emit(apriSchedaStazione(item->text()));
	
}

//slot
//cambia il codice della stazione e riaggiorna la scheda
void SchedaStazione::cambiaCodiceStazione(const QString& nuovoCodice)
{
	m_codice = nuovoCodice;
	setCurrentIndex(m_idTabella);
	aggiorna();
	avvia();	
}


//Salva i dati necessari per poter ripristinare la scheda alla prossima apertura
void SchedaStazione::salvaScheda(QSettings& settings)
{
	settings.setValue("tipo scheda", "stazione");
	settings.setValue("intervallo", intervallo());
	settings.setValue("nome stazione", nomeStazione());
	settings.setValue("codice stazione", codiceStazione());
}
