/* Copyright (C) 2004 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#include "config.h"

#include <qstatusbar.h>
#include <qmessagebox.h>
#include <qtextbrowser.h>
#include <qlineedit.h>

// For the version strings.
#include "trd.h"
#include "vmvsn.h"

// For retrieving game information.
#include "gameinfo.h"

#include "qtadsio.h"
#include "qtadssettings.h"
#include "confdialog.h"
#include "gameinformation.h"
#include "about.h"
#include "shortcuts.h"
#include "qtadstimedisplay.h"
#include "qtadswarning.h"


void
QTadsMainWindow::init()
{
	// Init and add the "Themes" submenu to the "Display" menu.
	// Make it a child of the "Display" menu so we won't have to
	// delete it manually.
	this->fThemesMenu = new QPopupMenu(this->displayMenu, "themes submenu");
	// Make the theme menu's items "checkable" (the currently
	// active theme will appear as checked).
	this->fThemesMenu->setCheckable(true);
	// Add it as the first entry of the "Display" menu.
	this->displayMenu->insertItem(tr("Theme"), this->fThemesMenu, -1, 0);
	// When the user clicks on a theme-name, activate it.
	connect(this->fThemesMenu, SIGNAL(activated(int)), SLOT(switchToTheme(int)));

	// Init and add the "Recent Games" submenu to the "Game" menu.
	this->fRecentMenu = new QPopupMenu(this->gameMenu, "recent games submenu");
	this->gameMenu->insertItem(tr("Recent Games"), this->fRecentMenu, -1, 0);
	connect(this->fRecentMenu, SIGNAL(activated(int)), SLOT(loadRecentGame(int)));

	// Add the time display to the statusbar.
	this->fClockDisplay = new QTadsTimeDisplay(this->statusBar());
	this->statusBar()->addWidget(this->fClockDisplay, 0, true);
}


void
QTadsMainWindow::editConfigure()
{
	QTadsConfDialog dialog(this, "configuration dialog", true);
	dialog.exec();
}


void
QTadsMainWindow::displayFullScreen( bool yes )
{
	QTadsIO::fullScreen(yes);
}


void
QTadsMainWindow::gameQuit()
{
	if (QTadsIO::gameRunning) {
		QTadsIO::quit();
	} else {
		QTadsIO::quitApp = true;
	}
}


void
QTadsMainWindow::closeEvent( QCloseEvent* )
{
	this->gameExit();
}


void
QTadsMainWindow::displayDockableTool( bool yes )
{
	this->toolBar->setMovingEnabled(yes);
}


void
QTadsMainWindow::gameRestore()
{
	if (QTadsIO::gameRunning) {
		QTadsIO::enterCommand("restore");
	}
}


void
QTadsMainWindow::gameSave()
{
	if (QTadsIO::gameRunning) {
		QTadsIO::enterCommand("save");
	}
}


void
QTadsMainWindow::gameNew()
{
	if (QTadsIO::gameRunning) {
		bool wasImmediateQuitEnabled = QTadsIO::settings().immediateQuit();
		QTadsIO::settings().immediateQuit(true);
		if (QTadsIO::quit()) {
			QTadsIO::openNewGame = true;
		}
		QTadsIO::settings().immediateQuit(wasImmediateQuitEnabled);
	} else {
		QTadsIO::openNewGame = true;
	}
}


void
QTadsMainWindow::gameRestart()
{
	if (QTadsIO::gameRunning) {
		QTadsIO::restart();
	} else {
		QTadsIO::restartGame = true;
	}
}


/* Implementation of the game information enumerator callback interface.
 * See tads3/gameinfo.h for details.
 */
class QTadsGameInfoEnum: public CTadsGameInfo_enum
{
  public:
	QString gameName, byLine, htmlByLine, email, desc, htmlDesc, version, date, lang, license, copyRules;

	virtual void
	tads_enum_game_info( const char* name, const char* val )
	{
		const QString& tmp = QString::fromAscii(name).lower();
		if (tmp == "name") {
			this->gameName = QString("<b><center><font size=\"+1\">") + QStyleSheet::escape(val) + "</font></center></b>";
		} else if (tmp == "byline") {
			this->byLine = QString("<i><center>") + QStyleSheet::escape(val) + "</center></i>";
		} else if (tmp == "htmlbyline") {
			this->htmlByLine = QString("<i><center>") + val + "</center></i>";
		} else if (tmp == "authoremail") {
			this->email = val;
		} else if (tmp == "desc") {
			this->desc = QString("<p align=\"justify\">") + QStyleSheet::escape(val) + "</p>";
		} else if (tmp == "htmldesc") {
			this->htmlDesc = QString("<p align=\"justify\">") + val + "</p>";
		} else if (tmp == "version") {
			this->version = val;
		} else if (tmp == "releasedate") {
			this->date = val;
		} else if (tmp == "language") {
			this->lang = val;
		} else if (tmp == "licensetype") {
			this->license = val;
		} else if (tmp == "copyingrules") {
			this->copyRules = val;
		}
	}
};

void
QTadsMainWindow::gameInfo()
{
	CTadsGameInfo info;
	if (not info.read_from_file(QTadsIO::currentGame.ascii())) {
		qtadsWarning(tr("Game information resource not found."));
		return;
	}
	QTadsGameInfoEnum cb;
	info.enum_values(&cb);
	GameInfoDialog infoDiag(this);
	// Prefer the non-HTML "ByLine".
	QString tmp(cb.gameName + "<p>" + (cb.byLine.isEmpty() ? cb.htmlByLine : cb.byLine) + "<p>");
	// Use the HTML description if available.
	tmp += cb.htmlDesc.isEmpty() ? cb.desc : cb.htmlDesc;
	// Fill out the dialogs fields.
	QString unknown(QString("[") + tr("Unknown") + "]");
	infoDiag.display->setText(tmp);
	infoDiag.versionDisp->setText(cb.version.isEmpty() ? unknown : cb.version);
	infoDiag.dateDisp->setText(cb.date.isEmpty() ? unknown : cb.date);
	infoDiag.emailDisp->setText(cb.email.isEmpty() ? unknown : cb.email);
	infoDiag.langDisp->setText(cb.lang.isEmpty() ? unknown : cb.lang);
	infoDiag.licenseDisp->setText(cb.license.isEmpty() ? unknown : cb.license);
	infoDiag.copyDisp->setText(cb.copyRules.isEmpty() ? unknown : cb.copyRules);
	infoDiag.exec();
}


void
QTadsMainWindow::gameExit()
{
	if (QTadsIO::gameRunning) {
		bool wasImmediateQuitEnabled = QTadsIO::settings().immediateQuit();
		QTadsIO::settings().immediateQuit(true);
		QTadsIO::quit();
		QTadsIO::settings().immediateQuit(wasImmediateQuitEnabled);
	}
	if (not QTadsIO::gameRunning) {
		QTadsIO::quitApp = true;
	}
}


void
QTadsMainWindow::showHideMenu()
{
	if (this->menuBar()->isVisible()) {
		this->menuBar()->hide();
	} else {
		this->menuBar()->show();
	}
}


/* Updates the filenames that are listed in the "Game->Recent Games"
 * menu.
 */
void
QTadsMainWindow::updateRecentGamesList()
{
	this->fRecentMenu->clear();
	const std::vector<QString>& games = QTadsIO::settings().recentGames();
	for (std::vector<QString>::const_iterator i(games.begin()); i != games.end(); ++i) {
		this->fRecentMenu->insertItem(*i);
	}
}


void QTadsMainWindow::loadRecentGame( int gameId )
{
	if (QTadsIO::gameRunning) {
		bool wasImmediateQuitEnabled = QTadsIO::settings().immediateQuit();
		QTadsIO::settings().immediateQuit(true);
		if (QTadsIO::quit()) {
			QTadsIO::openNewGame = true;
			QTadsIO::newGameToOpen = QTadsIO::settings().recentGames()[this->fRecentMenu->indexOf(gameId)];
		}
		QTadsIO::settings().immediateQuit(wasImmediateQuitEnabled);
	} else {
		QTadsIO::openNewGame = true;
		QTadsIO::newGameToOpen = QTadsIO::settings().recentGames()[this->fRecentMenu->indexOf(gameId)];
	}
}


/* Updates the themes that are listed in the "Display->Theme" menu.
 */
void
QTadsMainWindow::updateThemeList()
{
	// Since this operation is fast, we simply clear everything and
	// then add every theme.
	this->fThemesMenu->clear();
	const std::vector<QString>& themes = QTadsIO::settings().themes();
	for (std::vector<QString>::const_iterator i(themes.begin()); i != themes.end(); ++i) {
		int id;
		id = this->fThemesMenu->insertItem(*i);
		if (*i == QTadsIO::settings().nameOfCurrentTheme()) {
			// It's the current theme.  Check it.
			this->fThemesMenu->setItemChecked(id, true);
		}
	}
}


void
QTadsMainWindow::switchToTheme( int themeId )
{
	// Uncheck the previously selected theme.  We simply do it
	// brute-force and uncheck them all.
	for (uint i = 0; i < this->fThemesMenu->count(); ++i) {
		this->fThemesMenu->setItemChecked(this->fThemesMenu->idAt(i), false);
	}

	// "Check" the new selection.
	this->fThemesMenu->setItemChecked(themeId, true);

	// Activate the new theme and apply it.
	QTadsIO::settings().currentTheme(this->fThemesMenu->text(themeId));
	QTadsIO::applySettings();
}


void
QTadsMainWindow::startTimer()
{
	this->fClockDisplay->start();
}


void
QTadsMainWindow::stopTimer()
{
	this->fClockDisplay->stop();
}


void
QTadsMainWindow::clockDisable()
{
	this->fClockDisplay->hide();

	// Update the menu-entry, if needed.
	if (not this->clockDisableAction->isOn()) {
		this->clockDisableAction->setOn(true);
	}
}


void
QTadsMainWindow::clockEnable()
{
	this->fClockDisplay->displaySeconds(false);
	this->fClockDisplay->show();

	// Update the menu-entry, if needed.
	if (not this->clockEnableAction->isOn()) {
		this->clockEnableAction->setOn(true);
	}
}


void
QTadsMainWindow::clockEnableWithSeconds()
{
	this->fClockDisplay->displaySeconds(true);
	this->fClockDisplay->show();

	// Update the menu-entry, if needed.
	if (not this->clockEnableSecondsAction->isOn()) {
		this->clockEnableSecondsAction->setOn(true);
	}
}


void
QTadsMainWindow::helpAbout()
{
	AboutDialog dialog(this, "about dialog", true);
	dialog.exec();
}


void
QTadsMainWindow::helpShortcuts()
{
	ShortcutsDialog dialog(this, "shortcuts dialog", true);
	dialog.exec();
}


void
QTadsMainWindow::helpVersion()
{
	QMessageBox::information(this, "Version Information",
				 QTADSVERSION
				 "\nTads 2 Virtual Machine: " TADS_RUNTIME_VERSION
				 "\nTads 3 Virtual Machine: " T3VM_VSN_STRING,
				 QMessageBox::Ok);
}


bool
QTadsMainWindow::showDockMenu( const QPoint & /* globalPos */ )
{
	return false;
}


void
QTadsMainWindow::showHideToolbar()
{
	if (this->toolBar->isHidden()) {
		this->toolBar->show();
	} else {
		this->toolBar->hide();
	}
}
