//
// C++ Implementation:
//
// Description:
//
//
// Author: Sebastian Holtermann <sebholt@xwmw.org>, (C) 2011
//
// Copyright: See COPYING file that comes with this distribution
//
//

#include "controls_model.hpp"

#include <qsnd/alsa.hpp>
#include <qsnd/card_info.hpp>
#include <qsnd/control_address.hpp>
#include <QFont>
#include <iostream>


namespace MWdg
{


Controls_Model::Controls_Model ( ) :
_alsa_cfg_model ( 0 )
{
	_ttip_row_tmpl = "<div><b>%1:</b> %2</div>";
}


Controls_Model::~Controls_Model ( )
{
	clear_data();
}


void
Controls_Model::set_alsa_cfg_model (
	QAbstractItemModel * model_n )
{
	reset_begin();

	if ( _alsa_cfg_model != 0 ) {
		disconnect ( _alsa_cfg_model, 0, this, 0 );
	}

	_alsa_cfg_model = model_n;

	if ( _alsa_cfg_model != 0 ) {
		connect ( _alsa_cfg_model, SIGNAL ( modelAboutToBeReset() ),
			this, SLOT ( reset_begin() ) );

		connect ( _alsa_cfg_model, SIGNAL ( modelReset() ),
			this, SLOT ( reset_finish() ) );
	}

	reset_finish();
}


const ::QSnd::Card_Info *
Controls_Model::card_info (
	int index_n ) const
{
	if ( index_n < num_cards() ) {
		return _card_infos[index_n];
	}
	return 0;
}


const ::QSnd::Control_Address *
Controls_Model::ctl_info (
	const QModelIndex & idx_n ) const
{
	const ::QSnd::Control_Address * res ( 0 );

	const QVariant & vnt ( idx_n.data ( Qt::UserRole + 1 ) );
	if ( vnt.canConvert ( QVariant::Int ) ) {
		int idx ( vnt.toInt() );
		if ( ( idx >= 0 ) && ( idx < _ctl_addrs.size() ) ) {
			res = _ctl_addrs[idx];
		}
	}

	return res;
}


QModelIndex
Controls_Model::ctl_info_index (
	const ::QSnd::Control_Address & ctl_addr_n ) const
{
	QModelIndex res;

	for ( int ii=0; ii < _ctl_addrs.size(); ++ii ) {
		if ( ctl_addr_n == *_ctl_addrs[ii] ) {
			res = _ctl_items[ii]->index();
			break;
		}
	}

	return res;
}


void
Controls_Model::revert ( )
{
	reset_begin();
	reset_finish();
}


void
Controls_Model::reset_begin ( )
{
	beginResetModel();

	clear_data();
}


void
Controls_Model::reset_finish ( )
{
	load_ctl_plugins();
	load_cards();

	endResetModel();
}


void
Controls_Model::clear_data ( )
{
	if ( _card_infos.size() > 0 ) {
		for ( int ii=0; ii < _card_infos.size(); ++ii ) {
			delete _card_infos[ii];
		}
		_card_infos.clear();
	}

	if ( _ctl_addrs.size() > 0 ) {
		for ( int ii=0; ii < _ctl_addrs.size(); ++ii ) {
			delete _ctl_addrs[ii];
		}
		_ctl_addrs.clear();
	}


	if ( _ctl_items.size() > 0 ) {
		removeRows ( 0, rowCount() );
		_ctl_items.clear();
	}
}


void
Controls_Model::load_ctl_plugins ( )
{
	if ( _alsa_cfg_model == 0 ) {
		return;
	}

	// Find ctl plugin section in the ALSA configuration tree
	QModelIndex idx_ctl;
	for ( int ii=0; ii < _alsa_cfg_model->rowCount(); ++ii ) {
		QModelIndex idx ( _alsa_cfg_model->index ( ii, 0 ) );
		if ( _alsa_cfg_model->data ( idx ) == "ctl" ) {
			idx_ctl = idx;
			break;
		}
	}

	if ( idx_ctl.isValid() ) {

		// Create control addresses
		for ( int ii=0; ii < _alsa_cfg_model->rowCount ( idx_ctl ); ++ii ) {
			QModelIndex idx ( _alsa_cfg_model->index ( ii, 0, idx_ctl ) );
			if ( _alsa_cfg_model->rowCount ( idx ) > 0 ) {
				const QString ctl_str (
					_alsa_cfg_model->data ( idx ).toString() );

				// Create control address
				{
					::QSnd::Control_Address * qctli (
						new ::QSnd::Control_Address ( ctl_str ) );
					_ctl_addrs.append ( qctli );
				}

				// Create standard item
				{
					QStandardItem * sitem ( new QStandardItem );
					sitem->setText ( ctl_str );
					sitem->setEditable ( false );
					sitem->setSelectable ( true );
					sitem->setData ( ii, Qt::UserRole + 1 );
					_ctl_items.append ( sitem );
				}

			}
		}

		// Append rows and sort
		for ( int ii=0; ii < _ctl_items.size(); ++ii ) {
			appendRow ( _ctl_items[ii] );
		}
		sort ( 0 );
	}

}


void
Controls_Model::load_cards ( )
{
	if ( _alsa_cfg_model == 0 ) {
		return;
	}

	QStandardItem * hw_sitem ( 0 );
	for ( int ii=0; ii < _ctl_addrs.size(); ++ii ) {
		if ( _ctl_addrs[ii]->ctl_name() == "hw" ) {
			hw_sitem = _ctl_items[ii];
			break;
		}
	}
	if ( hw_sitem == 0 ) {
		return;
	}

	// Load card infos
	{
		int card_idx = -1;
		while (	( snd_card_next ( &card_idx ) == 0 ) &&
			( card_idx >= 0 ) )
		{
			::QSnd::Card_Info * cinfo (
				new ::QSnd::Card_Info ( card_idx ) );
			_card_infos.append ( cinfo );
		}
	}

	// remember start position in _ctl_addrs list
	const int cstart_idx ( _ctl_addrs.size() );

	for ( int ii=0; ii < _card_infos.size(); ++ii ) {
		::QSnd::Control_Address * ctl_addr (
			new ::QSnd::Control_Address );

		ctl_addr->set_ctl_name ( "hw" );
		{
			const ::QSnd::Card_Info * cinfo ( _card_infos[ii] );
			::QSnd::Control_Address_Arg argm ( "CARD", "string" );
			argm.arg_value = QString ( "%1" ).arg ( cinfo->card_index() );
			ctl_addr->append_arg ( argm );
		}

		_ctl_addrs.append ( ctl_addr );
	}


	// Create standard items
	for ( int ii=0; ii < _card_infos.size(); ++ii ) {
		const ::QSnd::Card_Info * cinfo ( _card_infos[ii] );

		QStandardItem * sitem ( new QStandardItem );
		sitem->setEditable ( false );
		sitem->setSelectable ( true );
		sitem->setData ( cstart_idx + ii, Qt::UserRole + 1 );

		// Text
		{
			QString txt ( "%1: " );
			txt = txt.arg ( cinfo->card_index() );
			txt.append ( cinfo->card_name() );
			sitem->setText ( txt );
		}

		// Tool tip
		{
			QString ttip;

			{
				QString valc ( _ttip_row_tmpl );
				valc = valc.arg ( tr ( "Name" ) );
				valc = valc.arg ( cinfo->card_name() );
				ttip.append ( valc );
			}

			{
				QString valc ( _ttip_row_tmpl );
				valc = valc.arg ( tr ( "Id" ) );
				valc = valc.arg ( cinfo->card_id() );
				ttip.append ( valc );
			}

			{
				QString valc ( _ttip_row_tmpl );
				valc = valc.arg ( tr ( "Mixer name" ) );
				valc = valc.arg ( cinfo->card_mixer_name() );
				ttip.append ( valc );
			}

			sitem->setToolTip ( ttip );
		}

		_ctl_items.append ( sitem );
	}

	// Append items to actual model
	for ( int ii=0; ii < _card_infos.size(); ++ii ) {
		hw_sitem->appendRow ( _ctl_items[ cstart_idx + ii ] );
	}
}



} // End of namespace

