# Pyntor::TPP - Slides component for TPP plain-text format
# Copyright (C) 2006 Josef Spillner <josef@coolprojects.org>
# Published under GNU GPL conditions

import pygame
from pygame.locals import *

import re

class TPPPage:
	def __init__(self):
		self.name = None
		self.heading = None
		self.text = []

class TPP:
	def init(self, options):
		self.title = None
		self.author = None
		self.date = None
		self.tpppages = []
		self.abstract = []
		tpppage = None

		tppfile = options["tppfile"]

		f = open(tppfile)
		lines = f.readlines()
		f.close()

		pagekeywords = (
			"beginoutput",
			"endoutput",
			"beginshelloutput",
			"endshelloutput",
			"huge",
			"horline",
			"color"
		)

		rkeyword = re.compile("^--(\w+)(?: ((?:\ ?\S+)+))?$")

		for line in lines:
			line = line[:-1]
			mkeyword = rkeyword.match(line)
			if mkeyword:
				keyword = mkeyword.group(1)
				if keyword == "title":
					self.title = mkeyword.group(2)
				elif keyword == "author":
					self.author = mkeyword.group(2)
				elif keyword == "date":
					self.date = mkeyword.group(2)
				elif keyword == "newpage":
					tpppage = TPPPage()
					tpppage.name = mkeyword.group(2)
					self.tpppages.append(tpppage)
				elif keyword == "heading":
					if tpppage:
						tpppage.heading = mkeyword.group(2)
					else:
						print "(TPP) Error: heading outside of page", keyword
				elif keyword in pagekeywords:
					if tpppage:
						tpppage.text.append(line)
					else:
						print "(TPP) Error: page keyword outside of page", keyword
				else:
					print "(TPP) Error: unsupported keyword", keyword
			else:
				if line[:4] == "--##":
					pass
				else:
					if tpppage:
						tpppage.text.append(line)
					else:
						self.abstract.append(line)

		self.pages = len(self.tpppages)

	def render(self, screen, page, globalpage):
		surface = pygame.Surface((screen.get_width(), screen.get_height()))
		#surface.fill((255, 200, 100))
		surface.fill((0, 0, 0))

		tpppage = self.tpppages[page - 1]

		titlefont = pygame.font.SysFont("Vera Sans", 30)
		textfont = pygame.font.SysFont("Vera Sans", 18)
		hugefont = pygame.font.SysFont("Vera Sans", 36)

		f = titlefont.render(tpppage.heading, 1, (0, 0, 0))
		surface.blit(f, ((surface.get_width() - f.get_width()) / 2, 20))

		i = 0
		defaultcolor = (255, 255, 255)
		color = defaultcolor
		for line in tpppage.text:
			rkeyword = re.compile("^--(\w+)(?: ((?:\ ?\S+)+))?$")
			mkeyword = rkeyword.match(line)
			if mkeyword:
				keyword = mkeyword.group(1)
				if keyword == "beginoutput":
					color = (255, 0, 0)
				elif keyword == "endoutput":
					color = defaultcolor
				elif keyword == "beginshelloutput":
					color = (0, 0, 255)
				elif keyword == "endshelloutput":
					color = defaultcolor
				elif keyword == "color":
					colstr = mkeyword.group(2)
					if colstr == "white":
						color = defaultcolor
					elif colstr == "red":
						color = (255, 0, 0)
					elif colstr == "green":
						color = (0, 255, 0)
					elif colstr == "blue":
						color = (0, 0, 255)
					else:
						print "(TPP) Unknown color", color
				elif keyword == "huge":
					hugetext = mkeyword.group(2)
					f = hugefont.render(hugetext, 1, color)
					surface.blit(f, (30, 50 + i * 28))
					i += 2
				elif keyword == "horline":
					surface.fill(defaultcolor, ((0, 50 + i * 28), (screen.get_width(), 2)))
				else:
					print "(TPP) Unhandled page keyword", keyword
			else:
				line = line.replace("\t", " " * 8)
				f = textfont.render(line, 1, color)
				surface.blit(f, (30, 50 + i * 28))
				i += 1

		screen.blit(surface, (0, 0))

	def interactive(self, event):
		if event.type == KEYDOWN:
			key = event.key
			if key == K_ESCAPE or pygame.event.peek(QUIT):
				return "exit"
			if key == K_RETURN:
				return "next"
			if key == K_BACKSPACE:
				return "previous"

component = TPP()

parameters = (
	("tppfile", "TPP file to use for presentation", None),
)

metainfo = {
	"version": "0.2",
	"author": "Josef Spillner",
	"authoremail": "<josef@coolprojects.org>",
	"licence": "GPL"
}

doc = """
TPP: creation of slides from Text Presentation Program (TPP) files

Usage: 'tpp' -tppfile <tppfile>
 Where <tppfile> is a usual .tpp text file

Display: yes

Interactivity: switching pages with enter/backspace
"""

