/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "gui_sql.h"
#include "jobmanager.h"
#include "job_querydatabase.h"
#include "job_storecustomquery.h"
#include "job_removecustomquery.h"
#include "database.h"
#include "gui.h"
#include "pixmapcache.h"

#include <qlayout.h>
#include <qvgroupbox.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qinputdialog.h>

#include <prokyon.h>

// ##############################################
// # initialize search gui
// ##############################################
GUI_SQL::GUI_SQL(QWidget *parent, const char *name)
: QFrame(parent, name) {
    int lineSpace = fontMetrics().lineSpacing();

    QHBoxLayout *searchblay = new QHBoxLayout(this);

    QVGroupBox *searchbox = new QVGroupBox(_("Query"), this);
    searchblay->addWidget(searchbox);

    // text Input
    QFrame *phraseframe = new QFrame(searchbox);
    QGridLayout *vlay = new QGridLayout(phraseframe, 5, 7);
    vlay->addColSpacing(0, lineSpace);
    vlay->addColSpacing(2, lineSpace / 2);
    vlay->addColSpacing(4, lineSpace);
    vlay->addRowSpacing(0, lineSpace);
    vlay->addRowSpacing(3, lineSpace / 2);
    vlay->addRowSpacing(6, lineSpace);
    QLabel *phraselabel = new QLabel("<b>SELECT * FROM tracks WHERE</b> ", phraseframe);
    vlay->addMultiCellWidget(phraselabel, 1, 1, 1, 3);
    phrase = new QLineEdit("", phraseframe);
    vlay->addMultiCellWidget(phrase, 2, 2, 1, 3);
    QLabel *customQueriesLabel = new QLabel(_("Stored Queries"), phraseframe);
    vlay->addMultiCellWidget(customQueriesLabel, 4, 4, 1, 3);
    customQueries = new QComboBox(false, phraseframe);
    customQueries->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    customQueries->setDuplicatesEnabled(false);
    vlay->addWidget(customQueries, 5, 1);
    button_remove = new QPushButton(pixmapcache->getFadeIconSet("action_remove.png"), _("Remove"), phraseframe);
    vlay->addWidget(button_remove, 5, 3);

    // the buttons
    QFrame *buttonframe = new QFrame(this);
    searchblay->addWidget(buttonframe);
    QGridLayout *buttongrid = new QGridLayout(buttonframe, 2, 3);
    buttongrid->addColSpacing(0, 10);
    buttongrid->addColSpacing(3, 10);
    button_search = new QPushButton(pixmapcache->getFadeIconSet("action_updateonly.png"), _("Search"), buttonframe);
    buttongrid->addWidget(button_search, 0, 1);
    button_reset = new QPushButton(pixmapcache->getFadeIconSet("action_discard.png"), _("Reset"), buttonframe);
    buttongrid->addWidget(button_reset, 1, 1);
    button_save = new QPushButton(pixmapcache->getFadeIconSet("filesave.png"), _("Save"), buttonframe);
    buttongrid->addWidget(button_save, 2, 1);

    connect(phrase, SIGNAL(returnPressed()), this, SLOT(slot_buttonSearch()));
    connect(button_search, SIGNAL(clicked()), this, SLOT(slot_buttonSearch()));
    connect(button_reset, SIGNAL(clicked()), this, SLOT(slot_buttonReset()));
    connect(button_save, SIGNAL(clicked()), this, SLOT(slot_buttonSave()));
    connect(button_remove, SIGNAL(clicked()), this, SLOT(slot_buttonRemove()));
    connect(customQueries, SIGNAL(activated(int)), this, SLOT(slot_runCustomQuery(int)));
}

void GUI_SQL::notifyNewCustomQueries(QMap<QString,QString> queries) {
    if(!queries.empty()) {
        customQueries->insertStringList(queries.keys());
    }
    customQueriesMap = queries;
}

// ##############################################
// # callbacks for the buttons
// ##############################################
// search
void GUI_SQL::slot_buttonSearch() {
    if(phrase->text().isEmpty())
        return;
    jobman->addJob(new Job_QueryDatabase(QUERY_BY_WHERE, phrase->text(), ""));
}

// reset phrase
void GUI_SQL::slot_buttonReset() {
    phrase->clear();
}

// save
void GUI_SQL::slot_buttonSave() {
    if(phrase->text().isEmpty())
        return;

    bool ok = false, done;
    QString name;
    do {
        name = QInputDialog::getText(_("Save Query"), QString(_("Enter name of query to save")) + ":", QLineEdit::Normal, name, &ok);
        if(ok) {
            done = (!name.isEmpty() && !existsStoredQuery(name));
            if(!done) {
                QMessageBox::information(gui, _("Message"),
                                         _("Name is empty\nOR\nalready exists in list of stored queries!"),
                                         QMessageBox::Ok);
            }
        } else
            done = true;
    } while(!done);

    if(ok) {
        customQueries->insertItem(name);
        customQueriesMap[name] = phrase->text();
        jobman->addJob(new Job_StoreCustomQuery(name, phrase->text()));
    }
}

// save
void GUI_SQL::slot_buttonRemove() {
    QString name(customQueries->currentText());
    if(name.isEmpty())
        return;

    int result = QMessageBox::warning(gui,
                                      _("Remove Custom Query"),
                                      /* TRANSLATORS: %1 = name of custom query */
                                      QString(_("OK to remove custom query '%1' from database?")).arg(name),
                                      QMessageBox::Yes, QMessageBox::No);

    if (result != QMessageBox::Yes)
        return;

    customQueries->removeItem(customQueries->currentItem());
    customQueriesMap.remove(name);
    jobman->addJob(new Job_RemoveCustomQuery(name));
}

void GUI_SQL::slot_runCustomQuery(int pos) {
    phrase->setText(customQueriesMap[customQueries->text(pos)]);
    slot_buttonSearch();
}

void GUI_SQL::slot_setConnectionState(bool state) {
    button_search->setEnabled(state);
    button_save->setEnabled(state);
    button_reset->setEnabled(state);
    button_remove->setEnabled(state);
    customQueries->clear();
}

bool GUI_SQL::existsStoredQuery(QString name) {
    for(int i = customQueries->count() - 1; i >= 0; --i)
        if(customQueries->text(i) == name)
            return true;

    return false;
}

GUI_SQL::~GUI_SQL() {}
