/*
 * ProFTPD: mod_case -- provides case-insensivity
 *
 * Copyright (c) 2004-2010 TJ Saunders
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307, USA.
 *
 * This is mod_case, contrib software for proftpd 1.2 and above.
 * For more information contact TJ Saunders <tj@castaglia.org>.
 *
 * $Id: mod_case.c,v 1.5 2010/04/07 16:33:20 tj Exp tj $
 *
 */

#include "conf.h"
#include "privs.h"

#define MOD_CASE_VERSION	"mod_case/0.4"

/* Make sure the version of proftpd is as necessary. */
#if PROFTPD_VERSION_NUMBER < 0x0001030201
# error "ProFTPD 1.3.2rc1 or later required"
#endif

static int case_engine = FALSE;
static int case_logfd = -1;

/* Support routines
 */

static int case_expr_eval_cmds(cmd_rec *cmd, array_header *list) {
  int found;
  register unsigned int i;

  for (i = 0; i < list->nelts; i++) { 
    char *c = ((char **) list->elts)[i];
    found = 0;

    if (*c == '!') {
      found = !found;
      c++;
    }

    if (strcmp(cmd->argv[0], c) == 0)
      found = !found;

    if (found)
      return 1;
  }

  return 0;
}

static void case_replace_path(cmd_rec *cmd, const char *dir, const char *file) {
  cmd->argv[1] = pstrcat(cmd->pool, dir, file, NULL);

  /* In the case of many commands, we also need to overwrite cmd->arg.
   */
  if (strcmp(cmd->argv[0], C_APPE) == 0 ||
      strcmp(cmd->argv[0], C_CWD) == 0 ||
      strcmp(cmd->argv[0], C_DELE) == 0 ||
      strcmp(cmd->argv[0], C_MKD) == 0 ||
      strcmp(cmd->argv[0], C_MDTM) == 0 ||
      strcmp(cmd->argv[0], C_RETR) == 0 ||
      strcmp(cmd->argv[0], C_RMD) == 0 ||
      strcmp(cmd->argv[0], C_RNFR) == 0 ||
      strcmp(cmd->argv[0], C_SIZE) == 0 ||
      strcmp(cmd->argv[0], C_STAT) == 0 ||
      strcmp(cmd->argv[0], C_STOR) == 0 ||
      strcmp(cmd->argv[0], C_XCWD) == 0 ||
      strcmp(cmd->argv[0], C_XMKD) == 0 ||
      strcmp(cmd->argv[0], C_XRMD) == 0)
    cmd->arg = pstrcat(cmd->pool, dir, file, NULL);
}

/* Command handlers
 */

MODRET case_pre_cmd(cmd_rec *cmd) {
  config_rec *c;
  char *path, *dir, *file, *tmp;
  DIR *dirh;
  struct dirent *dent;

  if (!case_engine)
    return PR_DECLINED(cmd);

  c = find_config(CURRENT_CONF, CONF_PARAM, "CaseIgnore", FALSE);
  if (c == NULL)
    return PR_DECLINED(cmd);

  if (*((unsigned int *) c->argv[0]) != TRUE)
    return PR_DECLINED(cmd);

  if (c->argv[1] &&
      case_expr_eval_cmds(cmd, *((array_header **) c->argv[1])) == 0)
    return PR_DECLINED(cmd);

  path = pstrdup(cmd->tmp_pool, cmd->argv[1]);

  /* Separate the path into directory and file components. */
  tmp = strrchr(path, '/');
  if (tmp == NULL) {
    dir = ".";
    file = path;

  } else {
    if (tmp != path) {
      *tmp++ = '\0';
      dir = path;
      file = tmp;

    } else {
      /* Handle the case where the path is "/path". */
      dir = "/";
      file = tmp + 1;
    }
  }
  
  /* Open the directory. */
  dirh = pr_fsio_opendir(dir);
  if (dirh == NULL) {
    (void) pr_log_writefile(case_logfd, MOD_CASE_VERSION,
      "error opening directory '%s': %s", dir, strerror(errno));
    return PR_DECLINED(cmd);
  }

  /* Two-pass check.  First, check to see if any files in the directory
   * exactly match the given file.  If so, we do nothing.  Otherwise,
   * scan the directory again, treating the given file as a glob.
   */

  /* For each file in the directory, check it against the given name, both
   * as an exact match and as a possible glob match.
   */
  dent = pr_fsio_readdir(dirh);
  while (dent) {
    if (strcmp(dent->d_name, file) == 0) {
      (void) pr_log_writefile(case_logfd, MOD_CASE_VERSION,
        "found exact match");
      pr_fsio_closedir(dirh);
      return PR_DECLINED(cmd);
    }

    dent = pr_fsio_readdir(dirh);
  }

  /* Rewind the directory to the beginning. */
  /* pr_fsio_rewinddir(dirh); */
  rewinddir((DIR *) dirh);

  /* Escape any existing glob characters in the file name. */
  if (strchr(file, '?'))
    file = sreplace(cmd->tmp_pool, file, "?", "\\?", NULL);

  if (strchr(file, '*'))
    file = sreplace(cmd->tmp_pool, file, "*", "\\*", NULL);

  if (strchr(file, '['))
    file = sreplace(cmd->tmp_pool, file, "[", "\\[", NULL);

  dent = pr_fsio_readdir(dirh);
  while (dent) {
    if (pr_fnmatch(file, dent->d_name, PR_FNM_CASEFOLD) == 0) {
      (void) pr_log_writefile(case_logfd, MOD_CASE_VERSION,
        "found case-insensitive match '%s' for '%s'", dent->d_name, file);

      /* Overwrite the client-given path. */
      case_replace_path(cmd, tmp ? pstrcat(cmd->pool, dir, "/", NULL) : "",
        dent->d_name);

      pr_fsio_closedir(dirh);
      return PR_DECLINED(cmd);
    }

    dent = pr_fsio_readdir(dirh);
  }

  /* Close the directory. */
  pr_fsio_closedir(dirh);

  return PR_DECLINED(cmd);
}

/* Configuration handlers
 */

/* usage: CaseEngine on|off */
MODRET set_caseengine(cmd_rec *cmd) {
  int bool;
  config_rec *c;

  CHECK_CONF(cmd, CONF_ROOT|CONF_VIRTUAL|CONF_GLOBAL);
  CHECK_ARGS(cmd, 1);

  bool = get_boolean(cmd, 1);
  if (bool == -1)
    CONF_ERROR(cmd, "expected Boolean parameter");

  c = add_config_param(cmd->argv[0], 1, NULL);
  c->argv[0] = pcalloc(c->pool, sizeof(unsigned int));
  *((unsigned int *) c->argv[0]) = bool;

  return PR_HANDLED(cmd);
}

/* usage: CaseIgnore on|off|cmd-list */
MODRET set_caseignore(cmd_rec *cmd) {
  int bool, argc;
  char **argv;
  config_rec *c;

  CHECK_CONF(cmd, CONF_ROOT|CONF_VIRTUAL|CONF_GLOBAL|CONF_ANON|CONF_DIR);
  CHECK_ARGS(cmd, 1);

  bool = get_boolean(cmd, 1);

  c = add_config_param(cmd->argv[0], 2, NULL, NULL);
  c->flags |= CF_MERGEDOWN_MULTI;

  c->argv[0] = pcalloc(c->pool, sizeof(unsigned int));
  *((unsigned int *) c->argv[0]) = 1;

  if (bool != -1) {
    *((unsigned int *) c->argv[0]) = bool;
    return PR_HANDLED(cmd);
  }

  /* Parse the parameter as a command list. */
  argc = cmd->argc-1;
  argv = cmd->argv;

  c->argv[1] = pcalloc(c->pool, sizeof(array_header *));
  *((array_header **) c->argv[1]) = pr_expr_create(c->pool, &argc, argv);

  return PR_HANDLED(cmd);
}

/* usage: CaseLog path|"none" */
MODRET set_caselog(cmd_rec *cmd) {
  CHECK_CONF(cmd, CONF_ROOT|CONF_VIRTUAL|CONF_GLOBAL);
  CHECK_ARGS(cmd, 1);

  if (pr_fs_valid_path(cmd->argv[1]) < 0)
    CONF_ERROR(cmd, "must be an absolute path");

  add_config_param_str(cmd->argv[0], 1, cmd->argv[1]);

  return PR_HANDLED(cmd);
}

/* Initialization functions
 */

static int case_sess_init(void) {
  config_rec *c;
  int res = 0;

  c = find_config(main_server->conf, CONF_PARAM, "CaseEngine", FALSE);
  if (c && *((unsigned int *) c->argv[0]) == TRUE)
    case_engine = TRUE;

  else
    return 0;

  c = find_config(main_server->conf, CONF_PARAM, "CaseLog", FALSE);
  if (c == NULL)
    return 0;

  if (strcasecmp((char *) c->argv[0], "none") == 0)
    return 0;

  pr_signals_block();
  PRIVS_ROOT
  res = pr_log_openfile((char *) c->argv[0], &case_logfd, 0660);
  PRIVS_RELINQUISH
  pr_signals_unblock();

  if (res < 0) {
    pr_log_pri(PR_LOG_NOTICE, MOD_CASE_VERSION
      ": error opening CaseLog '%s': %s", (char *) c->argv[0],
      strerror(errno)); 
  }

  return 0;
}

/* Module API tables
 */

static conftable case_conftab[] = {
  { "CaseEngine",	set_caseengine,		NULL },
  { "CaseIgnore",	set_caseignore,		NULL },
  { "CaseLog",		set_caselog,		NULL },
  { NULL }
};

static cmdtable case_cmdtab[] = {
  { PRE_CMD,	C_APPE,	G_NONE,	case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_CWD,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_DELE,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_MDTM,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_MKD,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_RETR,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_RMD,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_RNFR,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_RNTO,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_SIZE,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_STOR,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_XCWD,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_XMKD,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { PRE_CMD,	C_XRMD,	G_NONE, case_pre_cmd,	TRUE,	FALSE },
  { 0, NULL }
};

module case_module = {
  NULL, NULL,

  /* Module API version 2.0 */
  0x20,

  /* Module name */
  "case",

  /* Module configuration handler table */
  case_conftab,

  /* Module command handler table */
  case_cmdtab,

  /* Module authentication handler table */
  NULL,

  /* Module initialization function */
  NULL,

  /* Session initialization function */
  case_sess_init
};
