/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <ao/ao.h>
#include <glib.h>
#include "buffer.h"
#include "format.h"
#include "output.h"
#include "output-ao.h"

typedef struct output_ao {
	ao_device *dev;
	sample_format old_fmt;
} output_ao;

static void close_ao(output_ao *a) {
	if (a->dev != NULL && ao_close(a->dev) != 1)
		g_error("ao close failed");
	a->dev = NULL;
}

static int play(struct output *o, buffer *buf, sample_format *fmt) {
	output_ao *a = (output_ao *) o->data;

	if (a->dev == NULL || !format_same(&a->old_fmt, fmt)) {
		ao_sample_format ao_fmt;
		ao_fmt.bits = bytes_per_sample(fmt->bits) * 8;
		ao_fmt.rate = fmt->rate;
		ao_fmt.channels = fmt->channels;
		switch (fmt->byte_format) {
		case END_LITTLE:
			ao_fmt.byte_format = AO_FMT_LITTLE;
			break;
		case END_BIG:
			ao_fmt.byte_format = AO_FMT_BIG;
			break;
		case END_NATIVE:
			ao_fmt.byte_format = AO_FMT_NATIVE;
			break;
		}

		close_ao(a);
		a->dev = ao_open_live(ao_default_driver_id(), &ao_fmt, NULL);
		if (a->dev == NULL)
			return -1;

		copy_format(&a->old_fmt, fmt);
	}

	if (ao_play(a->dev, (char *) buf->data, buf->used) < 0)
		return -1;
	buf->used = 0;

	return 0;
}

static void release(struct output *o) {
	output_ao *a = (output_ao *) o->data;

	close_ao(a);
}

static void close(struct output *o) {
	output_ao *a = (output_ao *) o->data;

	close_ao(a);

	free(a);
	free(o);
}

output *output_new_ao(sample_format **force_fmt) {
	output *o = output_alloc();

	output_ao *a = malloc(sizeof *a);
	if (a == NULL)
		g_error("out of memory");
	a->dev = NULL;
	o->data = a;

	o->play = play;
	o->release = release;
	o->close = close;

	static int ao_initialised = 0;
	if (!ao_initialised) {
		ao_initialize();
		ao_initialised = 1;
	}

	*force_fmt = NULL;

	return o;
}
