/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include "main.h"
#define WANT_PROCESS_LABELS
#include "process.h"
#include "buffer.h"
#include "format.h"
#include "player.h"
#include "stock.h"
#include "dnd.h"
#include "util.h"

enum {
	COLUMN_PLAYING,
	COLUMN_NAME,
	COLUMN_PATH,
	N_COLUMNS
};

typedef enum {
	STATE_STOPPED,
	STATE_PLAYING,
	STATE_PAUSED,
} playerstate;

GladeXML *glade;
GtkWidget *mainwindow, *position, *status, *list, *listscroll, *processing;
GtkWidget *previous, *playpause, *next, *stop, *clear, *shuffle;
GtkListStore *model;
GRand *grand;
playerstate state = STATE_STOPPED;
gint current_row = -1;
gulong seek_conn;
static gboolean seek_in_progress = FALSE, seek_grabbed = FALSE;
player_status pstatus;

static gboolean make_iter(gint row, GtkTreeIter *iter) {
	return gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(model),
	                                     iter, NULL, row);
}

static void set_image(const gchar *name, const gchar *stock_id) {
	GtkImage *image = GTK_IMAGE(glade_xml_get_widget(glade, name));
	gtk_image_set_from_stock(image, stock_id, GTK_ICON_SIZE_LARGE_TOOLBAR);
}

static gchar *format_position(GtkScale *scale, gdouble value) {
	gint ivalue = value;
	gint minutes = ivalue / 60;
	gint seconds = ivalue % 60;
	return g_strdup_printf("%d:%02d", minutes, seconds);
}

static void set_length(gdouble length) {
	if (length <= 0.0)
		length = 1.0;
	gtk_range_set_range(GTK_RANGE(position), 0.0, length);
}

static void set_can_seek(gboolean can_seek) {
	gtk_widget_set_sensitive(GTK_WIDGET(position), can_seek);
}

static void set_position(gdouble pos) {
	if (pos <= 0.0)
		pos = 0.0;
	g_signal_handler_block(G_OBJECT(position), seek_conn);
	gtk_range_set_value(GTK_RANGE(position), pos);
	g_signal_handler_unblock(G_OBJECT(position), seek_conn);
}

void stop_playing() {
	if (state == STATE_STOPPED)
		return;

	player_stop();
	state = STATE_STOPPED;

	GtkTreeIter current;
	make_iter(current_row, &current);
	gtk_list_store_set(model, &current, COLUMN_PLAYING, FALSE, -1);

	set_length(0.0);
	set_can_seek(FALSE);
	set_position(0.0);
}

void start_playing() {
	GtkTreeIter current;
	make_iter(current_row, &current);

	gchar *path;
	gtk_tree_model_get(GTK_TREE_MODEL(model), &current,
	                   COLUMN_PATH, &path, -1);

	if (player_start(path)) {
		state = STATE_PLAYING;

		gtk_list_store_set(model, &current, COLUMN_PLAYING, TRUE, -1);
	} else {
		// FIXME display error
	}
}

void pause_playing() {
	player_pause();
	state = STATE_PAUSED;
}

void unpause_playing() {
	player_unpause();
	state = STATE_PLAYING;
}

void update_status() {
	static GString *s = NULL;
	static float history[10] = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
	static int history_pos = 0;

	if (s == NULL) {
		s = g_string_new("");
	} else {
		g_string_truncate(s, 0);
	}

	const gchar *songname = "No file";
	if (state != STATE_STOPPED) {
		GtkTreeIter current;
		make_iter(current_row, &current);

		gtk_tree_model_get(GTK_TREE_MODEL(model), &current,
		                   COLUMN_NAME, &songname, -1);
	}
	gchar *escaped = escape_for_markup(songname);
	g_string_append_printf(s, "<b>%s</b>\n", escaped);
	g_free(escaped);

	switch (state) {
	case STATE_STOPPED:
		g_string_append(s, "Stopped");
		break;
	case STATE_PAUSED:
		g_string_append(s, "Paused");
		break;
	case STATE_PLAYING:
		if (pstatus.bitrate >= 0.0) {
			history[history_pos] = pstatus.bitrate;
			history_pos = (history_pos + 1) % 10;

			float bitrate = 0.0;
			for (int i = 0; i < 10; i++)
				bitrate += history[i];
			g_string_append_printf(s, "%.0fkbps ", bitrate / 10);
		}

		g_string_append_printf(s, "%dch %dHz %dbit",
		                       pstatus.in_fmt.channels, pstatus.in_fmt.rate, pstatus.in_fmt.bits);
		if (pstatus.converting) {
			g_string_append_printf(s, " &#x2192; %dch %dHz %dbit",
					       pstatus.out_fmt.channels, pstatus.out_fmt.rate, pstatus.out_fmt.bits);
		}
		break;
	}

	gtk_label_set_markup(GTK_LABEL(status), s->str);
}

void update_ui() {
	update_status();

	if (state == STATE_PLAYING)
		set_image("playpauseimage", POTAMUS_STOCK_PAUSE);
	else
		set_image("playpauseimage", POTAMUS_STOCK_PLAY);

	GtkTreeIter it;
	gboolean has_items =
	    gtk_tree_model_get_iter_first(GTK_TREE_MODEL(model), &it);
	gtk_widget_set_sensitive(GTK_WIDGET(playpause), has_items);
	gtk_widget_set_sensitive(GTK_WIDGET(clear), has_items);
	gtk_widget_set_sensitive(GTK_WIDGET(shuffle), has_items);

	gboolean can_stop = (state != STATE_STOPPED);
	gtk_widget_set_sensitive(GTK_WIDGET(stop), can_stop);

	gboolean can_previous = FALSE, can_next = FALSE;
	if (state != STATE_STOPPED) {
		GtkTreeIter current;
		make_iter(current_row, &current);

		GtkTreePath *path =
		    gtk_tree_model_get_path(GTK_TREE_MODEL(model), &current);
		gint row = gtk_tree_path_get_indices(path)[0];
		gtk_tree_path_free(path);
		if (row > 0)
			can_previous = TRUE;

		if (gtk_tree_model_iter_next(GTK_TREE_MODEL(model), &current))
			can_next = TRUE;
	} else if (state == STATE_STOPPED && has_items) {
		can_next = TRUE;
	}
	gtk_widget_set_sensitive(GTK_WIDGET(previous), can_previous);
	gtk_widget_set_sensitive(GTK_WIDGET(next), can_next);
}

gboolean handle_quit(GtkWidget *w, GdkEvent *e, gpointer d) {
	stop_playing();
	gtk_main_quit();
	return TRUE;
}

void move_direction(gint dir) {
	gint new_row;

	if (state == STATE_STOPPED) {
		new_row = 0;
	} else {
		stop_playing();
		new_row = current_row + dir;
	}

	GtkTreeIter current;
	GtkTreePath *path = NULL;
	if ((new_row >= 0)
	    && gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(model), &current,
	                                     NULL, new_row)) {
		path = gtk_tree_model_get_path(GTK_TREE_MODEL(model), &current);
		current_row = new_row;
		start_playing();
	} else {
		player_release();
	}

	update_ui();

	if (path != NULL) {
		// We have to do this after update_ui(), because expanding the
		// status string can make the list smaller.
		ensure_path_visible(GTK_TREE_VIEW(list), path);
		gtk_tree_path_free(path);
	}
}

void handle_previous(GtkButton *w, gpointer d) {
	move_direction(-1);
}

void handle_next(GtkButton *w, gpointer d) {
	move_direction(1);
}

void handle_stop(GtkButton *w, gpointer d) {
	stop_playing();
	player_release();
	update_ui();
}

void handle_playpause(GtkButton *w, gpointer d) {
	switch (state) {
	case STATE_STOPPED: {
		GtkTreeIter current;
		if (gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(model),
		                                  &current, NULL, 0)) {
			current_row = 0;
			start_playing();
		}
		break;
	}
	case STATE_PLAYING:
		pause_playing();
		break;
	case STATE_PAUSED:
		unpause_playing();
		break;
	}
	update_ui();
}

void handle_clear(GtkButton *w, gpointer d) {
	stop_playing();
	gtk_list_store_clear(model);
	update_ui();
}

void handle_row_deleted(GtkTreeModel *model, GtkTreePath *path,
                        gpointer d) {
	gint row = gtk_tree_path_get_indices(path)[0];

	if (row < current_row)
		current_row--;
}

void handle_row_inserted(GtkTreeModel *model, GtkTreePath *path,
                         GtkTreeIter *iter, gpointer d) {
	gint row = gtk_tree_path_get_indices(path)[0];

	if (row < current_row)
		current_row++;
}

void handle_row_changed(GtkTreeModel *model, GtkTreePath *path,
                        GtkTreeIter *iter, gpointer d) {
	gint row = gtk_tree_path_get_indices(path)[0];

	// If the row is being marked as playing, either:
	// - we've just done it;
	// - or it's just been dropped in a new location.
	// Either way, it's definitely the current row.

	gint playing;
	gtk_tree_model_get(GTK_TREE_MODEL(model), iter,
	                   COLUMN_PLAYING, &playing, -1);

	if (playing)
		current_row = row;
}

void handle_row_activated(GtkTreeView *w, GtkTreePath *path,
                          GtkTreeViewColumn *col, gpointer d) {
	stop_playing();

	current_row = gtk_tree_path_get_indices(path)[0];

	GtkTreeIter current;
	if (gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &current, path))
		start_playing();

	gtk_tree_selection_unselect_all(gtk_tree_view_get_selection(GTK_TREE_VIEW(list)));

	update_ui();
}

// FIXME This is far too slow.
// It may work better to use GtkTreeModelSort and a dummy column for shuffling;
// I could then make the shuffle button a toggle that would return the list to
// its original order when deselected.
void handle_shuffle(GtkButton *w, gpointer d) {
	gint num_rows = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(model),
	                                               NULL);
	for (gint i = 0; i < num_rows; i++) {
		GtkTreeIter from, to;
		make_iter(i, &from);
		make_iter(g_rand_int_range(grand, i, num_rows), &to);
		gtk_list_store_swap(GTK_LIST_STORE(model), &from, &to);
	}
}

void handle_processing(GtkComboBox *combo, gpointer d) {
	pstatus.processing = gtk_combo_box_get_active(combo);
}

void handle_seek(GtkRange *range, gpointer d) {
	gdouble pos = gtk_range_get_value(range);

	seek_in_progress = TRUE;
	switch (state) {
	case STATE_STOPPED:
		break;
	case STATE_PAUSED:
		unpause_playing();
		/* fall through */
	case STATE_PLAYING:
		player_seek(pos);
		break;
	}
	seek_in_progress = FALSE;

	update_ui();
}

gboolean handle_seek_grab(GtkWidget *w, GdkEventButton *e, gpointer d) {
	seek_grabbed = TRUE;
	return FALSE;
}

gboolean handle_seek_ungrab(GtkWidget *w, GdkEventButton *e, gpointer d) {
	seek_grabbed = FALSE;
	return FALSE;
}

void handle_player_finished() {
	if (pstatus.error == NULL) {
		move_direction(1);
	} else {
		stop_playing();
		update_ui();

		GtkWidget *dialog;
		dialog = gtk_message_dialog_new(GTK_WINDOW(mainwindow),
		                                GTK_DIALOG_DESTROY_WITH_PARENT,
		                                GTK_MESSAGE_ERROR,
		                                GTK_BUTTONS_CLOSE,
		                                "%s", pstatus.error);
		gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);

		status_set_error(&pstatus, NULL);
	}
}

gboolean handle_timeout(gpointer data) {
	set_can_seek(pstatus.seekable);
	set_length(MAX(pstatus.len, pstatus.pos));
	if (!(seek_in_progress || seek_grabbed)) {
		set_position(pstatus.pos);
	}

	update_status();

	return TRUE;
}

static void add_file(const gchar *path) {
	gchar *name = g_path_get_basename(path);

	GtkTreeIter iter;
	gtk_list_store_append(model, &iter);
	gtk_list_store_set(model, &iter,
	                   COLUMN_PLAYING, FALSE,
	                   COLUMN_PATH, path,
	                   COLUMN_NAME, name,
	                   -1);

	g_free(name);
}

static void build_dir_list(const gchar *path, GPtrArray *files) {
	if (g_file_test(path, G_FILE_TEST_IS_DIR)) {
		GDir *dir = g_dir_open(path, 0, NULL);
		if (dir == NULL)
			return;
	
		while (1) {
			const gchar *entry = g_dir_read_name(dir);
			if (entry == NULL)
				break;
	
			gchar *full = g_build_filename(path, entry, NULL);
			build_dir_list(full, files);
			g_free(full);
		}
	
		g_dir_close(dir);
	} else {
		g_ptr_array_add(files, g_strdup(path));
	}
}

static gint compare(gconstpointer a, gconstpointer b) {
	return (gint) strcmp(*(const char **) a, *(const char **) b);
}

void add_recursively(const gchar *path) {
	GPtrArray *files = g_ptr_array_new();

	build_dir_list(path, files);
	g_ptr_array_sort(files, compare);

	for (int i = 0; i < files->len; i++) {
		add_file((const gchar *) g_ptr_array_index(files, i));
	}

	g_ptr_array_free(files, TRUE);
	update_ui();
}

void playing_datafunc(GtkTreeViewColumn *col, GtkCellRenderer *cell,
                      GtkTreeModel *mod, GtkTreeIter *it, gpointer data) {
	gboolean playing;
	gtk_tree_model_get(mod, it, COLUMN_PLAYING, &playing, -1);

	g_object_set(G_OBJECT(cell), "stock-id",
	             playing ? POTAMUS_STOCK_PLAYING : "", NULL);
}

int main(int argc, char **argv) {
	gtk_init(&argc, &argv);
	grand = g_rand_new();
	player_init(&argc, &argv, &pstatus);
	stock_init();

	if (!gtk_window_set_default_icon_from_file(DATADIR "/potamus.png",
	                                           NULL))
		g_error("Failed to set window icon");

	glade = glade_xml_new(DATADIR "/potamus.glade", NULL, NULL);
	if (glade == NULL)
		g_error("Failed to construct interface");

	mainwindow = glade_xml_get_widget(glade, "mainwindow");
	position = glade_xml_get_widget(glade, "position");
	status = glade_xml_get_widget(glade, "status");
	list = glade_xml_get_widget(glade, "list");
	listscroll = glade_xml_get_widget(glade, "listscroll");

	set_image("previousimage", POTAMUS_STOCK_PREVIOUS);
	set_image("nextimage", POTAMUS_STOCK_NEXT);
	set_image("stopimage", POTAMUS_STOCK_STOP);
	set_image("clearimage", POTAMUS_STOCK_CLEAR);
	set_image("shuffleimage", POTAMUS_STOCK_SHUFFLE);

	g_signal_connect(G_OBJECT(mainwindow), "delete_event",
	                 G_CALLBACK(handle_quit), NULL);

	dnd_init(mainwindow);

	previous = glade_xml_get_widget(glade, "previous");
	g_signal_connect(G_OBJECT(previous), "clicked",
	                 G_CALLBACK(handle_previous), NULL);
	playpause = glade_xml_get_widget(glade, "playpause");
	g_signal_connect(G_OBJECT(playpause), "clicked",
	                 G_CALLBACK(handle_playpause), NULL);
	next = glade_xml_get_widget(glade, "next");
	g_signal_connect(G_OBJECT(next), "clicked",
	                 G_CALLBACK(handle_next), NULL);
	stop = glade_xml_get_widget(glade, "stop");
	g_signal_connect(G_OBJECT(stop), "clicked",
	                 G_CALLBACK(handle_stop), NULL);
	clear = glade_xml_get_widget(glade, "clear");
	g_signal_connect(G_OBJECT(clear), "clicked",
	                 G_CALLBACK(handle_clear), NULL);
	shuffle = glade_xml_get_widget(glade, "shuffle");
	g_signal_connect(G_OBJECT(shuffle), "clicked",
	                 G_CALLBACK(handle_shuffle), NULL);

	processing = glade_xml_get_widget(glade, "processing");
	g_signal_connect(G_OBJECT(processing), "changed",
	                 G_CALLBACK(handle_processing), NULL);
	for (const char **p = process_labels; *p != NULL; p++) {
		gtk_combo_box_append_text(GTK_COMBO_BOX(processing), *p);
	}
	pstatus.processing = 0;
	gtk_combo_box_set_active(GTK_COMBO_BOX(processing), pstatus.processing);

	g_signal_connect(G_OBJECT(position), "format-value",
	                 G_CALLBACK(format_position), NULL);
	gtk_range_set_update_policy(GTK_RANGE(position),
	                            GTK_UPDATE_DISCONTINUOUS);
	seek_conn = g_signal_connect(G_OBJECT(position), "value-changed",
	                             G_CALLBACK(handle_seek), NULL);
	g_signal_connect(G_OBJECT(position), "button-press-event",
	                 G_CALLBACK(handle_seek_grab), NULL);
	g_signal_connect(G_OBJECT(position), "button-release-event",
	                 G_CALLBACK(handle_seek_ungrab), NULL);
	set_length(0.0);
	set_can_seek(FALSE);
	set_position(0.0);

	GtkAdjustment *adjustment =
	    gtk_range_get_adjustment(GTK_RANGE(listscroll));
	gtk_tree_view_set_vadjustment(GTK_TREE_VIEW(list), adjustment);

	model = gtk_list_store_new(N_COLUMNS,
	                           G_TYPE_BOOLEAN,
	                           G_TYPE_STRING,
	                           G_TYPE_STRING);
	gtk_tree_view_set_reorderable(GTK_TREE_VIEW(list), TRUE);
	gtk_tree_view_set_model(GTK_TREE_VIEW(list), GTK_TREE_MODEL(model));

	g_signal_connect(G_OBJECT(model), "row-deleted",
	                 G_CALLBACK(handle_row_deleted), NULL);
	g_signal_connect(G_OBJECT(model), "row-inserted",
	                 G_CALLBACK(handle_row_inserted), NULL);
	g_signal_connect(G_OBJECT(model), "row-changed",
	                 G_CALLBACK(handle_row_changed), NULL);

	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	renderer = gtk_cell_renderer_pixbuf_new();
	column =
	    gtk_tree_view_column_new_with_attributes("Playing", renderer, NULL);
	gtk_tree_view_column_set_cell_data_func(column, renderer,
	                                        playing_datafunc, NULL, NULL);
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width(column, 16);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);

	renderer = gtk_cell_renderer_text_new();
	GValue weight;
	memset(&weight, 0, sizeof weight);
	g_value_init(&weight, G_TYPE_INT);
	g_value_set_int(&weight, 800);
	g_object_set_property(G_OBJECT(renderer), "weight", &weight);
	column =
	    gtk_tree_view_column_new_with_attributes("Name", renderer,
	                                             "text", COLUMN_NAME,
	                                             "weight-set", COLUMN_PLAYING,
	                                             NULL);
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);

	g_signal_connect(G_OBJECT(list), "row-activated",
	                 G_CALLBACK(handle_row_activated), NULL);

	g_timeout_add (100, handle_timeout, NULL);

	for (int i = 1; i < argc; i++)
		add_recursively(argv[i]);

	update_ui();

	gtk_main();

	return 0;
}

