/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

void assert_string_equal (string a, string b) {
    if (a != b) {
        GLib.error ("'" + a + "' != '" + b + "'");
    }
}

struct TestCase {
    public string data;
    public string? expected;
}

const TestCase[] folders = {
    { "Sent Mail", null },
    { "Messages envoy&AOk-s", "Messages envoyés" },
    { "&ZeVnLIqe-", "日本語" },
    { "&U,BTFw-", "台北" },
    { "&BkUGMwZIBi8GJwYq-", "مسودات" }
};

void parsing_filename_folder () {
    foreach (var foldername in folders) {
        string decoded = Postler.Folders.decode_foldername (foldername.data);
        string expected = foldername.expected;
        if (expected == null)
            expected = foldername.data;
        assert_string_equal (decoded, expected);
    }
}

const TestCase[] filenames = {
    { "127.162_10.blutschink,U=129:2,",    ":2,S" },
    { "127.162_10.blutschink,U=129:2,S",   ":2," },
    { "127.162_10.blutschink,U=129:2,T",   ":2,ST" },
    { "127.162_10.blutschink,U=129:2,D",   ":2,DS" },
    { "127.162_10.blutschink,U=129:2,DT",  ":2,DST" },
    { "127.162_10.blutschink,U=129:2,DST", ":2,DT" }
};

void parsing_filename_flags () {
    foreach (var filename in filenames) {
        string toggled = Postler.Messages.toggle_flag (filename.data, 'S');
        string expected = filename.expected;
        if (expected == null)
            expected = filename.data.chr (-1, ':');
        assert_string_equal (toggled.chr (-1, ':'), expected);
    }
}

const TestCase[] addresses = {
    { "Klaus <klaus@heim.at>", "Klaus klaus@heim.at" },
    { "Klaus Kaufmann <klausk@heim.at>", "Klaus Kaufmann klausk@heim.at" },
    { "Klaus", null },
    { "kaufmann@heim.at", null },
    { "<kaufmann@heim.at>", "kaufmann@heim.at kaufmann@heim.at" },
    { "<mailto:kaufmann@heim.at>", "kaufmann@heim.at kaufmann@heim.at" },
    { "'Klaus' <klaus@heim.at>", "Klaus klaus@heim.at" },
    { "\"Klaus\" <klaus@heim.at>", "Klaus klaus@heim.at" },
    { "Bert O'Brian <bert+@uni.edu>", "Bert O'Brian bert+@uni.edu" },
    { "\"gtk+\" (b.g.o) <b@g.o>", "gtk+ (b.g.o) b@g.o" },
    { "\"S \\\"T\\\" D\" <t@u.com>", "S `T` D t@u.com" },
    { "=?iso-8859-1?Q?M=2C_Alf?= <alf.m@vw.de>",
      "=?iso-8859-1?Q?M=2C_Alf?= alf.m@vw.de" }
};

void parsing_headers_address () {
    foreach (var address in addresses) {
        string[] formatted = Postler.Messages.parse_address (address.data);
        string expected = address.expected;
        if (expected == null)
            expected = address.data + " " + address.data;
        assert_string_equal (string.joinv (" ", formatted), expected);
    }
}

const TestCase[] linkifies = {
    { "Kl <kl@ha.at>", "<a href=\"Kl &lt;kl@ha.at&gt;\" title=\"Kl &lt;kl@ha.at&gt;\">Kl</a>" },
    { "kl@ha.at", "<a href=\"kl@ha.at\" title=\"kl@ha.at\">kl@ha.at</a>" },
    { "\"Kl\" <kl@ha.at>",
      "<a href=\"&quot;Kl&quot; &lt;kl@ha.at&gt;\" title=\"&quot;Kl&quot; &lt;kl@ha.at&gt;\">Kl</a>" },
    { "=?iso-8859-1?Q?M=2C_Alf?= <alf.m@vw.de>",
      "<a href=\"=?iso-8859-1?Q?M=2C_Alf?= &lt;alf.m@vw.de&gt;\" "
      + "title=\"=?iso-8859-1?Q?M=2C_Alf?= &lt;alf.m@vw.de&gt;\">"
      + "=?iso-8859-1?Q?M=2C_Alf?=</a>" }
};

void parsing_headers_linkify () {
    foreach (var linkify in linkifies) {
        string formatted = Postler.Content.linkify_address (linkify.data, null);
        string expected = linkify.expected;
        assert_string_equal (formatted, expected);
    }
    assert_string_equal (Postler.Content.linkify_address ("Kl <kl@ha.at>",
        "?from=Kl &lt;kl@ha.at&gt;&subject=Re: Tomaten"),
      "<a href=\"Kl &lt;kl@ha.at&gt;?from=Kl &lt;kl@ha.at&gt;&subject=Re: Tomaten\" "
      + "title=\"Kl &lt;kl@ha.at&gt;\">Kl</a>");
}

const TestCase[] mailers = {
    { "Claws Mail 3.7.6 (GTK+ 2.20.1; x86_64)", "Claws Mail 3.7.6" },
    { "SAP Web 7.00", "SAP Web 7.00" },
    { "Apple Mail (2.936)", "Apple Mail" },
    { "Evolution 2.30.1 ", "Evolution 2.30.1" },
    { "Evolution 2.28.3 (2.28.3-1.fc12) ", "Evolution 2.28.3" },
    { "Mozilla/5.0 (X11; U) Gecko/201 Thunderbird/3.0.4", "Thunderbird 3.0.4" },
    { "postler/0.0.1-6d417e0 (X11; U) WebKit/531.2+", "postler 0.0.1-6d417e0" }
};

void parsing_headers_mailer () {
    foreach (var mailer in mailers) {
        string formatted = Postler.Content.format_x_mailer (mailer.data);
        assert_string_equal (formatted, mailer.expected);
    }
}

const TestCase[] encodeds = {
    { "=?iso-8859-1?Q?Lila_L=F6ffel?=", "Lila Löffel" },
    { "=?iso-8859-15?q?S=FCddeutsche?=", "Süddeutsche" },
    { "=?iso-8859-15?q?=FCddeutsche?=", "üddeutsche" },
    { "=?UTF-8?B?QW5kcsOpIFN0w7ZzZWw=?=", "André Stösel" },
    { "=?iso-8859-15?q?S=FCddeutsche_?==?iso-8859-15?q?W=FCrste?=",
      "Süddeutsche Würste" },
    { "[Kritiker] =?iso-8859-15?q?S=FCddeutsche_?==?iso-8859-15?q?W=FCrste?=",
      "[Kritiker] Süddeutsche Würste" }
};

void parsing_headers_encoded () {
    string charset;
    foreach (var encoded in encodeds) {
        string decoded = Postler.Messages.parse_encoded (encoded.data, out charset);
        assert_string_equal (decoded, encoded.expected);
    }
}

void main (string[] args) {
    Test.init (ref args);
    Test.add_func ("/parsing/filename/folder", parsing_filename_folder);
    Test.add_func ("/parsing/filename/flags", parsing_filename_flags);
    Test.add_func ("/parsing/headers/address", parsing_headers_address);
    Test.add_func ("/parsing/headers/linkify", parsing_headers_linkify);
    Test.add_func ("/parsing/headers/mailer", parsing_headers_mailer);
    Test.add_func ("/parsing/headers/encoded", parsing_headers_encoded);
    Test.run ();
}


