/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file balloon_layer.cpp
 * \brief Implementation of the ptb::balloon_layer class.
 * \author Sebastien Angibaud
 */
#include "ptb/layer/balloon_layer.hpp"

#include "engine/level.hpp"

/*----------------------------------------------------------------------------*/
const std::string ptb::balloon_layer::s_default_name( "balloon_layer" );
const unsigned int ptb::balloon_layer::s_border = 20;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::balloon_layer::msg_add_speaker::msg_add_speaker()
  : m_speaker(NULL)
{

} // msg_add_speaker::msg_add_speaker()

/*----------------------------------------------------------------------------*/
/*
 * \brief Set the speaker.
 * \param speaker The speaker.
 */
void ptb::balloon_layer::msg_add_speaker::set_speaker
( ptb::speaker_item* speaker)
{
  m_speaker = speaker;
} // msg_add_speaker::set_speaker()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to the balloon layer.
 * \param that The balloon layer to apply the message to.
 */
bool ptb::balloon_layer::msg_add_speaker::apply_to( balloon_layer& that )
{
  bool ok = false;

  if ( m_speaker != NULL )
    ok = that.add_speaker(m_speaker);

  return ok;
} // msg_add_speaker::apply_to()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::balloon_layer::balloon_layer()
  : bear::communication::messageable(default_name())
{

} // balloon_layer::balloon_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the layer.
 */
void ptb::balloon_layer::build()
{
  get_level_globals().register_item(*this);
} // balloon_layer::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the players balloon and update local values.
 * \param elapsed_time The time elapsed since the last call.
 */
void ptb::balloon_layer::progress( bear::universe::time_type elapsed_time )
{
  if ( ! get_level().is_paused() ) 
    {
      std::list<speaker_item*>::iterator it;
      for ( it = m_speakers.begin(); it != m_speakers.end(); ++it)
	if ( !(*it)->has_finished_to_chat() )
	  {
	    (*it)->get_balloon()->progress(elapsed_time);
	    adjust_balloon_position(*it);
	  }
      
      for ( it = m_speakers.begin(); it != m_speakers.end(); )
	if ( (*it)->has_finished_to_chat() )
	  {
	    std::list<speaker_item*>::iterator it2 = it;
	    ++it;
	    m_speakers.erase(it2);
	  }
	else
	  ++it;
    }
} // balloon_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the scene elements of the layer.
 * \param e (out) The scene elements.
 */
void ptb::balloon_layer::render( scene_element_list& e ) const
{
  std::list<speaker_item*>::const_iterator it;

  for ( it = m_speakers.begin(); it != m_speakers.end(); ++it)
    if ( !(*it)->has_finished_to_chat() )
      (*it)->get_balloon()->render(e);
} // balloon_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a speaker.
 * \param speaker The speaker.
 * Return true if the speaker is added.
 */
bool ptb::balloon_layer::add_speaker( ptb::speaker_item* speaker)
{
  bool contain = false;
  std::list<speaker_item*>::iterator it;
  for ( it = m_speakers.begin();
        ( it != m_speakers.end() ) && ( ! contain ); ++it)
    if ( *it == speaker )
      contain = true;

  if ( ! contain )
    {
      m_speakers.push_back(speaker);
      speaker->create_balloon(get_level_globals());
    }

  return ( ! contain );
} // balloon_layer::add_balloon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the default name of the balloon_layer.
 */
const std::string& ptb::balloon_layer::default_name()
{
  return s_default_name;
} // balloon_layer::default_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the position of balloon of a speaker.
 * \param speaker The speaker.
 */
void ptb::balloon_layer::adjust_balloon_position(ptb::speaker_item* speaker)
{
  bool top = speaker->get_balloon()->get_spike_top();
  bool right = speaker->get_balloon()->get_spike_right();

  bear::universe::rectangle_type cam( get_level().get_camera_focus() );
  bear::universe::position_type pos(0,0);

  pos.x = adjust_position_x(speaker,cam,right);
  pos.y = adjust_position_y(speaker,cam,top);

  speaker->get_balloon()->set_position
    ( pos.cast_value_type_to<unsigned int>() );
  speaker->get_balloon()->set_spike_right(right);
  speaker->get_balloon()->set_spike_top(top);
} // balloon_layer::adjust_balloon_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the x-position of the fram.
 * \param speaker The speaker.
 * \param cam The focus of the camera.
 * \param right Indicates if the spike is at the right.
 */
bear::universe::coordinate_type ptb::balloon_layer::adjust_position_x
( speaker_item* speaker, const bear::universe::rectangle_type& cam,
  bool& right)
{
  double ratio = get_size().x / cam.size().x;

  bear::universe::coordinate_type pos;
  bear::gui::frame* frame = speaker->get_balloon()->get_frame();

  // compute relative positions
  bear::universe::coordinate_type r_left = ratio *
    (speaker->get_speaker()->get_left() - cam.left());
  bear::universe::coordinate_type r_right = ratio *
    (speaker->get_speaker()->get_left() +
     speaker->get_speaker()->get_size().x - cam.left());

  if ( right )
    right = ( r_right + frame->width() + 2*s_border <= get_size().x );
  else
    right = ( r_left < frame->width() + 2*s_border );

  if ( right )
    pos = r_right + s_border;
  else
    pos = r_left - frame->width() - s_border;

  if( r_left + s_border >= get_size().x )
    {
      pos = get_size().x - frame->width() - s_border;
      right = false;
    }
  else if ( r_right <= s_border )
    {
      pos = s_border;
      right = true;
    }

  return pos;
} // balloon_layer::adjust_position_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the y-position of the frame.
 * \param speaker The speaker.
 * \param cam The focus of the camera.
 * \param top Indicates if the spike is on the top.
 */
bear::universe::coordinate_type ptb::balloon_layer::adjust_position_y
( speaker_item* speaker, const bear::universe::rectangle_type& cam, bool& top)
{
  double ratio = get_size().y / cam.size().y;

  bear::universe::coordinate_type pos;
  bear::gui::frame* frame = speaker->get_balloon()->get_frame();

  // compute relative positions
  bear::universe::coordinate_type r_top = ratio *
    ( cam.bottom() + cam.size().y - speaker->get_speaker()->get_top());
  bear::universe::coordinate_type r_bottom = ratio *
    ( cam.bottom() + cam.size().y - speaker->get_speaker()->get_bottom());

  if ( top )
    top = ( r_top >= frame->height() + s_border );
  else
    top = ( r_bottom + frame->height() + 2*s_border >= get_size().y );

  if ( top )
    pos = r_top - frame->height() - s_border;
  else
    pos = r_bottom + s_border;

  if ( r_top + s_border >= get_size().y )
    {
      top = true;
      pos = get_size().y - frame->height() - s_border;
    }
  else if ( r_bottom <= s_border )
    {
      top = false;
      pos = s_border;
    }

  return pos;
} // balloon_layer::adjust_position_y()
