/*
    Bear Engine - model editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/model_editor.cpp"
 * \brief Implementation of the bf::model_editor class.
 * \author Julien Jorge.
 */
#include "bf/model_editor.hpp"

#include "bf/compiled_file.hpp"
#include "bf/main_frame.hpp"
#include "bf/gui_model.hpp"
#include "bf/version.hpp"
#include "bf/wx_facilities.hpp"
#include "bf/xml/model_file.hpp"

#include <wx/tooltip.h>
#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bf::model_editor::model_editor()
  : m_locale( wxLocale::GetSystemLanguage() ), m_main_frame(NULL)
{
  claw::logger.set( new claw::console_logger() );
  claw::logger.set_level( claw::log_verbose );

  m_locale.AddCatalog( wxT("bear-factory") );
} // model_editor::model_editor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bf::model_editor::~model_editor()
{
  // frames are deleted by wxWidgets

  claw::logger.clear();
} // model_editor::~model_editor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when the application is initializing.
 */
bool bf::model_editor::OnInit()
{
  bool result = false;

  if ( !show_help() )
    if ( !show_version() )
      {
        if ( find_and_erase_option( wxT("--compile"), wxT("-c") ) )
          compile_arguments();
        else
          {
            result = true;
            init_app();
          }
      }

  return result;
} // model_editor::OnInit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the application.
 */
void bf::model_editor::init_app()
{
  wxToolTip::Enable(true);

  m_main_frame = new main_frame;
  m_main_frame->Show();

  for (int i=1; i<argc; ++i)
    if ( wxString(argv[i]) != wxT("--") )
      m_main_frame->load_model( argv[i] );
} // model_editor::init_app()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the models and exit.
 */
bool bf::model_editor::compile_arguments()
{
  bool result(true);

  for (int i=1; i<argc; ++i)
    if ( wxString(argv[i]) != wxT("--") )
      {
        bool ok(false);
        model* mdl(NULL);

        try
          {
            xml::model_file reader;
            mdl = reader.load(argv[i]);

            if ( check_model(*mdl) )
              ok = compile_model(*mdl, argv[i]);
          }
        catch(...)
          { }

        delete mdl;

        if ( !ok )
          {
            std::cerr << "Error when processing '" << argv[i] << '\''
                      << std::endl;
            result = false;
          }
      }

  return result;
} // model_editor::compile_arguments()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if a model is valid.
 * \param mdl The model to check.
 */
bool bf::model_editor::check_model( const model& mdl )
{
  bool result(true);
  model::const_action_iterator it;

  for ( it=mdl.action_begin(); it!=mdl.action_end(); ++it )
    if ( !it->get_auto_next().empty() ) 
      if ( !mdl.has_action( it->get_auto_next() )  )
        {
          result = false;
          std::cerr << "Unknown action '" << it->get_auto_next() << '\''
                    << std::endl;
        }

  return result;
} // model_editor::check_model()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a model.
 * \param mdl The model to compile.
 * \param path The path to the model file.
 * \return true if the compilation went ok.
 */
bool
bf::model_editor::compile_model( const model& mdl, const wxString& path )
{
  bool result(true);

  try
    {
      std::string std_path( wx_to_std_string(path) );
      std::size_t pos = std_path.rfind(".mdl");

      if ( pos != std::string::npos )
        std_path = std_path.substr(0, pos);

      std_path += ".cm";

      std::ofstream f( std_path.c_str() );
      if (f)
        {
          compiled_file cf(f);
          mdl.compile(cf);
        }
    }
  catch(...)
    {
      result = false;
    }

  return result;
} // model_editor::compile_model()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the command line usage.
 * \return true if the usage has been shown.
 */
bool bf::model_editor::show_help()
{
  if ( find_and_erase_option(wxT("--help"), wxT("-h")) )
    {
      std::cout << "usage:\n" << wx_to_std_string(argv[0])
                << " [option] [file...]\n"
        "Where the options are:\n\n"
        "\t--compile, -c\tCompile the models and exit,\n"
        "\t--help, -h\tDisplay this help and exit,\n"
        "\t--version, -v\tDisplay the version of the program and exit."
                << std::endl;
      return true;
    }
  else
    return false;
} // model_editor::show_help()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the version of the program
 * \return true if the version has been shown.
 */
bool bf::model_editor::show_version()
{
  if ( find_and_erase_option(wxT("--version"), wxT("-v")) )
    {
      std::cout << BF_VERSION_STRING << std::endl;
      return true;
    }
  else
    return false;
} // model_editor::show_version()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if an option is present on the command line and remove it.
 * \param long_name The long name of the option.
 * \param show_name The short name of the option.
 */
bool bf::model_editor::find_and_erase_option
( const wxString& long_name, const wxString& short_name )
{
  int index(0);
  bool stop(false);

  for (int i=1; !stop && (index==0) && (i<argc); ++i)
    if ( (argv[i] == long_name) || (argv[i] == short_name))
      index = i;
    else
      stop = wxString(argv[i]) == wxT("--");

  if ( index != 0 )
    {
      for ( int i=index; (i+1 != argc); ++i )
        argv[i] = argv[i+1];

      --argc;
      argv[argc] = NULL;
    }

  return index != 0;
} // model_editor::find_and_erase_option()
