/*
  Bear Engine - Model editor

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/mark_selection.cpp
 * \brief Implementation of the bf::mark_selection class.
 * \author Julien Jorge
 */
#include "bf/mark_selection.hpp"

#include "bf/mark.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
bf::mark_selection::mark_selection()
  : m_mark(NULL)
{

} // mark_selection::mark_selection()

/*----------------------------------------------------------------------------*/
void bf::mark_selection::insert( mark* mark, bool main_selection )
{
  CLAW_PRECOND( mark != NULL );

  m_group.insert(mark);

  if ( main_selection || (m_mark == NULL) )
    m_mark = mark;
} // mark_selection::insert()

/*----------------------------------------------------------------------------*/
void bf::mark_selection::insert( const mark_selection& s )
{
  if ( !s.empty() )
    {
      m_group.insert(s.m_group.begin(), s.m_group.end());
      m_mark = s.m_mark;
    }
} // mark_selection::insert()

/*----------------------------------------------------------------------------*/
void bf::mark_selection::remove( mark* mark )
{
  CLAW_PRECOND( mark != NULL );

  m_group.erase(mark);

  if ( m_mark == mark )
    {
      if ( m_group.empty() )
        m_mark = NULL;
      else
        m_mark = *m_group.begin();
    }
} // mark_selection::remove()

/*----------------------------------------------------------------------------*/
void bf::mark_selection::remove( const mark_selection& s )
{
  for (const_iterator it=s.begin(); it!=s.end(); ++it)
    m_group.erase(*it);

  if ( s.m_group.find(m_mark) != s.m_group.end() )
    {
      if ( m_group.empty() )
        m_mark = NULL;
      else
        m_mark = *m_group.begin();
    }
} // mark_selection::remove()

/*----------------------------------------------------------------------------*/
bool bf::mark_selection::is_selected( mark* mark ) const
{
  CLAW_PRECOND( mark != NULL );

  return m_group.find(mark) != m_group.end();
} // mark_selection::is_selected()

/*----------------------------------------------------------------------------*/
bool bf::mark_selection::is_main_selection( mark* mark ) const
{
  CLAW_PRECOND( mark != NULL );

  return m_mark == mark;
} // mark_selection::is_main_selection()

/*----------------------------------------------------------------------------*/
void bf::mark_selection::clear()
{
  m_mark = NULL;
  m_group.clear();
} // mark_selection::clear()

/*----------------------------------------------------------------------------*/
bool bf::mark_selection::empty() const
{
  CLAW_PRECOND( (m_mark != NULL) || m_group.empty() );

  return m_mark == NULL;
} // mark_selection::empty()

/*----------------------------------------------------------------------------*/
bf::mark* bf::mark_selection::get_main_selection() const
{
  return m_mark;
} // mark_selection::get_main_selection()

/*----------------------------------------------------------------------------*/
bf::mark_selection::const_iterator bf::mark_selection::begin() const
{
  return m_group.begin();
} // mark_selection::begin()

/*----------------------------------------------------------------------------*/
bf::mark_selection::const_iterator bf::mark_selection::end() const
{
  return m_group.end();
} // mark_selection::end()
