/*
    Bear Engine - Model editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/mark_placement.cpp
 * \brief Implementation of the bf::mark_placement class.
 * \author Julien Jorge
 */
#include "bf/mark_placement.hpp"

#include "bf/compiled_file.hpp"
#include "bf/mark.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param m The identifier of mark concerned by this placement.
 */
bf::mark_placement::mark_placement( mark* m )
  : m_mark(m), m_x(0), m_y(0), m_depth(0), m_angle(0), m_visible(true)
{

} // mark_placement::mark_placement()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor by copy.
 * \param p The mark_placment to copy.
 */
bf::mark_placement::mark_placement( const mark_placement& p )
  : m_mark(p.m_mark), m_x(p.m_x), m_y(p.m_y), m_depth(p.m_depth), 
    m_angle(p.m_angle), m_visible(p.m_visible)
{

} // mark_placement::mark_placement()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the x-position of the mark.
 * \param x The new position.
 */
void bf::mark_placement::set_position_x( double x )
{
  m_x = x;
} // mark_placement::set_position_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the y-position of the mark.
 * \param y The new position.
 */
void bf::mark_placement::set_position_y( double y )
{
  m_y = y;
} // mark_placement::set_position_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the mark
 * \param x The new x-position.
 * \param y The new y-position.
 */
void bf::mark_placement::set_position( double x, double y )
{
  set_position_x(x);
  set_position_y(y);
} // mark_placement::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the x-position of the mark.
 */
double bf::mark_placement::get_position_x() const
{
  return m_x;
} // mark_placement::get_position_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the y-position of the mark.
 */
double bf::mark_placement::get_position_y() const
{
  return m_y;
} // mark_placement::get_position_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the visibility of the mark
 * \param v Tell if the mark is visible or not.
 */
void bf::mark_placement::set_visibility( bool v )
{
  m_visible = v;
} // mark_placement::set_visibility()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the visibility of the mark
 */
bool bf::mark_placement::is_visible() const
{
  return m_visible;
} // mark_placement::is_visible()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the depth-position of the mark.
 * \param p The new position.
 */
void bf::mark_placement::set_depth_position( int p )
{
  m_depth = p;
} // mark_placement::set_depth_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the depth-position of the mark.
 */
int bf::mark_placement::get_depth_position() const
{
  return m_depth;
} // mark_placement::get_depth_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the angle of the mark.
 * \param a The new angle.
 */
void bf::mark_placement::set_angle( double a )
{
  m_angle = a;
} // mark_placement::set_angle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the angle of the mark.
 */
double bf::mark_placement::get_angle() const
{
  return m_angle;
} // mark_placement::get_angle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the mark concerned by this placement.
 */
const bf::mark* bf::mark_placement::get_mark() const
{
  return m_mark;
} // mark_placement::get_mark()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the mark_placement.
 * \param f The file in which we compile.
 */
void bf::mark_placement::compile( compiled_file& f ) const
{
  f << m_x << m_y << m_depth << m_angle << m_visible;
} // mark_placement::compile()
