/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/item_selection.hpp
 * \brief A selection of items.
 * \author Julien Jorge
 */
#ifndef __BF_ITEM_SELECTION_HPP__
#define __BF_ITEM_SELECTION_HPP__

#include <set>

namespace bf
{
  class item_instance;

  /**
   * \brief A selection of items.
   * \author Julien Jorge
   */
  class item_selection
  {
  public:
    /** \brief Iterator on the selected items. */
    typedef std::set<item_instance*>::const_iterator const_iterator;

  public:
    /** \brief Constructor. */
    item_selection();

    /**
     * \brief Insert an item in the selection.
     * \param item The item to insert.
     * \param main_selection Tell if the main selection is set on the item.
     */
    void insert( item_instance* item, bool main_selection = false );

    /**
     * \brief Insert a selection in the selection.
     * \param s The items to insert.
     */
    void insert( const item_selection& s );

    /**
     * \brief Remove an item from the selection.
     * \param item The item to insert.
     */
    void remove( item_instance* item );

    /**
     * \brief Remove a selection from the selection.
     * \param s The items to remove.
     */
    void remove( const item_selection& s );

    /**
     * \brief Tell if an item is selected.
     * \param item The item to check.
     */
    bool is_selected( item_instance const* item ) const;

    /**
     * \brief Tell if an item is the main selection.
     * \param item The item to check.
     */
    bool is_main_selection( item_instance const* item ) const;

    /** \brief Make the selection empty. */
    void clear();

    /** \brief Tell if the selection is empty. */
    bool empty() const;

    /** \brief Get the main selection. */
    item_instance* get_main_selection() const;

    /** \brief Get an iterator on the first item in the selection. */
    const_iterator begin() const;

    /** \brief Get an iterator after the last item in the selection. */
    const_iterator end() const;

  private:
    /** \brief The main selection. */
    item_instance* m_item;

    /** \brief All the selected items. */
    std::set<item_instance*> m_group;
      
  }; // class item_selection
} // namespace bf

#endif // __BF_ITEM_SELECTION_HPP__
