/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file mouse.cpp
 * \brief Implementation of the bear::input::mouse class.
 * \author Julien Jorge
 */
#include "input/mouse.hpp"
#include <SDL/SDL.h>
#include <algorithm>
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
std::vector<std::string> bear::input::mouse::s_button_strings;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::input::mouse::mouse()
{
  if ( s_button_strings.empty() )
    default_mouse_code_strings();

  int x, y;
  SDL_GetMouseState( &x, &y );
  m_position.set(x, y);
} // mouse::mouse()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a mouse_code to a human-readable string.
 * \param k The code to convert.
 */
const std::string& bear::input::mouse::get_name_of( mouse_code b )
{
  CLAW_PRECOND( b < s_button_strings.size() );

  return s_button_strings[b];
} // mouse::get_name_of()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the name of a key.
 * \param k The key to name.
 * \param s The string describing this key.
 */
void bear::input::mouse::set_name_of( mouse_code k, const std::string& s )
{
  CLAW_PRECOND( k < s_button_strings.size() );

  s_button_strings[k] = s;
} // mouse::set_name_of()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the first pressed button.
 */
bear::input::mouse::const_iterator bear::input::mouse::begin() const
{
  return m_pressed_buttons.begin();
} // mouse::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator after the last pressed button.
 */
bear::input::mouse::const_iterator bear::input::mouse::end() const
{
  return m_pressed_buttons.end();
} // mouse::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if no keys are pressed.
 */
bool bear::input::mouse::empty() const
{
  return m_pressed_buttons.empty();
} // mouse::empty()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the mouse.
 */
const claw::math::coordinate_2d<unsigned int>&
bear::input::mouse::get_position() const
{
  return m_position;
} // mouse::get_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Re-read the status of all keys.
 * \pre The caller is an instance of bear::input::system.
 */
void bear::input::mouse::refresh()
{
  Uint8 buttons;
  int x, y;

  buttons = SDL_GetMouseState( &x, &y );
  m_position.set(x, y);
  m_pressed_buttons.clear();

  // magic number: 6 is the number of codes for the SDL default mouse buttons
  for (unsigned int i=1; i!=6; ++i)
    if ( buttons & SDL_BUTTON(i) )
      m_pressed_buttons.push_back( sdl_button_to_local(i) );
} // mouse::refresh()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a SDLK_* value to the corresponding mouse_code.
 * \param sdl_val The SDL value to convert.
 * \param shift Tell if a shift button is considered pressed.
 * \param alt Tell if an alt button is considered pressed.
 */
bear::input::mouse::mouse_code bear::input::mouse::sdl_button_to_local
( unsigned int sdl_val ) const
{
  switch(sdl_val)
    {
    case 1 : return mc_left_button;   break;
    case 2 : return mc_middle_button; break;
    case 3 : return mc_right_button;  break;
    case 4 : return mc_wheel_up;      break;
    case 5 : return mc_wheel_down;    break;
    default: return mc_invalid;
    }

} // mouse::sdl_mouse_to_local()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the s_button_string table with default strings.
 */
void bear::input::mouse::default_mouse_code_strings()
{
  s_button_strings.resize(c_mouse_codes_count);
  std::fill( s_button_strings.begin(), s_button_strings.end(), "Undefined" );

  s_button_strings[mc_left_button]   = "mouse left";
  s_button_strings[mc_middle_button] = "mouse middle";
  s_button_strings[mc_right_button]  = "mouse right";

  s_button_strings[mc_wheel_up]   = "wheel up";
  s_button_strings[mc_wheel_down] = "wheel down";
} // mouse::default_buttons_strings()
