/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file engine/brick/code/with_trigger.cpp
 * \brief Implementation of the bear::engine::with_trigger class.
 * \author Julien Jorge
 */
#include "engine/item_brick/with_trigger.hpp"

#include "engine/item_brick/with_toggle.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::engine::with_trigger::with_trigger()
  : m_colliding_item(NULL), m_collision_info(NULL)
{

} // with_trigger::with_trigger()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The instance to copy from.
 */
bear::engine::with_trigger::with_trigger( const with_trigger& that )
  : m_toggle(that.m_toggle), m_colliding_item(NULL), m_collision_info(NULL)
{
  // the collision_in_expression in the expression cannot be copied, so we just
  // not copy the expression.
} // with_trigger::with_trigger()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assignment.
 * \param that The instance to copy from.
 */
bear::engine::with_trigger
bear::engine::with_trigger::operator=( const with_trigger& that )
{
  m_toggle = that.m_toggle;
  // the collision_in_expression in the expression cannot be copied, so we just
  // not copy the expression.

  return *this;
} // with_trigger::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Turn the toggles on if the condition is true. Do nothing otherwise.
 * \param activator The item that activates the toggle.
 */
void bear::engine::with_trigger::toggle_if_condition( base_item* activator )
{
  if ( m_condition )
    {
      handle_list::iterator it;
      std::list<handle_list::iterator> dead;

      for (it=m_toggle.begin(); it!=m_toggle.end(); ++it)
        if ( *it == (with_toggle*)NULL )
          dead.push_back(it);
        else
          (*it)->toggle(true, activator);

      for ( ; !dead.empty(); dead.pop_front() )
        m_toggle.erase(dead.front());
    }
} // with_trigger::toggle_if_condition()

/*----------------------------------------------------------------------------*/
/**
 * \brief Turn the toggles on/off according to the condition.
 * \param activator The item that activates the toggle.
 */
void bear::engine::with_trigger::toggle_as_condition( base_item* activator )
{
  handle_list::iterator it;
  std::list<handle_list::iterator> dead;
  const bool b = m_condition.evaluate();

  for (it=m_toggle.begin(); it!=m_toggle.end(); ++it)
    if ( *it == (with_toggle*)NULL )
      dead.push_back(it);
    else
      (*it)->toggle(b, activator);

  for ( ; !dead.empty(); dead.pop_front() )
    m_toggle.erase(dead.front());
} // with_trigger::toggle_as_condition()

/*----------------------------------------------------------------------------*/
/**
 * \brief Switch the status of the toggles if the condition is verified.
 * \param activator The item that activates the toggle.
 */
void bear::engine::with_trigger::switch_on_condition( base_item* activator )
{
  if ( m_condition )
    {
      handle_list::iterator it;
      std::list<handle_list::iterator> dead;

      for (it=m_toggle.begin(); it!=m_toggle.end(); ++it)
        if ( *it == (with_toggle*)NULL )
          dead.push_back(it);
        else
          (*it)->toggle(activator);

      for ( ; !dead.empty(); dead.pop_front() )
        m_toggle.erase(dead.front());
    }
} // with_trigger::switch_on_condition()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a toggle.
 * \param t The item with the toggle (must inherit from with_toggle).
 */
void bear::engine::with_trigger::add_toggle( base_item* t )
{
  m_toggle.push_back( handle_type((universe::physical_item*)t) );
} // with_trigger::add_toggle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the condition to verify to turn on the toggles.
 */
void bear::engine::with_trigger::set_condition( const boolean_expression& e )
{
  m_condition = e;
} // with_trigger::set_condition()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the informations on a collision that can be used in an expression.
 */
bear::engine::collision_in_expression
bear::engine::with_trigger::get_collision_in_expression() const
{
  return collision_in_expression(&m_colliding_item, &m_collision_info);
} // with_trigger::get_collision_in_expression()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the informations on a collision that can be used in an expression.
 * \param that The colliding item.
 * \param info Some informations about the collision.
 */
void bear::engine::with_trigger::set_collision_data
( base_item& that, universe::collision_info& info )
{
  m_colliding_item = &that;
  m_collision_info = &info;
} // with_trigger::set_collision_data()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the informations on a collision that can be used in an expression.
 */
void bear::engine::with_trigger::clear_collision_data()
{
  m_colliding_item = NULL;
  m_collision_info = NULL;
} // with_trigger::clear_collision_data()
