/*
	 pingpong - Really free server for Amateur Radio convers
	 Copyright (C) 2005 Joop Stakenborg <pg4i@amsat.org>

	 This program is free software; you can redistribute it and/or modify
	 it under the terms of the GNU General Public License as published by
	 the Free Software Foundation; either version 2, or (at your option)
	 any later version.

	 This program is distributed in the hope that it will be useful,
	 but WITHOUT ANY WARRANTY; without even the implied warranty of
	 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
	 GNU General Public License for more details.

	 You should have received a copy of the GNU General Public License
	 along with this program; if not, write to the Free Software Foundation,
	 Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	
*/

#include <glib.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include "pingpong.h"
#include "types.h"
#include "client.h"
#include "init.h"
#include "messages.h"
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

extern gchar *buffer, *confdir;
extern GList *client_list;

void send_msg(gint fd, gchar *id, gchar *msg)
{
	ssize_t res;
	gchar *sendmsg = NULL;

	if (msg)
		if (g_strcasecmp(id, PINGPONG_MSG) == 0)
			sendmsg = g_strdup_printf("%s", msg);

	if (sendmsg)
	{
		res = write(fd, sendmsg, strlen(sendmsg));
		g_free(sendmsg);
	}
}

gchar *recv_msg(gint fd)
{
	gchar **rxsplit =	NULL, *command = NULL;
	ssize_t len;

	buffer = g_new0(gchar, 1024); /* start with a new buffer */

	/* return with len =0 for a disconnect */
	if ((len = read(fd, buffer, 1024)) == 0) return NULL;

	/* check if it is a command and split it up*/
	if (g_strncasecmp(buffer, "/", 1) == 0)
	{
		(void *)g_strdelimit(buffer, "\n", ' '); /* strip out CR's */
		(void *)g_strstrip(buffer);
		rxsplit = g_strsplit(buffer, " ", 0);
		if (rxsplit[0])
		{ 
			command = g_strdup(rxsplit[0]); /* "/q" */
			if (rxsplit[1])
			{
				buffer = g_strdup(rxsplit[1]); /* "/q <message>", buffer wil hold message */
				if (rxsplit[2]) buffer = g_strdup_printf("%s %s", buffer, rxsplit[2]);
			} /* "/n <name> <channel>", buffer will hold name and channel */
			else 
				buffer = NULL; /* empty the buffer when a single command like "/h" or "/q" is entered */
		} 
		g_strfreev(rxsplit);
	}

	if (command) return(command); 
	else return(buffer);
}

void user_msg(Client *client)
{
	guint i, j, k;
	Client *pos;
	gchar *sendmsg;
	gint channelpos, channelcli;

	if (!buffer || strlen(buffer) > MAX_BUFFER) return;

	/* send a message to everyone, except the one who has send it*/
	/* only send to users who share the same channel in their channel list */
	for(i = 0; i < g_list_length(client_list); i++)
	{
		pos = g_list_nth_data(client_list, i);
		if (pos == client) continue;
		if (pos->channel == client->channel)
		{
			sendmsg = g_strdup_printf("<%s>: %s", client->name, buffer);
			send_msg(pos->fd, PINGPONG_MSG, sendmsg);
		}
		for (j = 0; j < g_list_length(pos->channel_list); j++) 
			for (k = 0; k < g_list_length(client->channel_list); k++)
			{
				channelpos = GPOINTER_TO_INT(g_list_nth_data(pos->channel_list, j));
				channelcli = GPOINTER_TO_INT(g_list_nth_data(client->channel_list, k));
				if ((channelpos == channelcli) && 
						(pos->channel != client->channel) &&
						(pos->channel) != channelcli)
				{
					sendmsg = g_strdup_printf("<%d:%s>: %s", channelpos, client->name, buffer);
					send_msg(pos->fd, PINGPONG_MSG, sendmsg);
				}
			}
	}
}


void send_joined_msg(gint fd, gchar *name, gint channel)
{
	gchar *sendmsg, nowtime[16];
	size_t timeres;
	time_t now;

	now = time(NULL);
	timeres = strftime(nowtime, 16, "%H:%M", localtime(&now));

	sendmsg = g_strdup_printf("*** (%s) %s@%s joined channel %d\n", 
		nowtime, name, getservername(), channel);

	send_msg(fd, PINGPONG_MSG, sendmsg);

	g_free(sendmsg);
}

void send_left_msg(gint fd, gchar *name, gint channel, gchar *command)
{
	gchar *sendmsg, nowtime[16], *bye_msg = g_strdup("connection closed");
	size_t timeres;
	time_t now;

	now = time(NULL);
	timeres = strftime(nowtime, 16, "%H:%M", localtime(&now));

	/* check which command has been used to quit and see if a goodbye message has been sent */
	if (command)
	{
		if (g_strncasecmp(command, PINGPONG_CMD_QUIT, 2) == 0)
		{
			if (buffer) bye_msg = g_strdup_printf("\"%s\"", buffer);
			else bye_msg = g_strdup("/quit");
		}
		else if (g_strncasecmp(command, PINGPONG_CMD_LEAVE, 2) == 0)
			bye_msg = g_strdup("/leave");
	}


	sendmsg = g_strdup_printf("*** (%s) %s@%s left channel %d (%s)\n", 
		nowtime, name, getservername(), channel, bye_msg);

	send_msg(fd, PINGPONG_MSG, sendmsg);

	g_free(sendmsg);
	g_free(bye_msg);
}

void help_msg(gint fd)
{
	gchar *help, *header, *footer;
	gboolean help_exist = TRUE, helper = FALSE;
	FILE *fp;
	GString *helpline = g_string_new("");
	gint c;

	if (confdir) help = g_strdup_printf("%s/pingpong.help", confdir);
		else help = g_strdup_printf("%s/pingpong.help", PINGPONG_SYSCONFDIR);

	if ((fp = fopen(help, "r")) == NULL) help_exist = FALSE;
	if (help_exist)
	{
		if (!buffer)
		{ /*client has type "/h" */
			header = g_strdup_printf("Conversd Commands for %s version %s at %s\n", PACKAGE, VERSION, getservername());
			footer = g_strdup_printf("***\n");
			send_msg(fd, PINGPONG_MSG, header);
			while((c = fgetc(fp)) != EOF)
			{
				helpline = g_string_append_c(helpline, c);
				if (c == '\n')
				{
					/* search for a line starting with "/" and display it */
					if (g_strncasecmp(helpline->str, "/", 1) == 0)
					{
						if (g_strncasecmp(helpline->str, "/ ", 2) == 0 || 
								g_strncasecmp(helpline->str, "/\n", 2) == 0)
							helpline = g_string_erase(helpline, 0, 1);
						send_msg(fd, PINGPONG_MSG, helpline->str);
					}
					helpline = g_string_new("");
				}
			}
			send_msg(fd, PINGPONG_MSG, footer);
			g_free(header);
			g_free(footer);
			fclose(fp);
		} 
		else
		{ /* client has type "/h <subject>" */
			helper = FALSE;
			header = g_strdup_printf("*** Usage:\n");
			footer = g_strdup_printf("***\n");
			send_msg(fd, PINGPONG_MSG, header);
			while((c = fgetc(fp)) != EOF) {
				helpline = g_string_append_c(helpline, c);
				if (c == '\n') {
					/* search for a line starting with "/", compare with subject and display until the next "/" */
					if (helper && g_strncasecmp(helpline->str, "/", 1) != 0)
						send_msg(fd, PINGPONG_MSG, helpline->str);
					if (g_strncasecmp(helpline->str, "/", 1) == 0)
					{
						helper = FALSE;
						helpline = g_string_erase(helpline, 0, 1);
						if (g_strncasecmp(buffer, helpline->str, 1) == 0)
						{
							helpline = g_string_insert_c(helpline, 0, '/');
							send_msg(fd, PINGPONG_MSG, helpline->str);
							helper = TRUE;
						}
					}
				helpline = g_string_new("");
				}
			}
			send_msg(fd, PINGPONG_MSG, footer);
			g_free(header);
			g_free(footer);
		}
	}
	g_free(help);
	g_string_free(helpline, TRUE);
}

void who_msg(gint fd)
{
	guint i;
	Client *pos;
	gchar *list = NULL, *sendmsg;

	for (i = 0; i < g_list_length(client_list); i++)
	{
		pos = g_list_nth_data(client_list, i);
		if (i == 0)
			list = g_strdup(pos->name); 
		else
			list = g_strdup_printf("%s %s", list, pos->name);
	}
	send_msg(fd, PINGPONG_MSG, "Channel Users\n");
	sendmsg = g_strdup_printf( "        %s\n", list);
	send_msg(fd, PINGPONG_MSG, sendmsg);
	send_msg(fd, PINGPONG_MSG, "***\n");
	if (list) g_free(list);
}
