/***************************************************************************
 *            palm.c
 *
 *  Mon Nov 21 22:56:18 2005
 *  Copyright  2005-2009  Neil Williams <linux@codehelp.co.uk>
 *  Copyright (C) 2002, 2003, 2004 Philip Blundell <philb@gnu.org>
 *  Copyright (C) 2005, 2006 Florian Boor <florian@kernelconcepts.de>
 ****************************************************************************/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _GNU_SOURCE
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <regex.h>
#include <ctype.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <glib/gprintf.h>
#include <qof.h>
#include <popt.h>
#include "pi-source.h"
#include "pi-debug.h"
#include "pi-socket.h"
#include "pi-file.h"
#include "pi-header.h"
#include "pi-util.h"
#include "pi-todo.h"
#include "pi-expense.h"
#include "qof-main.h"
#include "pilot-qof.h"
#include "pilot-todo.h"
#include "qof-address.h"
#include "qof-datebook.h"
#ifdef HAVE_QOFEXPENSES
#include <qof-expenses.h>
#endif
#include "pilot-expenses.h"
#include "pilot-expenses-p.h"
#include "palm.h"

/** used to print debug logs. */
static QofLogModule log_module = PQ_MOD_CLI;

#ifdef HAVE_QOFEXPENSES
static Expense_t *
expense_get_pilot (QofEntity * ent)
{
	glong nanosecs;
	QofDate *qd;
	QofTime * qt;
	Expense_t *qe;
	gchar * string;
	const QofParam * param;
	gint32 (*int32_getter) (QofEntity*, const QofParam*);

	g_return_val_if_fail (ent != NULL, NULL);
	if (g_strcmp0(ent->e_type, PILOT_LINK_QOF_EXPENSES) == 0)
		return pq_expense_get_pilot(ent);
	qe = g_new0 (Expense_t, 1);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_DATE);
	qt = param->param_getfcn (ent, param);
	if (!qt)
		return NULL;
	nanosecs = qof_time_get_nanosecs (qt);
	qd = qof_date_from_qtime (qt);
	qof_date_to_struct_tm (qd, &qe->date, &nanosecs);
	qof_date_free (qd);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_TYPE);
	string = param->param_getfcn (ent, param);
	qe->type = ExpenseTypefromString (string);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_PAYMENT);
	string = param->param_getfcn (ent, param);
	qe->payment = ExpensePaymentfromString (string);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_CURRENCY);
	int32_getter = (gint32 (*)(QofEntity*, const QofParam*)) param->param_getfcn;
	qe->currency = int32_getter(ent, param);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_AMOUNT);
	qe->amount = param->param_getfcn (ent, param);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_VENDOR);
	qe->vendor = param->param_getfcn (ent, param);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_CITY);
	qe->city = param->param_getfcn (ent, param);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_ATTENDEES);
	qe->attendees = param->param_getfcn (ent, param);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_NOTE);
	qe->note = param->param_getfcn (ent, param);
	return qe;
}
#else
#define expense_get_pilot pq_expense_get_pilot
#endif

static gint
exp_unpack (QofEntity * ent, gpointer user_data)
{
	Expense_t *qe;
	pi_buffer_t *pi_buf;
	PQContext *context;
	const QofParam * param;
	size_t G_GNUC_UNUSED len;
	gint size;
	gchar * string;

	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	g_return_val_if_fail (ent != NULL, -1);
	qe = expense_get_pilot (ent);
	g_return_val_if_fail (qe != NULL, -1);
	pi_buf = (pi_buffer_t *) context->pi_buf;
	len = sizeof (pi_buf->data);
	size = unpack_Expense (qe, pi_buf->data, pi_buf->allocated);
	param = qof_class_get_parameter (PILOT_LINK_QOF_EXPENSES, EXP_CATEGORY);
	qof_util_param_edit ((QofInstance*)ent, param);
	qof_util_param_set_string (ent, param, context->names[context->ent_category]);
	qof_util_param_commit ((QofInstance*)ent, param);
	/* lookup currency_code in currency table. */
	param = qof_class_get_parameter (PILOT_LINK_QOF_EXPENSES, EXP_CURRENCY);
	string = g_strdup_printf ("%d", qe->currency);
	qof_util_param_edit ((QofInstance*)ent, param);
	qof_util_param_set_string (ent, param, string);
	qof_util_param_commit ((QofInstance*)ent, param);
	g_free (string);
	return size;
}

static gint
exp_pack (QofEntity * ent, gpointer user_data)
{
	PQContext *context;
	Expense_t *qe;
	gint size, len;

	size = 0;
	len = PQ_DEF_BUFSZ;
	context = (PQContext *) user_data;
	g_return_val_if_fail ((context), -1);
	qe = expense_get_pilot (ent);
	/* pack_Expense still uses the old prototype
	   using len instead of pi_buf->used. */
	size = pack_Expense (qe, context->pi_buf->data, len);
	/* work around the old prototype */
	if (size > 0)
		context->pi_buf->used = size;
	return size;
}

static gint
exp_pref_unpack (QofEntity * ent, gpointer user_data)
{
	struct ExpensePref pref_e;
	PQContext *context;

	/* There is never an entity at this stage */
	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	ENTER (" ");
	unpack_ExpensePref (&pref_e, context->pref_buf, PQ_DEF_BUFSZ);
	context->pi_exp_pref.default_currency = pref_e.defaultCurrency;
	context->pi_exp_pref.unit_of_distance = pref_e.unitOfDistance;
//	populate_currencies ();
	LEAVE (" ");
	return 0;
}

static gint
exp_appinfo_unpack (QofEntity * ent, gpointer user_data)
{
	ExpenseAppInfo_t app_e;
	PQContext *context;
	gint name_count;

	/* There is never an entity at this stage */
	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	ENTER (" ");
	unpack_ExpenseAppInfo (&app_e, context->app_buf->data, PQ_DEF_BUFSZ);
	for (name_count = 0; name_count < 16; name_count++)
	{
		g_sprintf (context->names[name_count], "%s",
			app_e.category.name[name_count]);
	}
	context->pi_cat = &app_e.category;
	LEAVE (" ");
	return 0;
}

static gint
qof_exp_free (QofEntity * ent, gpointer user_data)
{
	Expense_t *qe;

	g_return_val_if_fail (ent != NULL, -1);
	ENTER (" ");
	qe = expense_get_pilot (ent);
	free_Expense (qe);
	LEAVE (" ");
	return 0;
}

static PQPack expenses_pack_def = {
  e_type          :  PILOT_LINK_QOF_EXPENSES,
  pack_func       :  exp_pack,
  unpack_func     :  exp_unpack,
  free_pack_func  :  qof_exp_free,
  palm_db_name    :  Expense_DB,
  app_info_unpack :  exp_appinfo_unpack,
  db_pref_unpack  :  exp_pref_unpack,
  pref_creator    :  EXPENSE_CREATOR,
  pref_flag       :  Expense_Pref,
};

static gint
qof_todo_free (QofEntity * ent, gpointer user_data)
{
	ToDo_t *qt;

	g_return_val_if_fail (ent != NULL, -1);
	ENTER (" ");
	qt = todo_get_pilot((QofInstance *)ent);
	free_ToDo (qt);
	LEAVE (" ");
	return 0;
}

static gint
todo_unpack (QofEntity * ent, gpointer user_data)
{
	pi_buffer_t *pi_buf;
	ToDo_t *qt;
	PQContext *context;
	const QofParam * param;
	gint size;

	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	g_return_val_if_fail (ent != NULL, -1);
	ENTER (" ent_type=%s", ent->e_type);
	qt = todo_get_pilot((QofInstance *)ent);
	g_return_val_if_fail (qt != NULL, -1);
	pi_buf = (pi_buffer_t *) context->pi_buf;
	size = 0;
	size = unpack_ToDo (qt, pi_buf, TODO_VERSION);
	param = qof_class_get_parameter (PILOT_LINK_QOF_TODO, TODO_CATEGORY);
	qof_util_param_edit ((QofInstance*)ent, param);
	qof_util_param_set_string (ent, param, context->names[context->ent_category]);
	PINFO (" category=%s", context->names[context->ent_category]);
	qof_util_param_commit ((QofInstance*)ent, param);
	LEAVE (" ");
	return size;
}

static gint
todo_pack (QofEntity * ent, gpointer user_data)
{
	ToDo_t *qt;
	PQContext *context;
	gint size;

	size = 0;
	context = (PQContext *) user_data;
	g_return_val_if_fail ((context || ent), -1);
	ENTER (" ");
	qt = todo_get_pilot((QofInstance *)ent);
	size = pack_ToDo (qt, context->pi_buf, TODO_VERSION);
	LEAVE (" ");
	return size;
}

static gint
todo_appinfo_unpack (QofEntity * ent, gpointer user_data)
{
	ToDoAppInfo_t app_t;
	PQContext *context;
	gint name_count;

	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	ENTER (" ");
	unpack_ToDoAppInfo (&app_t, context->app_buf->data, PQ_DEF_BUFSZ);
	for (name_count = 0; name_count < 16; name_count++)
		g_sprintf (context->names[name_count], "%s",
			app_t.category.name[name_count]);
	context->pi_cat = &app_t.category;
	LEAVE (" ");
	return 0;
}

static PQPack todo_pack_def = {
  e_type          :  PILOT_LINK_QOF_TODO,
  pack_func       :  todo_pack,
  unpack_func     :  todo_unpack,
  free_pack_func  :  qof_todo_free,
  palm_db_name    :  "ToDoDB",
  app_info_unpack :  todo_appinfo_unpack,
};

/** @name Packing and unpacking objects.

Each pilot-link object has a structure that reflects the
relevant database on the Palm in a binary form that can be easily
manipulated. This binary format needs to be converted (packed) before
it can be understood by the Palm and converted back (unpacked) before
to be usable in pilot-qof. Each object therefore needs to be packed
(prior to writing to the Palm) and unpacked (after reading from
the Palm) using it's own specialised functions. QOF uses a generic
function pointer to call the correct function for the current
QOF object and this function then calls the correct pack or
unpack function for the underlying pilot-link object.
@{ */

/** \brief Pack an Address object for the Palm.

Convert the pilot-link object into a form suitable for
HotSync to the Palm.

 \todo Pack and unpack multiple addresses per name.
*/
static gint
address_pack (QofEntity * ent, gpointer user_data)
{
	PQContext *context;
	Address_t *qa;
	gint size;

	size = 0;
	context = (PQContext *) user_data;
	g_return_val_if_fail ((context || ent), -1);
	qa = address_get_pilot((QofInstance *)ent);
	size = pack_Address (qa, context->pi_buf, ADDRESS_VERSION);
	PINFO (" result=%d", size);
	return size;
}

/** \brief Unpack and Address object from the Palm.

Convert the record read from the Palm HotSync into a
usable pilot-link object - itself contained within a
QOF object.
*/
static gint
address_unpack (QofEntity * ent, gpointer user_data)
{
	pi_buffer_t *pi_buf;
	Address_t *qa;
	PQContext *context;
	const QofParam * param;
	gint size;

	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	g_return_val_if_fail (ent != NULL, -1);
	qa = address_get_pilot((QofInstance *)ent);
	g_return_val_if_fail (qa != NULL, -1);
	pi_buf = (pi_buffer_t *) context->pi_buf;
	size = 0;
	size = unpack_Address (qa, pi_buf, ADDRESS_VERSION);
	param = qof_class_get_parameter (PILOT_LINK_QOF_ADDRESS, ADDR_CATEGORY);
	qof_util_param_edit ((QofInstance*)ent, param);
	PINFO (" category=%s", context->names[context->ent_category]);
	qof_util_param_set_string (ent, param, context->names[context->ent_category]);
	qof_util_param_commit ((QofInstance*)ent, param);
//	addr_setCategory (obj, context->names[context->ent_category]);
	return size;
}

/** \brief unpack the application information from the Palm database.

The application information contains the list of category names
and other elements that are consistent for all records in the
database.
*/
static gint
addr_appinfo_unpack (QofEntity * ent, gpointer user_data)
{
	AddressAppInfo_t app_a;
	PQContext *context;
	gint name_count;

	/* There is never an entity at this stage */
	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	ENTER (" ");
	unpack_AddressAppInfo (&app_a, context->app_buf->data, PQ_DEF_BUFSZ);
	for (name_count = 0; name_count < 16; name_count++)
	{
		g_sprintf (context->names[name_count], "%s",
			app_a.category.name[name_count]);
	}
	context->pi_cat = &app_a.category;
	LEAVE (" ");
	return 0;
}

/** \brief free the memory associated with the pilot-link object.

Use specialised pilot-link functions to free the underlying
object. The QOF entity can then be freed without problems.
*/
static gint
qof_address_free (QofEntity * ent, gpointer user_data)
{
	Address_t *qa;

	g_return_val_if_fail (ent != NULL, -1);
	ENTER (" ");
	qa = address_get_pilot((QofInstance *)ent);
	free_Address (qa);
	LEAVE (" ");
	return 0;
}
/** @} */

/** @} */

static PQPack address_pack_def = {
  e_type           : PILOT_LINK_QOF_ADDRESS,
  pack_func        : address_pack,
  unpack_func      : address_unpack,
  free_pack_func   : qof_address_free,
  palm_db_name     : "AddressDB",
  app_info_unpack  : addr_appinfo_unpack,
};

/** unpack the Palm datebook record into the entity. 

\todo Problems with queries not finding repeat events.
*/

static gint
datebook_unpack (QofEntity * ent, gpointer user_data)
{
	pi_buffer_t *pi_buf;
	Appointment_t *qa;
//	QofDateBook *obj, *clone;
	QofInstance *inst;
//	QofBook *book;
	KvpFrame *inst_frame;
	QofTime *qt, *qt_increment, *qt_end, *qt_repeat;
	PQContext *context;
	const QofParam * param;
	gint size, i, day_interval, month_interval;

	context = (PQContext *) user_data;
	day_interval = 0;
	month_interval = 0;
	g_return_val_if_fail (context != NULL, -1);
	g_return_val_if_fail (ent != NULL, -1);
	inst = (QofInstance *) ent;
	inst_frame = qof_instance_get_slots (inst);
	qa = datebook_get_pilot ((QofInstance *) ent);
	g_return_val_if_fail (qa != NULL, -1);
	pi_buf = (pi_buffer_t *) context->pi_buf;
	size = 0;
	size = unpack_Appointment (qa, pi_buf, DATEBOOK_VERSION);	// 0.12
	param = qof_class_get_parameter (PILOT_LINK_QOF_DATEBOOK, DATEBOOK_CATEGORY);
	qof_util_param_edit ((QofInstance*)ent, param);
	qof_util_param_set_string (ent, param, context->names[context->ent_category]);
	qof_util_param_commit ((QofInstance*)ent, param);
//	datebook_setCategory (obj, context->names[context->ent_category]);
	/* Use <= and handle zero if < omits last entry (&623) */
	for (i = 0; i < qa->exceptions; i++)
	{
		gchar *extend;

		extend = NULL;
		DEBUG (" Number of datebook repeat exceptions=%d", qa->exceptions);
		qt = qof_time_from_tm (&qa->exception[i], 0);
		extend = g_strdup_printf ("%s/%d", DATEBOOK_KVP_PATH, i + 1);
		kvp_frame_set_time (inst_frame, extend, qt);
		inst->kvp_data = inst_frame;
		g_free (extend);
	}
	if (qa->repeatType == repeatNone)
		qa->repeatEnd = qa->end;
	/* set a marker for the interval. Do the iteration once, outside the switch. */
	switch (qa->repeatType)
	{
	case repeatNone:
		{
			day_interval = 0;
			month_interval = 0;
			break;
		}
	case repeatDaily:
		{
			day_interval = 1;
			month_interval = 0;
			break;
		}
	case repeatWeekly:
		{
			day_interval = 7;
			month_interval = 0;
			break;
		}
	case repeatMonthlyByDay:
	case repeatMonthlyByDate:
		{
			day_interval = 0;
			month_interval = 1;
			break;
		}
	case repeatYearly:
		{
			day_interval = 0;
			month_interval = 12;
			break;
		}
	default:
		{
			PERR (" unsupported repeatType=%d", qa->repeatType);
		}
	}
	if (day_interval == 0 && month_interval == 0)
		return size;
	/* Now create a repeater in the SAME book. */
	param = qof_class_get_parameter (PILOT_LINK_QOF_DATEBOOK, DATEBOOK_BEGIN);
	qt = param->param_getfcn (ent, param);
	param = qof_class_get_parameter (PILOT_LINK_QOF_DATEBOOK, DATEBOOK_END);
	qt_end = param->param_getfcn (ent, param);
	if (qa->repeatForever == 0)
	{
		param = qof_class_get_parameter (PILOT_LINK_QOF_DATEBOOK, DATEBOOK_REPEAT_END);
		qt_repeat = param->param_getfcn (ent, param);
//		qt_repeat = datebook_getRepeatEnd (obj);
	}
	else
	{
		QofDate *qd;
		/*  if qa->repeatForever == 1 (true), calculate year and a day from
		   start_time. qof_date_add_months(12)
		   qof_date_add_days(1). Store for use as repeatEnd
		 */
		DEBUG (" qa->repeatForever == 1");
		qt_repeat = qt;
		qd = qof_date_from_qtime (qt_repeat);
		qof_date_addmonths (qd, 12, FALSE);
		qof_date_adddays (qd, 1);
		qof_date_free (qd);
	}
	qt_increment = qt;
	/*  qa->exception is an array of struct tm* qa->exceptions long. */
	/* while datebook_getBegin is less (earlier) than repeat_end */
	while (qof_time_cmp (qt_increment, qt_repeat) < 0)
	{
		gboolean skip;

		skip = FALSE;
		if (day_interval)
		{
			QofDate *qd;
			qd = qof_date_from_qtime (qt_increment);
			qof_date_adddays (qd, day_interval);
			qt_increment = qof_date_to_qtime (qd);
			qof_date_free (qd);
			qd = qof_date_from_qtime (qt_end);
			qof_date_adddays (qd, day_interval);
			qt_end = qof_date_to_qtime (qd);
			qof_date_free (qd);
		}
		if (month_interval)
		{
			QofDate *qd;
			qd = qof_date_from_qtime (qt_increment);
			qof_date_addmonths (qd, month_interval, FALSE);
			qt_increment = qof_date_to_qtime (qd);
			qof_date_free (qd);
			qd = qof_date_from_qtime (qt_end);
			qof_date_addmonths (qd, month_interval, FALSE);
			qt_end = qof_date_to_qtime (qd);
			qof_date_free (qd);
		}
		for (i = 0; i < qa->exceptions; i++)
		{
			QofDate *qd;

			qd = qof_date_from_qtime(qt_increment);
			if ((qd->qd_year == qa->exception[i].tm_year) &&
				(qd->qd_mday == qa->exception[i].tm_mday) &&
				(qd->qd_mon == qa->exception[i].tm_mon))
			{
				/* exclude */
				skip = TRUE;
			}
		}
		/* create the repeater */
		if (!skip)
			datebook_repeater_clone (ent, qt_end, qt_increment);
	}
	return size;
}

/** \brief pack the entity into a Palm datebook record */
static gint
datebook_pack (QofEntity * ent, gpointer user_data)
{
	PQContext *context;
	Appointment_t *qa;
//	QofDateBook *obj;
	QofTime *qt;
	KvpFrame *frame;
	gint size, i;
	gchar *path;

	size = 0;
	i = 1;
	context = (PQContext *) user_data;
	g_return_val_if_fail ((context || ent), -1);
	ENTER (" ");
	qa = datebook_get_pilot ((QofInstance *) ent);
//	obj = (QofDateBook *) ent;
//	if (obj->repeater == TRUE)
//		return 0;
//	qa = &obj->wrap;
	size = pack_Appointment (qa, context->pi_buf, DATEBOOK_VERSION);	// 0.12
	/* pack slots into exceptions */
	frame = qof_instance_get_slots ((QofInstance *) ent);
	if (frame)
	{
		path = g_strdup_printf ("%s/%d", DATEBOOK_KVP_PATH, i);
		while (kvp_frame_get_value (frame, path))
		{
			qt = kvp_frame_get_time (frame, path);
			if (qt)
			{
				QofDate *qd;
				gboolean result;
				qd = qof_date_from_qtime (qt);
				result = qof_date_to_struct_tm (qd,
					&qa->exception[i], 0);
				if(!result)
					PERR (" failed to set exception time: "
					"out of range");
			}
			g_free (path);
			i++;
			path = g_strdup_printf ("%s/%d", DATEBOOK_KVP_PATH, i);
		}
		g_free (path);
	}
	LEAVE (" ");
	return size;
}

/** \brief unpack the application information.

The AppInfo contains the category list (each record only holds
the index value of this list as the category value) and other
preferences.
*/
static gint
datebook_appinfo_unpack (QofEntity * ent, gpointer user_data)
{
	AppointmentAppInfo_t app_db;
	PQContext *context;
	gint name_count;

	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	ENTER (" ");
	unpack_AppointmentAppInfo (&app_db, context->app_buf->data,
		PQ_DEF_BUFSZ);
	for (name_count = 0; name_count < 16; name_count++)
	{
		g_sprintf (context->names[name_count], "%s",
			app_db.category.name[name_count]);
	}
	context->pi_cat = &app_db.category;
	LEAVE (" ");
	return 0;
}

/** free all memory related to the datebook record. */
static gint
qof_datebook_free (QofEntity * ent, gpointer user_data)
{
	Appointment_t *qd;
//	QofDateBook *obj;

	g_return_val_if_fail (ent != NULL, -1);
	ENTER (" ");
	qd = datebook_get_pilot ((QofInstance *)ent);
//	obj = (QofDateBook *) ent;
//	qd = &obj->wrap;
	free_Appointment (qd);
	LEAVE (" ");
	return 0;
}

static PQPack datebook_pack_def = {
  e_type          : PILOT_LINK_QOF_DATEBOOK,
  pack_func       : datebook_pack,
  unpack_func     : datebook_unpack,
  free_pack_func  : qof_datebook_free,
  palm_db_name    : "DatebookDB",
  app_info_unpack : datebook_appinfo_unpack,
};

gboolean packing_registration (void)
{
	pilot_qof_pack_register (&expenses_pack_def);
	pilot_qof_pack_register (&todo_pack_def);
	pilot_qof_pack_register (&address_pack_def);
	pilot_qof_pack_register (&datebook_pack_def);
	return TRUE;
}
