/***************************************************************************
 *            pilot-todo.c
 *
 *  Sun Jan 30 16:07:21 2005
 *  Copyright  2005-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/** @addtogroup ToDo
	@{ */
/** @file  pilot-todo.c
	@brief QOF ToDo definitions for pilot-link
	@author Copyright (c) 2004-2007 Neil Williams <linux@codehelp.co.uk>
*/

#include <glib.h>
#include <glib/gprintf.h>
#include <qof.h>
#include "pilot-todo.h"
#include "qof-main.h"
#include "pilot-qof.h"
#include "pi-todo.h"

#ifdef COMPATIBILITY_MODE
typedef enum
{
	todo_v1,
} todoType;
#endif

#define QOF_TODO_DESC "Pilot-Link QOF ToDo"

static QofLogModule log_module = PQ_MOD_PILOT;

typedef struct
{
	QofInstance inst;
	gchar *category;
	ToDo_t wrap;				// 0.12
} QofTodo;

static QofTodo *
todo_create (QofBook * book)
{
	ToDo_t *qt;
	QofTodo *obj;
	QofCollection *coll;
	GList *all;

	obj = g_new0 (QofTodo, 1);
	qof_instance_init (&obj->inst, PILOT_LINK_QOF_TODO, book);
	coll = qof_book_get_collection (book, PILOT_LINK_QOF_TODO);
	all = qof_collection_get_data (coll);
	all = g_list_prepend (all, obj);
	qof_collection_set_data (coll, all);
	qt = &obj->wrap;
	return obj;
}

ToDo_t * todo_get_pilot (QofInstance * inst)
{
	QofTodo * t;

	t = (QofTodo*)inst;
	return &t->wrap;
}

static gint
todo_getLength (QofTodo * t)
{
	ToDo_t *qt;

	g_return_val_if_fail (t != NULL, 0);
	qt = &t->wrap;
	return qt->indefinite;
}

static QofTime*
todo_getTimeDue (QofTodo * t)
{
	ToDo_t *qtd;
	QofTime *qt;

	g_return_val_if_fail (t != NULL, NULL);
	qtd = &t->wrap;
	qt = qof_time_from_tm (&qtd->due, 0);
	return qt;
}

static gint
todo_getPriority (QofTodo * t)
{
	ToDo_t *qt;

	g_return_val_if_fail (t != NULL, 0);
	qt = &t->wrap;
	return qt->priority;
}

static gint
todo_getComplete (QofTodo * t)
{
	ToDo_t *qt;

	g_return_val_if_fail (t != NULL, 0);
	qt = &t->wrap;
	return qt->complete;
}

static gchar *
todo_getDescription (QofTodo * t)
{
	ToDo_t *qt;

	g_return_val_if_fail (t != NULL, NULL);
	qt = &t->wrap;
	return qt->description;
}

static gchar *
todo_getNote (QofTodo * t)
{
	ToDo_t *qt;

	g_return_val_if_fail (t != NULL, NULL);
	qt = &t->wrap;
	return qt->note;
}

static gchar *
todo_getCategory (QofTodo * t)
{
	g_return_val_if_fail (t != NULL, NULL);
	return t->category;
}

static void
todo_setLength (QofTodo * t, gint l)
{
	ToDo_t *qt;

	g_return_if_fail (t != NULL);
	qt = &t->wrap;
	qt->indefinite = l;
}

static void
todo_setTimeDue (QofTodo * t, QofTime *qt)
{
	ToDo_t *qtd;
	gboolean result;
	QofDate *qdate;

	g_return_if_fail (t != NULL);
	qtd = &t->wrap;
	qdate = qof_date_from_qtime (qt);
	result = qof_date_to_struct_tm (qdate, &qtd->due, 0);
	if(!result)
		PERR (" Date too large for due.");
	qof_date_free (qdate);
}

static void
todo_setPriority (QofTodo * t, gint p)
{
	ToDo_t *qt;

	g_return_if_fail (t != NULL);
	if ((p < 1) || (p > 5))
		p = 1;
	qt = &t->wrap;
	qt->priority = p;
}

static void
todo_setComplete (QofTodo * t, gint c)
{
	ToDo_t *qt;

	g_return_if_fail (t != NULL);
	qt = &t->wrap;
	qt->complete = c;
}

static void
todo_setDescription (QofTodo * t, gchar * d)
{
	ToDo_t *qt;

	g_return_if_fail (t != NULL);
	qt = &t->wrap;
	qt->description = g_strdup (qof_util_make_utf8 (d));
}

static void
todo_setNote (QofTodo * t, gchar * n)
{
	ToDo_t *qt;

	g_return_if_fail (t != NULL);
	qt = &t->wrap;
	qt->note = g_strdup (qof_util_make_utf8 (n));
}

static void
todo_setCategory (QofTodo * t, gchar * n)
{
	g_return_if_fail (t != NULL);
	t->category = g_strdup (qof_util_make_utf8 (n));
}

static const gchar *
todoPrintable (gpointer instance)
{
	return todo_getDescription ((QofTodo *) instance);
}

static QofObject todo_object_def = {
  interface_version:QOF_OBJECT_VERSION,
  e_type:PILOT_LINK_QOF_TODO,
  type_label:QOF_TODO_DESC,
  create:(gpointer) todo_create,
  book_begin:NULL,
  book_end:NULL,
  is_dirty:qof_collection_is_dirty,
  mark_clean:qof_collection_mark_clean,
  foreach:qof_collection_foreach,
  printable:todoPrintable,
  version_cmp:(gint (*)(gpointer, gpointer)) qof_instance_version_cmp,
};

gboolean
ToDoRegister (void)
{
	static QofParam params[] = {
		{TODO_LENGTH, QOF_TYPE_INT32, (QofAccessFunc) todo_getLength,
				(QofSetterFunc) todo_setLength},
		{TODO_DATE, QOF_TYPE_TIME, (QofAccessFunc) todo_getTimeDue,
				(QofSetterFunc) todo_setTimeDue},
		{TODO_PRIORITY, QOF_TYPE_INT32, (QofAccessFunc) todo_getPriority,
				(QofSetterFunc) todo_setPriority},
		{TODO_COMPLETE, QOF_TYPE_INT32, (QofAccessFunc) todo_getComplete,
				(QofSetterFunc) todo_setComplete},
		{TODO_DESCRIPTION, QOF_TYPE_STRING,
				(QofAccessFunc) todo_getDescription,
				(QofSetterFunc) todo_setDescription},
		{TODO_CATEGORY, QOF_TYPE_STRING, (QofAccessFunc) todo_getCategory,
				(QofSetterFunc) todo_setCategory},
		{TODO_NOTE, QOF_TYPE_STRING, (QofAccessFunc) todo_getNote,
				(QofSetterFunc) todo_setNote},
		{QOF_PARAM_BOOK, QOF_ID_BOOK,
				(QofAccessFunc) qof_instance_get_book, NULL},
		{QOF_PARAM_GUID, QOF_TYPE_GUID,
				(QofAccessFunc) qof_instance_get_guid, NULL},
		{NULL},
	};
	qof_class_register (PILOT_LINK_QOF_TODO, NULL, params);

	return qof_object_register (&todo_object_def);
}

/** @} */
