
/*
 * Copyright (C) 2006 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: vdr.c 2601 2007-07-25 08:29:04Z mschwerin $
 *
 */
#include "config.h"

#include <errno.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "codeset.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "oxine.h"
#include "utils.h"
#include "vdr.h"

#ifdef HAVE_VDR

extern oxine_t *oxine;

#define BUFSIZE     2048

static int read_cnt;
static char *read_ptr;
static char read_buf[BUFSIZE];

static size_t
_read (int fd, char *ptr)
{
    if (read_cnt <= 0) {
      again:
        if ((read_cnt = recv (fd, read_buf, sizeof (read_buf), 0)) < 0) {
            if (errno == EINTR) {
                goto again;
            }
            return (-1);
        }
        else if (read_cnt == 0) {
            return (0);
        }
        read_ptr = read_buf;
    }

    read_cnt--;
    *ptr = *read_ptr++;
    return (1);
}


static size_t
readline (int fd, void *vptr, size_t maxlen)
{
    size_t n;
    size_t rc;
    char c;
    char *ptr;

    ptr = vptr;
    for (n = 1; n < maxlen; n++) {
        if ((rc = _read (fd, &c)) == 1) {
            *ptr++ = c;
            if (c == '\n')
                break;                                          /* newline is stored, like fgets() */
        }
        else if (rc == 0) {
            *ptr = 0;
            return (n - 1);                                     /* EOF, n - 1 bytes were read */
        }
        else {
            return (-1);                                        /* error, errno set by read() */
        }
    }

    *ptr = 0;                                                   /* null terminate like fgets() */
    return (n);
}


static bool
send_command (int sd, const char *command)
{
    debug ("->VDR: %s", command);

    if (send (sd, command, strlen (command), 0) == -1) {
        error (_("Could not send command to server: %s!"), strerror (errno));
        return false;
    }
    if (send (sd, "\n", 1, 0) == -1) {
        error (_("Could not send command to server: %s!"), strerror (errno));
        return false;
    }

    return true;
}


static bool
recv_answer (int sd, char *buf, int buf_size, char *expected_code)
{
    if (readline (sd, buf, buf_size) == -1) {
        error (_("Could not read from server: %s!"), strerror (errno));
        return false;
    }

    int len = strlen (buf) - 1;
    if ((buf[len] == '\n')
        || (buf[len] == '\r'))
        buf[len--] = '\0';

    if (strncmp (buf, expected_code, strlen (expected_code)) != 0) {
        error (_("Invalid SVDRP response: %s"), buf);
        return false;
    }

    debug ("<-VDR: %s", buf);

    return true;
}


static bool
simple_command (int sd, const char *command, char *expected_code)
{
    if (!send_command (sd, command)) {
        close (sd);
        return false;
    }

    char buf[BUFSIZE];
    if (!recv_answer (sd, buf, BUFSIZE, expected_code)) {
        close (sd);
        return false;
    }

    return true;
}


static int
svdrp_connect (const char *host, int port)
{
    char buf[BUFSIZE];
    int sd;
    struct hostent *hp;
    struct sockaddr_in pin;

    /* Go find out about the desired host machine. */
    if ((hp = gethostbyname (host)) == 0) {
        error (_("Could not resolve '%s': %s!"), host, strerror (errno));
        return 0;
    }

    /* Fill in the socket structure with host information. */
    memset (&pin, 0, sizeof (pin));
    pin.sin_family = AF_INET;
    pin.sin_addr.s_addr = ((struct in_addr *) (hp->h_addr))->s_addr;
    pin.sin_port = htons (port);

    /* Grab an internet domain socket. */
    if ((sd = socket (AF_INET, SOCK_STREAM, 0)) == -1) {
        error (_("Could not create socket: %s!"), strerror (errno));
        return 0;
    }

    /* Connect to port on host. */
    if (connect (sd, (struct sockaddr *) &pin, sizeof (pin)) == -1) {
        error (_("Could not connect to '%s': %s!"), host, strerror (errno));
        close (sd);
        return 0;
    }

    /* Wait for server to say hello. */
    if (!recv_answer (sd, buf, BUFSIZE, "220")) {
        close (sd);
        return 0;
    }

    return sd;
}


static bool
vdr_swap_cb (void *p1, void *p2)
{
    fileitem_t *f1 = (fileitem_t *) p1;
    fileitem_t *f2 = (fileitem_t *) p2;

    return (strcmp (f1->sort_by, f2->sort_by) < 0);
}


bool
vdr_timers_read (filelist_t * filelist)
{
    const int port = config_get_number ("television.vdr.svdrp.port");
    const char *host = config_get_string ("television.vdr.svdrp.host");
    int sd = svdrp_connect (host, port);
    if (!sd) {
        ho_free (filelist->error);
        filelist->error = ho_strdup (_("Connection failure..."));
        return false;
    }

    bool result = false;
    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init ("ISO-8859-1", codeset);

    if (!send_command (sd, "LSTT")) {
        goto error;
    }

    char buf[BUFSIZE];
    do {
        if (!recv_answer (sd, buf, BUFSIZE, "250")) {
            goto error;
        }

        int num = atoi (buf + 4);
        char *txt = recode (xr, index (buf + 4, ' ') + 1);

        char *name = trim_whitespace (txt + 31);
        char *end = index (name, ':');
        if (end) {
            end[0] = '\0';
        }
        char *dp = index (name, '|');
        while (dp) {
            dp[0] = ':';
            dp = index (name, '|');
        }

        struct tm ts;
        char *ts_str = txt + 4;
        strptime (ts_str, "%Y-%m-%d:%H%M", &ts);
        ts_str = ho_strdup_strftime ("%d.%m.%Y\t%H:%M", &ts);

        struct tm te;
        char *te_str = txt + 20;
        strptime (te_str, "%H%M", &te);
        te_str = ho_strdup_strftime ("%H:%M", &te);

        char *title = ho_strdup_printf ("%s-%s\t%s", ts_str,
                                        te_str, name);
        char *mrl = ho_strdup_printf ("%s%d", FILELIST_VDR_TIMERS_MRL, num);
        fileitem_t *fileitem = filelist_add (filelist, title, mrl,
                                             FILE_TYPE_VDR_TIMER);
        fileitem->sort_by = ho_strdup_strftime ("%F %H:%M", &ts);

        ho_free (te_str);
        ho_free (ts_str);
        ho_free (title);
        ho_free (mrl);
        ho_free (txt);
    } while (strncmp (buf, "250 ", 4) != 0);

    filelist_sort (filelist, vdr_swap_cb);

    /* Close connection */
    if (!simple_command (sd, "QUIT", "221")) {
        goto error;
    }

    result = true;
  error:
    close (sd);
    recode_done (xr);
    ho_free (codeset);

    return result;
}


bool
vdr_recordings_read (filelist_t * filelist)
{
    const int port = config_get_number ("television.vdr.svdrp.port");
    const char *host = config_get_string ("television.vdr.svdrp.host");
    int sd = svdrp_connect (host, port);
    if (!sd) {
        ho_free (filelist->error);
        filelist->error = ho_strdup (_("Connection failure..."));
        return false;
    }

    bool result = false;
    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init ("ISO-8859-1", codeset);

    if (!send_command (sd, "LSTR")) {
        goto error;
    }

    char buf[BUFSIZE];
    do {
        if (!recv_answer (sd, buf, BUFSIZE, "250")) {
            goto error;
        }

        int num = atoi (buf + 4);
        char *txt = recode (xr, index (buf + 4, ' ') + 1);

        char *name = trim_whitespace (txt + 16);

        struct tm ts;
        char *ts_str = txt;
        strptime (ts_str, "%d.%m.%y %H:%M", &ts);
        ts_str = ho_strdup_strftime ("%d.%m.%Y\t%H:%M", &ts);

        char *title = ho_strdup_printf ("%s\t%s", ts_str, name);
        char *mrl = ho_strdup_printf ("%s%d", FILELIST_VDR_RECORDINGS_MRL,
                                      num);
        fileitem_t *fileitem = filelist_add (filelist, title, mrl,
                                             FILE_TYPE_VDR_RECORDING);
        fileitem->sort_by = ho_strdup_strftime ("%F %H:%M", &ts);

        ho_free (ts_str);
        ho_free (title);
        ho_free (mrl);
        ho_free (txt);
    } while (strncmp (buf, "250 ", 4) != 0);

    filelist_sort (filelist, vdr_swap_cb);

    /* Close connection */
    if (!simple_command (sd, "QUIT", "221")) {
        goto error;
    }

    result = true;
  error:
    close (sd);
    recode_done (xr);
    ho_free (codeset);

    return result;
}


bool
vdr_recording_play (const char *mrl)
{
    if (strncmp (mrl, FILELIST_VDR_RECORDINGS_MRL,
                 strlen (FILELIST_VDR_RECORDINGS_MRL)) != 0) {
        error (_("This is not a valid VDR recordings MRL!"));
        return false;
    }

    int num = atoi (mrl + strlen (FILELIST_VDR_RECORDINGS_MRL));
    debug ("starting VDR recording %d", num);

    const int port = config_get_number ("television.vdr.svdrp.port");
    const char *host = config_get_string ("television.vdr.svdrp.host");
    int sd = svdrp_connect (host, port);
    if (!sd) {
        return false;
    }

    bool result = false;
    char command[32];
    snprintf (command, 32, "PLAY %d", num);
    if (!simple_command (sd, command, "250")) {
        goto error;
    }

    /* Pause the recording. */
    if (!simple_command (sd, "HITK Pause", "250")) {
        goto error;
    }

    /* Make sure recordings is showing. */
    if (!simple_command (sd, "HITK Recordings", "250")) {
        goto error;
    }

    /* Hide the menu. */
    if (!simple_command (sd, "HITK Menu", "250")) {
        goto error;
    }

    /* Show the position. */
    if (!simple_command (sd, "HITK Ok", "250")) {
        goto error;
    }

    /* Close connection. */
    if (!simple_command (sd, "QUIT", "221")) {
        goto error;
    }

    result = true;
  error:
    close (sd);
    return result;
}


bool
vdr_recording_remove (const char *mrl)
{
    if (!starts_with (mrl, FILELIST_VDR_RECORDINGS_MRL)) {
        error (_("This is not a valid MRL of a VDR recording!"));
        return false;
    }

    int num = atoi (mrl + strlen (FILELIST_VDR_RECORDINGS_MRL));
    debug ("removing VDR recording %d", num);

    const int port = config_get_number ("television.vdr.svdrp.port");
    const char *host = config_get_string ("television.vdr.svdrp.host");
    int sd = svdrp_connect (host, port);
    if (!sd) {
        return false;
    }

    bool result = false;
    char command[32];
    snprintf (command, 32, "DELR %d", num);
    if (!simple_command (sd, command, "250")) {
        goto error;
    }

    /* Close connection. */
    if (!simple_command (sd, "QUIT", "221")) {
        goto error;
    }

    result = true;
  error:
    close (sd);
    return result;
}


bool
vdr_timer_remove (const char *mrl)
{
    if (!starts_with (mrl, FILELIST_VDR_TIMERS_MRL)) {
        error (_("This is not a valid MRL of a VDR timer!"));
        return false;
    }

    int num = atoi (mrl + strlen (FILELIST_VDR_TIMERS_MRL));
    debug ("removing VDR timer %d", num);

    const int port = config_get_number ("television.vdr.svdrp.port");
    const char *host = config_get_string ("television.vdr.svdrp.host");
    int sd = svdrp_connect (host, port);
    if (!sd) {
        return false;
    }

    bool result = false;
    char command[32];
    snprintf (command, 32, "DELT %d", num);
    if (!simple_command (sd, command, "250")) {
        goto error;
    }

    /* Close connection. */
    if (!simple_command (sd, "QUIT", "221")) {
        goto error;
    }

    result = true;
  error:
    close (sd);
    return result;
}


#endif /* HAVE_VDR */
