//                                               -*- C++ -*-
/**
 *  @file  Multinomial.hxx
 *  @brief The Multinomial distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: Multinomial.hxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#ifndef OPENTURNS_MULTINOMIAL_HXX
#define OPENTURNS_MULTINOMIAL_HXX

#include "OTprivate.hxx"
#include "DiscreteDistribution.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Multinomial
       *
       * The Multinomial distribution.
       */
      class Multinomial
        : public Model::DiscreteDistribution
      {
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::DiscreteDistribution                          DiscreteDistribution;  // required by SWIG
        typedef DiscreteDistribution::Implementation                 Implementation;
        typedef DiscreteDistribution::NumericalPoint                 NumericalPoint;
        typedef DiscreteDistribution::NumericalSample                NumericalSample;
        typedef DiscreteDistribution::CovarianceMatrix               CovarianceMatrix;
        typedef DiscreteDistribution::NumericalPointWithDescriptionCollection       NumericalPointWithDescriptionCollection;
        typedef DiscreteDistribution::NotDefinedException            NotDefinedException;
        typedef DiscreteDistribution::StorageManager                 StorageManager;


        /** Default constructor */
        Multinomial()
          /* throw (InvalidArgumentException) */;


        /** Parameters constructor */
        Multinomial(const UnsignedLong n,
                    const NumericalPoint & p)
          /* throw (InvalidArgumentException) */;


        /** Comparison operator */
        Bool operator ==(const Multinomial & other) const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual Multinomial * clone() const;

        /** Get one realization of the distribution */
        NumericalPoint getRealization() const;

        /** Get the PDF of the distribution */
        using DiscreteDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution */
        using DiscreteDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point,
                                   const Bool tail = false) const;

        /** Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalPDF(const NumericalScalar x,
                                              const NumericalPoint & y) const;

        /** Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalCDF(const NumericalScalar x,
                                              const NumericalPoint & y) const;

        /** Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalQuantile(const NumericalScalar q,
                                                   const NumericalPoint & y) const;

        /** Get the i-th marginal distribution */
        using Model::DiscreteDistribution::getMarginal;
        Multinomial * getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */;

        /** Get the distribution of the marginal distribution corresponding to indices dimensions */
        Multinomial * getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

        /** Get the support of a discrete distribution that intersect a given interval */
        NumericalSample getSupport(const Interval & interval) const;

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;

        /* Interface specific to Multinomial */

        /** P vector accessor */
        void setP(const NumericalPoint & p)
          /* throw (InvalidArgumentException) */;
        NumericalPoint getP() const;

        /** N accessor */
        void setN(const UnsignedLong n);
        UnsignedLong getN() const;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);

      protected:


      private:

        /** Reference algorithm for the CDF */
        NumericalScalar computeCDFReferenceAlgorithm(const NumericalPoint & point,
                                                     const Bool tail = false) const;

        /** Quantile computation for dimension=1 */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const Bool tail = false,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** Compute the mean of the distribution */
        void computeMean() const;

        /** Compute the covariance of the distribution */
        void computeCovariance() const;

        /** The range of the output */
        UnsignedLong n_;

        /** The vector of probabilities of the Multinomial distribution */
        NumericalPoint p_;

        /** The sum of probabilities of the Multinomial distribution */
        NumericalScalar sumP_;


      }; /* class Multinomial */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_MULTINOMIAL_HXX */
