//                                               -*- C++ -*-
/**
 *  @file  Matrix.hxx
 *  @brief Matrix implements the classical mathematical matrix
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-06-30 10:19:34 +0200 (Thu, 30 Jun 2011) $
 *  Id:      $Id: Matrix.hxx 1972 2011-06-30 08:19:34Z schueller $
 */
#ifndef OPENTURNS_MATRIX_HXX
#define OPENTURNS_MATRIX_HXX

#include "MatrixImplementation.hxx"
#include "TypedInterfaceObject.hxx"




namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      class IdentityMatrix;
      class SymmetricMatrix;

      /**
       * @class Matrix
       *
       * Matrix implements the classical mathematical matrix
       */

      class Matrix :
        public Common::TypedInterfaceObject<MatrixImplementation>
      {
        CLASSNAME;

#ifndef SWIG
        /** Declaration of friend operators */
        friend Matrix operator * (const NumericalScalar s,
                                  const Matrix & m);
#endif


      public:

        typedef Common::InvalidDimensionException                                  InvalidDimensionException ;
        typedef Common::InvalidArgumentException                                   InvalidArgumentException ;
        typedef MatrixImplementation::NumericalComplexCollection                   NumericalComplexCollection;
        typedef Common::TypedInterfaceObject<MatrixImplementation>::Implementation Implementation ;

        /** Default constructor */
        Matrix();

        /** Constructor with implementation */
        Matrix(const Implementation & i);

        /** Constructor with implementation */
        Matrix(const MatrixImplementation & i);

        /** Constructor with size (rowDim and colDim) */
        Matrix(const UnsignedLong rowDim,
               const UnsignedLong colDim);

        /** Constructor from range of external collection */
        template <class InputIterator>
        Matrix(const UnsignedLong rowDim,
               const UnsignedLong colDim,
               const InputIterator first,
               const InputIterator last);

        /** Constructor from external collection */
        /** If the dimensions of the matrix and of the collection */
        /** do not correspond, either the collection is truncated */
        /** or the rest of the matrix is filled with zeros */
        Matrix(const UnsignedLong rowDim,
               const UnsignedLong colDim,
               const Collection<NumericalScalar> & elementsValues);


        /** String converter */
        virtual String __repr__() const;
        virtual String __str__(const String & offset = "") const;

#ifndef SWIG
        /** Operator () gives access to the elements of the matrix (to modify these elements) */
        /** The element of the matrix is designated by its row number i and its column number j */
        NumericalScalar & operator () (const UnsignedLong i,
                                       const UnsignedLong j) /* throw(InvalidDimensionException) */;

        /** Operator () gives access to the elements of the matrix (read only) */
        /** The element of the matrix is designated by its row number i and its column number j */
        const NumericalScalar & operator () (const UnsignedLong i,
                                             const UnsignedLong j) const /* throw(InvalidDimensionException) */;
#endif

        /** Get the dimensions of the matrix */
        /** Number of rows */
        const UnsignedLong getNbRows() const ;
        /** Number of columns */
        const UnsignedLong getNbColumns() const ;

        /** Matrix transpose */
        Matrix transpose () const ;

        /** Matrix additions (must have the same dimensions) */
        Matrix operator + (const Matrix & m) const /* throw(InvalidDimensionException) */;
        Matrix operator + (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */;

        /** Matrix substractions (must have the same dimensions) */
        Matrix operator - (const Matrix & m) const /* throw(InvalidDimensionException) */;
        Matrix operator - (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */;

        /** Matrix multiplications (must have consistent dimensions) */
        Matrix operator * (const Matrix & m) const /* throw(InvalidDimensionException) */;
        Matrix operator * (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */;
        Matrix operator * (const IdentityMatrix & m) const /* throw(InvalidDimensionException) */;

        /** Multiplication with a NumericalPoint (must have consistent dimensions) */
        NumericalPoint operator * (const NumericalPoint & pt) const /* throw(InvalidDimensionException) */;

        /** Multiplication with a NumericalScalar */
        Matrix operator * (const NumericalScalar s) const ;

        /** Division by a NumericalScalar*/
        Matrix operator / (const NumericalScalar s) const /* throw(InvalidArgumentException) */;

        /** Resolution of a linear system */
        NumericalPoint solveLinearSystem(const NumericalPoint & b,
					 const Bool keepIntact = true);
        Matrix solveLinearSystem(const Matrix & b,
				 const Bool keepIntact = true);

        /** Compute singular values */
        NumericalPoint computeSingularValues(const Bool keepIntact = true);

        NumericalPoint computeSingularValues(Matrix & u,
					     Matrix & vT,
					     const Bool fullSVD = false,
					     const Bool keepIntact = true);

        /** Comparison operators */
        Bool operator == (const Matrix & rhs) const ;

        /** Empty returns true if there is no element in the matrix */
        const Bool isEmpty() const;

      }; /* class Matrix */

      /** Declaration of friend operators */
      inline Matrix operator * (const NumericalScalar s,
                                const Matrix & m)
      {
        return m.operator*(s);
      }


      /** Constructor from range of external collection */
      template <class InputIterator>
      Matrix::Matrix(const UnsignedLong rowDim,
                     const UnsignedLong colDim,
                     const InputIterator first,
                     const InputIterator last)
        : Common::TypedInterfaceObject<MatrixImplementation>(new MatrixImplementation(rowDim, colDim, first, last))
      {
        // Nothing to do
      }

    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_MATRIX_HXX */
