//                                               -*- C++ -*-
/**
 *  @file  Staircase.cxx
 *  @brief Staircase class for staircase plots
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: Staircase.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include "Staircase.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Graph {

      CLASSNAMEINIT(Staircase);

      static Common::Factory<Staircase> RegisteredFactory("Staircase");

      /* Default constructor */
      Staircase::Staircase():
        DrawableImplementation(),
        pattern_()
      {
        // Nothing to do
      }

      /* Constructor with parameters */
      Staircase::Staircase(const NumericalSample & data,
                           const String & legend):
        DrawableImplementation(data, legend),
        pattern_(DefaultPattern)
      {
        // Check data validity
        setData(data);
      }

      /* Constructor with parameters */
      Staircase::Staircase(const NumericalSample & data,
                           const String & color,
                           const String & lineStyle,
                           const UnsignedLong lineWidth,
                           const String & pattern,
                           const String & legend)
        /* throw(InvalidArgumentException) */
        : DrawableImplementation(data, legend)
      {
        setColor(color);
        setLineStyle(lineStyle);
        setPattern(pattern);
        setLineWidth(lineWidth);
        // Check data validity
        setData(data);
      }

      /* Constructor with old parameters */
      Staircase::Staircase(const NumericalSample & data,
                           const String & color,
                           const String & lineStyle,
                           const String & pattern,
                           const String & legend)
        /* throw(InvalidArgumentException) */
        : DrawableImplementation(data, legend)
      {
        setColor(color);
        setLineStyle(lineStyle);
        setPattern(pattern);
        setLineWidth(1);
        // Check data validity
        setData(data);
      }

      /* String converter */
      String Staircase::__repr__() const
      {
        OSS oss;
        oss << "class=" << Staircase::GetClassName()
            << " name=" << getName()
            << " pattern=" << pattern_
            << " derived from " << DrawableImplementation::__repr__();
        return oss;
      }

      /* Draw method */
      String Staircase::draw() const
      {
        OSS oss;
        // Stores the data in a temporary file
        oss << DrawableImplementation::draw() << "\n";
        // The specific R command for drawing
        oss << "lines(data[, 1], data[, 2]"
            << ", lty=\"" << lineStyle_
            << "\", col=\"" << color_
            << "\", lwd=" << lineWidth_
            << ", type=\"" << pattern_
            << "\")";

        return oss;
      }

      /* Clone method */
      Staircase * Staircase::clone() const
      {
        return new Staircase(*this);
      }

      /* Accessor for pattern */
      String Staircase::getPattern() const
      {
        return pattern_;
      }

      /* Accessor for pattern */
      void Staircase::setPattern(const String style) /* throw(InvalidArgumentException) */
      {
        if(!isValidPattern(style)) throw InvalidArgumentException(HERE) << "Given staircase pattern = " << style << " is incorrect";
        pattern_ = style;
      }

      /* Check validity of data */
      Bool Staircase::isValidData(const NumericalSample & data) const
      {
        return (data.getDimension() == 2);
      }

      /* Method save() stores the object through the StorageManager */
      void Staircase::save(StorageManager::Advocate & adv) const
      {
        DrawableImplementation::save(adv);
        adv.saveAttribute( "pattern_", pattern_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void Staircase::load(StorageManager::Advocate & adv)
      {
        DrawableImplementation::load(adv);
        adv.loadAttribute( "pattern_", pattern_ );
      }

    } /* namespace Graph */

  }/* namespace Base */

}/* namespace OpenTURNS */
