//                                               -*- C++ -*-
/**
 *  @file  Contour.cxx
 *  @brief Contour class for contourchart plots
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (Tue, 24 May 2011) $
 *  Id:      $Id: Contour.cxx 1910 2011-05-24 17:30:41Z schueller $
 */
#include <cstdio>
#include <cstdlib>

#include "Contour.hxx"
#include "PersistentObjectFactory.hxx"
#include "Log.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Graph {

      CLASSNAMEINIT(Contour);

      typedef Common::Log         Log;
      typedef Common::ResourceMap ResourceMap;

      const UnsignedLong Contour::DefaultLevelsNumber = ResourceMap::GetAsUnsignedLong( "Contour-DefaultLevelsNumber" );

      static Common::Factory<Contour> RegisteredFactory("Contour");

      /* Default constructor */
      Contour::Contour(const UnsignedLong dimX,
                       const UnsignedLong dimY,
                       const NumericalSample & data,
                       const String & legend)
        /* throw(InvalidArgumentException) */
        : DrawableImplementation(data, legend),
          x_(NumericalSample(dimX, 1)),
          y_(NumericalSample(dimY, 1)),
          levels_(NumericalPoint(DefaultLevelsNumber)),
          labels_(DefaultLevelsNumber),
          drawLabels_(true)
      {
        if (dimX < 2) throw InvalidArgumentException(HERE) << "Error: the x dimension must be greater or equal to 2";
        if (dimY < 2) throw InvalidArgumentException(HERE) << "Error: the y dimension must be greater or equal to 2";
        if (dimX * dimY != data.getSize()) throw InvalidArgumentException(HERE) << "Error: the given dimensions are not compatible with the data";
        // Check data validity
        setData(data);
        // By default, x is assumed to be equally spaced in [0, 1]
        for (UnsignedLong i = 0; i < dimX; ++i) x_[i][0] = NumericalScalar(i) / (dimX - 1.0);
        // By default, y is assumed to be equally spaced in [0, 1]
        for (UnsignedLong i = 0; i < dimY; ++i) y_[i][0] = NumericalScalar(i) / (dimY - 1.0);
        // Build the levels
        buildDefaultLevels();
        // Build the labels
        buildDefaultLabels();
      }

      /* Constructor with parameters */
      Contour::Contour(const NumericalSample & x,
                       const NumericalSample & y,
                       const NumericalSample & data,
                       const NumericalPoint & levels,
                       const Description & labels,
                       const Bool drawLabels,
                       const String & legend)
        /* throw(InvalidArgumentException) */
        : DrawableImplementation(data, legend),
          x_(x),
          y_(y),
          levels_(levels),
          labels_(labels),
          drawLabels_(drawLabels)
      {
        if (levels.getDimension() == 0) buildDefaultLevels();
        if (drawLabels && (labels.getSize() == 0)) buildDefaultLabels();
        if (drawLabels && (levels.getDimension() > 0) && (labels.getSize() > 0) && (levels.getDimension() != labels.getSize())) throw InvalidArgumentException(HERE) << "Error: the levels are incompatible with the labels";
        // Check data validity
        setData(data);
      }

      /* String converter */
      String Contour::__repr__() const
      {
        OSS oss;
        oss << "class=" << Contour::GetClassName()
            << " name=" << getName()
            << " x=" << x_
            << " y=" << y_
            << " levels=" << levels_
            << " labels=" << labels_
            << " show labels=" << drawLabels_
            << " derived from " << DrawableImplementation::__repr__();
        return oss;
      }

      /* Accessor for first coordinate */
      Contour::NumericalSample Contour::getX() const
      {
        return x_;
      }

      void Contour::setX(const NumericalSample & x)
      {
        x_ = x;
      }

      /* Accessor for second coordinate */
      Contour::NumericalSample Contour::getY() const
      {
        return y_;
      }

      void Contour::setY(const NumericalSample & y)
      {
        y_ = y;
      }

      /* Accessor for levels */
      Contour::NumericalPoint Contour::getLevels() const
      {
        return levels_;
      }

      void Contour::setLevels(const NumericalPoint & levels)
      {
        levels_ = levels;
        if (levels.getDimension() != labels_.getSize()) buildDefaultLabels();
      }

      /* Accessor for labels */
      Contour::Description Contour::getLabels() const
      {
        return labels_;
      }

      void Contour::setLabels(const Description & labels)
      /* throw(InvalidArgumentException) */
      {
        if (labels.getSize() != levels_.getDimension()) throw InvalidArgumentException(HERE) << "Error: the labels size must be equal to the levels dimension";
        labels_ = labels;
      }

      /* Accessor for drawLabels */
      Bool Contour::getDrawLabels() const
      {
        return drawLabels_;
      }

      void Contour::setDrawLabels(const Bool & drawLabels)
      {
        drawLabels_ = drawLabels;
      }

      /* Accessor for boundingbox */
      Contour::BoundingBox Contour::getBoundingBox() const
      {
        BoundingBox boundingBox(BoundingBoxSize);
        boundingBox[0] = x_.getMin()[0];
        boundingBox[1] = x_.getMax()[0];
        boundingBox[2] = y_.getMin()[0];
        boundingBox[3] = y_.getMax()[0];
        return boundingBox;
      }

      /* Clean all the temporary data created by draw() method */
      void Contour::clean() const
      {
        DrawableImplementation::clean();
        if ((xFileName_ != "") && (remove(xFileName_.c_str()) == -1)) LOGWARN(OSS() << "GraphImplementation: error trying to remove file " << xFileName_);
        if ((yFileName_ != "") && (remove(yFileName_.c_str()) == -1)) LOGWARN(OSS() << "GraphImplementation: error trying to remove file " << yFileName_);
      }

      /* Draw method */
      String Contour::draw() const
      {
        xFileName_ = "";
        yFileName_ = "";
        OSS oss;
        // Stores the data in a temporary file
        // For a contour, it is a matrix of values
        oss << DrawableImplementation::draw() << "\n";
        // The specific R command for drawing
        // Here we store the discretizations of the axes
        if (x_.getDimension() * x_.getSize() > DataThreshold)
          {
            xFileName_ = x_.storeToTemporaryFile();
            yFileName_ = y_.storeToTemporaryFile();
            oss << "x <- data.matrix(read.table(\"" << xFileName_ << "\"))\n"
                << "y <- data.matrix(read.table(\"" << yFileName_ << "\"))\n";
          }
        else
          {
            oss << "x <- " << x_.streamToRFormat() << "\n"
                << "y <- " << y_.streamToRFormat() << "\n";
          }
        oss << "data <- matrix(data, length(x), length(y))\n"
            << "levels=c(";
        const UnsignedLong length(levels_.getSize() - 1);
        for(UnsignedLong i = 0; i < length; ++i)
          {
            oss << levels_[i] << ",";
          }
        oss << levels_[length] << ")\n"
            << "labels=c(\"";
        for(UnsignedLong i = 0; i < length; ++i)
          {
            oss << labels_[i] << "\",\"";
          }
        oss << labels_[length] << "\")\n";
        oss << "contour(x, y, data, "
            << "levels=levels,"
            << "labels=labels,"
            << "drawlabels=" << (drawLabels_? "TRUE":"FALSE") << ","
            << "lty=\"" << lineStyle_ << "\","
            << "col=\"" << color_ << "\","
            << "lwd=" << lineWidth_
            << ", add=TRUE,axes=FALSE)";
        return oss;
      }

      /* Clone method */
      Contour * Contour::clone() const
      {
        return new Contour(*this);
      }

      /* Check for data validity */
      Bool Contour::isValidData(const NumericalSample & data) const
      {
        return (data.getDimension() == 1);
      }

      /* Build default levels by sampling uniformly the range [min(data), max(data)] into DefaultLevelsNumber points
         starting from min(data) + 0.5 * (max(data) - min(data)) / DefaultLevelsNumber
      */
      void Contour::buildDefaultLevels(const UnsignedLong number)
      {
        const NumericalScalar min(data_.getMin()[0]);
        const NumericalScalar max(data_.getMax()[0]);
        levels_ = NumericalPoint(number);
        for (UnsignedLong i = 0; i < number; ++i)
          {
            levels_[i] = min + (max - min) * (i + 0.5) / number;
          }
      }

      /* Build default labels by taking the level values */
      void Contour::buildDefaultLabels()
      {
        const UnsignedLong number(levels_.getDimension());
        labels_ = Description(number);
        for (UnsignedLong i = 0; i < number; ++i)
          {
            labels_[i] = OSS() << levels_[i];
          }
      }

      /* Method save() stores the object through the StorageManager */
      void Contour::save(StorageManager::Advocate & adv) const
      {
        DrawableImplementation::save(adv);
        adv.saveAttribute( "x_", x_ );
        adv.saveAttribute( "y_", y_ );
        adv.saveAttribute( "levels_", levels_ );
        adv.saveAttribute( "labels_", labels_ );
        adv.saveAttribute( "drawLabels_", drawLabels_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void Contour::load(StorageManager::Advocate & adv)
      {
        DrawableImplementation::load(adv);
        adv.loadAttribute( "x_", x_ );
        adv.loadAttribute( "y_", y_ );
        adv.loadAttribute( "levels_", levels_ );
        adv.loadAttribute( "labels_", labels_ );
        adv.loadAttribute( "drawLabels_", drawLabels_ );
      }

    }/* namespace Graph */

  }/*namespace Base */

}/* namespace OpenTURNS */
