//                                               -*- C++ -*-
/**
 *  @file  WeibullFactory.cxx
 *  @brief Factory for Weibull distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: WeibullFactory.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "WeibullFactory.hxx"
#include "Weibull.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Model::DistributionImplementationFactory::NumericalPoint NumericalPoint;

      /* Default constructor */
      WeibullFactory::WeibullFactory():
	DistributionImplementationFactory()
      {
	// Nothing to do
      }

      /* Virtual constructor */
      WeibullFactory * WeibullFactory::clone() const
      {
	return new WeibullFactory(*this);
      }

      /* Here is the interface that all derived class must implement */

      WeibullFactory::DistributionImplementation * WeibullFactory::buildImplementation(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Weibull distribution from an empty sample";
	if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Weibull distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
	NumericalScalar gamma(sample.getMin()[0]);
	// Correction for bias control
	const NumericalScalar size(sample.getSize());
	if (gamma > 0.0)
	  {
	    gamma *= 1.0 - 1.0 / (2.0 + size);
	  }
	else
	  {
	    gamma *= 1.0 + 1.0 / (2.0 + size);
	  }
	const NumericalScalar mean(sample.computeMean()[0]);
	const NumericalScalar sigma(sample.computeStandardDeviationPerComponent()[0]);
	try {
	  return Weibull(mean, sigma, gamma, Weibull::MUSIGMA).clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot estimate parameters of a Weibull distribution from the given sample";
	  }
      }

      WeibullFactory::DistributionImplementation * WeibullFactory::buildImplementation(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
	try {
	  Weibull distribution;
	  distribution.setParametersCollection(parameters);
	  return distribution.clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot build a Weibull distribution from the given parameters";
	  }
      }

      WeibullFactory::DistributionImplementation * WeibullFactory::buildImplementation() const /* throw(InvalidArgumentException, InternalException) */
      {
	return Weibull().clone();
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
