//                                               -*- C++ -*-
/**
 *  @file  OrthogonalUniVariatePolynomial.hxx
 *  @brief This is an orthogonal 1D polynomial
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 17:44:02 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.hxx 818 2008-05-21 15:44:02Z dutka $
 */
#ifndef OPENTURNS_ORTHOGONALUNIVARIATEPOLYNOMIAL_HXX
#define OPENTURNS_ORTHOGONALUNIVARIATEPOLYNOMIAL_HXX

#include "UniVariatePolynomialImplementation.hxx"
#include "Collection.hxx"
#include "PersistentCollection.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      /**
       * @class OrthogonalUniVariatePolynomial
       *
       * This is an orthogonal 1D polynomial. The polynomial P0 is constant equal to 1.0, and by convention we note P-1(x) the null polynomial. For n>=1 we have: Pn(x) = (an * x + bn) * Pn-1(x) + cn * Pn-2(x)
       */

      class OrthogonalUniVariatePolynomial
        : public Base::Func::UniVariatePolynomialImplementation
      {
        CLASSNAME;
      public:

        typedef Base::Type::Collection<Coefficients>           CoefficientsCollection;
        typedef Base::Type::PersistentCollection<Coefficients> CoefficientsPersistentCollection;

        /** Default constructor */
        OrthogonalUniVariatePolynomial();

        /** Constructor from coefficients */
        OrthogonalUniVariatePolynomial(const CoefficientsCollection & recurrenceCoefficients);

        /** Virtual constructor */
        virtual OrthogonalUniVariatePolynomial * clone() const;

        /** OrthogonalUniVariatePolynomial are evaluated as functors */
        NumericalScalar operator() (const NumericalScalar x) const;

        /** Recurrence coefficients accessor */
        CoefficientsCollection getRecurrenceCoefficients() const;

        /** Roots of the polynomial of degree n as the eigenvalues of the associated Jacobi matrix */
        NumericalComplexCollection getRoots() const;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);

      protected:

	friend class OrthogonalUniVariatePolynomialFactory;

	/** Constructor from recurrence coefficients and coefficients. It is protected to prevent the end user to give incoherent coefficients. */
        OrthogonalUniVariatePolynomial(const CoefficientsCollection & recurrenceCoefficients,
				       const Coefficients & coefficients);

      private:

        /** Build the coefficients of the polynomial based on the recurrence coefficients */
        Coefficients buildCoefficients(const UnsignedLong n);

        /** The recurrence coefficients (an, bn, cn) that defines the orthogonal polynomial for n >= 1. The polynomial P0 is constant equal to 1.0, and by convention we note P-1(x) the null polynomial. For n>=2 we have: Pn(x) = (an * x + bn) * Pn-1(x) + cn * Pn-2(x). The recurrence coefficients are stored starting with (a1, b1, c1). */
        CoefficientsPersistentCollection recurrenceCoefficients_;

      } ; /* class OrthogonalUniVariatePolynomial */


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_ORTHOGONALUNIVARIATEPOLYNOMIAL_HXX */
