//                                               -*- C++ -*-
/**
 *  @file  ProjectionStrategyImplementation.cxx
 *  @brief This is a abstract class for projection strategy implementations
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "ProjectionStrategyImplementation.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "MonteCarloExperiment.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(ProjectionStrategyImplementation);

      static Base::Common::Factory<ProjectionStrategyImplementation> RegisteredFactory("ProjectionStrategyImplementation");

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;

      /* Default constructor */
      ProjectionStrategyImplementation::ProjectionStrategyImplementation()
	: Base::Common::PersistentObject(),
	  alpha_k_p_(0),
	  residual_p_(0.0),
	  measure_(),
	  p_weightedExperiment_(MonteCarloExperiment().clone()),
	  inputSample_(0, 0),
	  weights_(0),
	  outputSample_(0, 0)
      {
	// The ProjectionStrategyImplementation imposes its distribution to the weighted experiment
	p_weightedExperiment_->setDistribution(getMeasure());
      }


      /* Parameter constructor */
      ProjectionStrategyImplementation::ProjectionStrategyImplementation(const Distribution & measure)
	: Base::Common::PersistentObject(),
	  alpha_k_p_(0),
	  residual_p_(0.0),
	  measure_(measure),
	  p_weightedExperiment_(MonteCarloExperiment().clone()),
	  inputSample_(0, 0),
	  weights_(0),
	  outputSample_(0, 0)
      {
	// The ProjectionStrategyImplementation imposes the distribution of the weighted experiment
	p_weightedExperiment_->setDistribution(getMeasure());
      }

      
      /* Parameter constructor */
      ProjectionStrategyImplementation::ProjectionStrategyImplementation(const WeightedExperiment & weightedExperiment)
	: Base::Common::PersistentObject(),
	  alpha_k_p_(0),
	  residual_p_(0.0),
	  measure_(weightedExperiment.getDistribution()),
	  p_weightedExperiment_(weightedExperiment.clone()),
	  inputSample_(0, 0),
	  weights_(0),
	  outputSample_(0, 0)

      {
	// Nothing to do
      }

      
      /* Parameter constructor */
      ProjectionStrategyImplementation::ProjectionStrategyImplementation(const Distribution & measure,
									 const WeightedExperiment & weightedExperiment)
	: Base::Common::PersistentObject(),
	  alpha_k_p_(0),
	  residual_p_(0.0),
	  measure_(measure),
	  p_weightedExperiment_(weightedExperiment.clone()),
	  inputSample_(0, 0),
	  weights_(0),
	  outputSample_(0, 0)
      {
	// The ProjectionStrategyImplementation imposes the distribution of the weighted experiment
	p_weightedExperiment_->setDistribution(getMeasure());
      }

      
      /* Virtual constructor */
      ProjectionStrategyImplementation * ProjectionStrategyImplementation::clone() const
      {
	return new ProjectionStrategyImplementation(*this);
      }


      /* String converter */
      String ProjectionStrategyImplementation::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " measure=" << measure_;
      }


      /* Measure accessor */
      void ProjectionStrategyImplementation::setMeasure(const Distribution & measure)
      {
	measure_ = measure;
	// Set the measure as the distribution of the weighted experiment
	p_weightedExperiment_->setDistribution(measure);
      }

      ProjectionStrategyImplementation::Distribution ProjectionStrategyImplementation::getMeasure() const
      {
	return measure_;
      }

      /* Experiment accessors */
      void ProjectionStrategyImplementation::setExperiment(const WeightedExperiment & weightedExperiment)
      {
	p_weightedExperiment_ = weightedExperiment.clone();
	p_weightedExperiment_->setDistribution(getMeasure());
      }

      WeightedExperiment ProjectionStrategyImplementation::getExperiment() const
      {
	return *p_weightedExperiment_;
      }


      /* Compute the components alpha_k_p_ by projecting the model on the partial L2 basis */
      void ProjectionStrategyImplementation::computeCoefficients(const NumericalMathFunction & function,
								 const NumericalMathFunctionCollection & partialBasis,
								 const Indices & indices,
								 const Indices & addedRanks,
								 const Indices & conservedRanks,
								 const Indices & removedRanks)
      {
	throw NotYetImplementedException(HERE);
      }


      /* Method save() stores the object through the StorageManager */
      void ProjectionStrategyImplementation::save(StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void ProjectionStrategyImplementation::load(StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
