/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLAuthorityBinding.cpp - SAML authority binding implementation

   Scott Cantor
   5/9/02

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;


SAMLAuthorityBinding::SAMLAuthorityBinding(const saml::QName* kind, const XMLCh* binding, const XMLCh* location)
{
    RTTI(SAMLAuthorityBinding);
    m_kind=new saml::QName(*kind);
    m_binding=XML::assign(binding);
    m_location=XML::assign(location);
}

SAMLAuthorityBinding::SAMLAuthorityBinding(DOMElement* e) : m_kind(NULL), m_binding(NULL), m_location(NULL)
{
    RTTI(SAMLAuthorityBinding);
    fromDOM(e);
}

SAMLAuthorityBinding::SAMLAuthorityBinding(istream& in) : SAMLObject(in), m_kind(NULL), m_binding(NULL), m_location(NULL)
{
    RTTI(SAMLAuthorityBinding);
    fromDOM(m_document->getDocumentElement());
}

SAMLAuthorityBinding::~SAMLAuthorityBinding()
{
    if (m_bOwnStrings) {
        XMLString::release(&m_binding);
        XMLString::release(&m_location);
    }
    delete m_kind;
}

void SAMLAuthorityBinding::ownStrings()
{
    if (!m_bOwnStrings) {
        m_binding=XML::assign(m_binding);
        m_location=XML::assign(m_location);
        m_bOwnStrings=true;
    }
}

void SAMLAuthorityBinding::fromDOM(DOMElement* e)
{
    SAMLObject::fromDOM(e);

    if (SAMLConfig::getConfig().strict_dom_checking && !XML::isElementNamed(e,XML::SAML_NS,L(AuthorityBinding)))
        throw MalformedException("SAMLAuthorityBinding::fromDOM() requires saml:AuthorityBinding at root");

    m_kind=saml::QName::getQNameAttribute(e,NULL,L(AuthorityKind));
    m_binding = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(Binding)));
    m_location = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(Location)));
    
    checkValidity();
}

void SAMLAuthorityBinding::setAuthorityKind(saml::QName& kind)
{
    delete m_kind;
    m_kind = NULL;
    m_kind=new saml::QName(kind);
    ownStrings();
    setDirty();
}

void SAMLAuthorityBinding::setBinding(const XMLCh* binding)
{
    if (XML::isEmpty(binding))
        throw MalformedException("binding cannot be null or empty");
    
    if (m_bOwnStrings)
        XMLString::release(&m_binding);
    else {
        m_binding=NULL;
        ownStrings();
    }
    m_binding=XML::assign(binding);
    setDirty();
}

void SAMLAuthorityBinding::setLocation(const XMLCh* location)
{
    if (XML::isEmpty(location))
        throw MalformedException("location cannot be null or empty");
    
    if (m_bOwnStrings)
        XMLString::release(&m_location);
    else {
        m_location=NULL;
        ownStrings();
    }
    m_location=XML::assign(location);
    setDirty();
}

DOMElement* SAMLAuthorityBinding::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* ab = doc->createElementNS(XML::SAML_NS, L(AuthorityBinding));
    if (xmlns)
        ab->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAML_NS);
    return ab;
}
    
DOMNode* SAMLAuthorityBinding::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc,xmlns);
    DOMElement* ab=static_cast<DOMElement*>(m_root);
    
    if (m_bDirty) {
        ab->setAttributeNS(NULL,L(Binding),m_binding);
        ab->setAttributeNS(NULL,L(Location),m_location);
    
        XMLCh* kind=new XMLCh[6 + XMLString::stringLen(m_kind->getLocalName()) + 1];
        if (XMLString::compareString(XML::SAMLP_NS,m_kind->getNamespaceURI())) {
            ab->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,kind),m_kind->getNamespaceURI());
            kind[0]=chLatin_k;
            kind[1]=chLatin_i;
            kind[2]=chLatin_n;
            kind[3]=chLatin_d;
            kind[4]=chColon;
            kind[5]=chNull;
        }
        else {
            kind[0]=chLatin_s;
            kind[1]=chLatin_a;
            kind[2]=chLatin_m;
            kind[3]=chLatin_l;
            kind[4]=chLatin_p;
            kind[5]=chColon;
            kind[6]=chNull;
        }
        XMLString::catString(kind,m_kind->getLocalName());
        ab->setAttributeNS(NULL,L(AuthorityKind),kind);
        delete[] kind;
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(ab,XML::SAML_NS);
    }
    return m_root;
}

void SAMLAuthorityBinding::checkValidity() const
{
    if (XML::isEmpty(m_binding) || XML::isEmpty(m_location) || !m_kind)
        throw MalformedException("AuthorityBinding is invalid, must have Binding, Location, and AuthorityKind");
}

SAMLObject* SAMLAuthorityBinding::clone() const
{
    return new SAMLAuthorityBinding(m_kind,m_binding,m_location);
}
