/* $Id: nws_sensor.c,v 1.197 2005/06/06 19:48:19 graziano Exp $ */

#include "config_nws.h"
#include <unistd.h>           /* getopt() */

#include "nws_sensor.h"
#include "dnsutil.h"
#include "clique_protocol.h"
#include "protocol.h"
#include "host_protocol.h"
#include "diagnostic.h"
#include "messages.h"
#include "osutil.h"
#include "periodic.h"
#include "skills.h"
#include "strutil.h"
#include "nws_sensor.h"
#include "register.h"
#include "nws_daemon.h"

/*
 * handle SIGTERM: remove the pid and call NwsSensorExit
 */
static int
MyExit() {
	NwsSensorExit();

	return 1;
}


/*
 * try to find the leader for clique named #cliqueName#. Put the host in
 * #cookie#. Returns 1 on success, -1 if there is no such a clique
 * registered with the nameserver and 0 otherwise.
 */
static int
FindLeader(	const char *cliqueName,
		struct host_cookie *leader) {
	char *filter;
	ObjectSet retrieved = NULL;
	Object clique = NO_OBJECT;
	int ret = 1;

	/* sanity check */
	if (leader == NULL || cliqueName == NULL) {
		ERROR("FindLeader: NULL parameters\n");
		return 0;
	}

	/* get the registrations */
	filter = (char *) MALLOC(strlen(cliqueName) + 8, 0);
	if (filter == NULL) {
		ERROR("FindLeader: out of memory\n");
		return 0;
	}
	sprintf(filter, "(name=%s)", cliqueName);
	if (!RetrieveFromMyNameserver(filter, &retrieved)) {
		FREE(filter);
		ERROR("FindLeader: error talking with nameserver\n");
		return 0;
	}
	FREE(filter);

	/* did we get the clique? */
	for (clique = NextObject(retrieved, NULL); clique != NULL; clique = NextObject(retrieved, clique)) {
		filter = NwsAttributeValue_r(FindNwsAttribute(clique, "controlName"));
		if (filter != NULL && strcmp(filter, "clique") == 0) {
			/* found the clique */
			break;
		}
	}
	FREE(filter);
	if (clique == NULL) {
		WARN1("FindLeader: couldn't find clique %s\n", cliqueName);
		return -1;
	}

	/* we got the clique: let's get the leader */
	filter = NwsAttributeValue_r(FindNwsAttribute(clique, "host"));
	if (filter == NULL) {
		ERROR1("FindLeader: clique registration bogus %s\n", clique);
		ret = 0;
	} else {
		/* fill the cookie */
		Host2Cookie(filter, DefaultHostPort(SENSOR_HOST), leader);
	}
	FreeObject(&retrieved);
	FREE(filter);

	return 1;

}

#define NEVER 0.0

void usage() {
	DaemonUsage("nws_sensor", "Sensor");
	printf("\t-c [yes|no]         start CpuMonitor at startup (default yes)\n");
	printf("\t-m [yes|no]         start MemoryMonitor at startup (default no)\n");
	printf("\t-A [yes|no]         start Availability Monitor at startup (default no)\n");
	printf("\t-s [yes|no]         start memorySpeedMonitor at startup (default no)\n");
	printf("\t-f                  don't fork (experimental)\n");
	printf("\t-x port             use port when doing network experiment (experimental)\n");
	printf("\t-j clique           join this clique\n");
	printf("\t-J                  join the automatic clique\n");
}


int
main(		int argc,
		char *argv[]) {
	char opts[255+1];
	int opt, ind, tmp, forceClique;
	char cliqueName[MAX_CLIQUE_NAME_SIZE];
	double nextBeatTime,
		nextWorkTime,
		wakeup,
		now;
	extern char *optarg;
	int skillToStart[SKILL_COUNT];

	/* Set up default values */
	for (tmp = 0; tmp < SKILL_COUNT; tmp++) {
		skillToStart[tmp] = 0;
	}
	skillToStart[cpuMonitor] = 1;
	cliqueName[0] = opts[0] = '\0';

	/* no error messages from getopt() */
	opterr = 0;
	forceClique = 0;
	ind = InitializeDaemonStructure(SENSOR_HOST, "c:m:A:s:fx:j:F", MyExit);

	while((opt = getopt(argc, argv, DaemonAllSwitches(ind))) != EOF) {
		if (DaemonSwitch(opt, optarg, ind)) {
			continue;
		}

		switch(opt) {
		case 'A':
			skillToStart[availabilityMonitor] = (*optarg == 'y') || (*optarg == 'Y');
			break;

		case 's':
			skillToStart[memorySpeedMonitor] = (*optarg == 'y') || (*optarg == 'Y');
			break;

		case 'm':
			skillToStart[memoryMonitor] = (*optarg == 'y') || (*optarg == 'Y');
			break;

		case 'c':
			skillToStart[cpuMonitor] = (*optarg == 'y') || (*optarg == 'Y');
			break;

		case 'f':
			tmp = strlen(opts);
			if (tmp == 0) {
				snprintf(opts + tmp, 255 - tmp, "fork:no");
			} else {
				snprintf(opts + tmp, 255 - tmp, "\tfork:no");
			}
			break;

		case 'j':
			SAFESTRCPY(cliqueName, optarg);
			break;

		case 'F':
			forceClique = 1;
			break;

		case 'x':
			tmp = strlen(opts);
			if (tmp == 0) {
				snprintf(opts + tmp, 255 - tmp, "forceport:%d", (unsigned short)atol(optarg));
			} else {
				snprintf(opts + tmp, 255 - tmp, "\tforceport:%d", (unsigned short)atol(optarg));
			}
			break;

		default:
			usage();
			exit(1);
			break;

		}
	}

	/* let's get ready to listen */
	if (!InitializeDaemon(0, 1, 1, ind)) {
		exit(-1);
	}

	/* initialize the sensor */
	if (!NwsSensorInit(opts)) {
		WARN("Sensor: failed to init the sensors\n");
	}

	for (tmp = 0; tmp < SKILL_COUNT; tmp++) {
		if (!skillToStart[tmp]) {
			continue;
		}
		/* start the activity with the default name */
		if(!StartPeriodicActivity(NULL, SkillName(tmp), "")) {
			WARN1("StartSensor: auto-start of %s failed\n", SkillName(tmp));
		}
	}

	/* have we been asked to join a clique? */
	if (cliqueName[0] != '\0') {
		struct host_cookie k;
		char options[128];

		/* yep let's join the clique: first of all find the
		 * leader of the clique */
		switch (FindLeader(cliqueName, &k)) {
		case 1:
			/* then ask it to join the club */
			CliqueJoin(&k, cliqueName, EstablishedRegistration(), -1);
			break;
		case -1:
			if (!forceClique) {
				WARN1("StartSensor: clique %s non existent or leader unreachable\n", cliqueName);
				break;
			}

			/* there is no such a clique: we start one */
			INFO1("StartSensor: clique %s non existent, so I'm starting it right now\n", cliqueName);
			snprintf(options, 128, "member:%s", EstablishedRegistration());
			if (StartCliqueActivity(cliqueName,
					SkillName(tcpMessageMonitor),
					options)) {
			} else {
				WARN1("StartSensor: failed to start %s\n", cliqueName);
			}
			break;

		default:
			/* some errors */
			break;
		}
	}

	nextBeatTime = 0;

	while(1) {
		/* time to register again with the name server */
		now = CurrentTime();
		if(now >= nextBeatTime) {
			if (RegisterHost(DEFAULT_HOST_BEAT * 2) == 0) {
				INFO("nws_sensor: couldn't register\n");
			}
			nextBeatTime = now + (HostHealthy() ? DEFAULT_HOST_BEAT : SHORT_HOST_BEAT);
		}

		/* let's see when we need to wake up next */
		nextWorkTime = NwsSensorNextWork();
		if((nextWorkTime != NEVER) && (nextWorkTime < nextBeatTime)) {
			/* if now > wakeup ListenForMessage will just
			 * poll for messages. */
			wakeup = nextWorkTime - now;
		} else {
			wakeup = nextBeatTime - now;
		}

		ListenForMessages((wakeup > 0) ? wakeup : -1);

		/* time to do some work? NwsSensorWork will check if it's
		 * the right time to do work */
		NwsSensorWork();
	}
	return 0; /* never reached */
}
