/* $Id: autoFetch.c,v 1.3 2005/05/24 19:38:25 graziano Exp $ */

#include "config_nws.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "diagnostic.h"
#include "host_protocol.h"
#include "messages.h"
#include "strutil.h"
#include "dnsutil.h"
#include "nws_memory.h"
#include "register.h"
#include "osutil.h"
#include "messages.h"
#include "nws_api.h"

#include "nws_forecast_api.h"

/* module lock */
static void *lock = NULL;

extern int NwsGetSeriesMemory(	const char *seriesName,
				struct host_cookie *memory,
				int timeout);

/*
 * Information cached for auto-fetch series that we're tracking.  A connection
 * to the memory that holds the series and the (tab-deliminted) list of series
 * names that this memory is sending to us. #lostSeries# contains the
 * list of series that have been disconnected.
 */
typedef struct {
	struct host_cookie memory;
	registrations *series;
} AutoFetchInfo;
static registrations *lostSeries = NULL;


/*
 * Module globals.  #autoFetches# holds a list of memory connections that we're
 * using to track auto-fetched series, and #autoFetchCount# is the length of
 * this list.  
 */
static AutoFetchInfo *autoFetches = NULL;
static unsigned int autoFetchCount = 0;

/* utility functions to add/remove a list of series from lostSeries
 * (which contains the series which aren't autofetching anymore due to
 * some errors) */
static void
AddToLost(const char *myList) {
	char name[255 + 1];
	const char *ind;
	int i;
	Object obj;

	/* quick check */
	if (myList == NULL || myList[0] == '\0') {
		return;
	}

	GetNWSLock(&lock);
	if (lostSeries == NULL) {
		if (!InitRegistrations(&lostSeries)) {
			ABORT("AddToLost: failed to initialize structure\n");
		}
	}
	for (ind = myList; GETWORD(name, ind, &ind); ) {
		if (SearchForName(lostSeries, name, 1, &i)) {
			/* we already have it */
			continue;
		}
		
		/* we need to insert it */
		obj = NewObject();
		AddNwsAttribute(&obj, "name", name);
		if (!InsertRegistration(lostSeries, obj, 0, i)) {
			WARN1("AddToLost: couldn't insert %s\n", name);
		}
		FreeObject(&obj);
	}
	LOG1("AddToLost: we have %d lost series\n", lostSeries->howMany);
	ReleaseNWSLock(&lock);

	return;
}
static void
AddToLostFromSocket(Socket sock) {
	int i;
	char * tmp;

	if (sock < 0) {
		return;
	}

	tmp = NULL;
	GetNWSLock(&lock);
	for (i = 0; i < autoFetchCount; i++) {
		if (autoFetches[i].memory.sd == sock) {
			/* found it */
			break;
		}
	}
	if (i < autoFetchCount) {
		tmp = GenerateList(autoFetches[i].series);
	}
	ReleaseNWSLock(&lock);

	AddToLost(tmp);
	FREE(tmp);

	return;
}
static void
RemoveFromLost(const char *myList) {
	char name[255 + 1];
	const char *ind;
	int i;

	/* sanity check */
	if (myList == NULL || lostSeries == NULL) {
		return;
	}

	GetNWSLock(&lock);
	for (ind = myList; GETWORD(name, ind, &ind); ) {
		if (!SearchForName(lostSeries, name, 1, &i)) {
			/* we don't have this series */
			continue;
		}
		DeleteRegistration(lostSeries, i);
	}
	LOG1("RemoveFromLost: we have %d lost series\n", lostSeries->howMany);
	ReleaseNWSLock(&lock);
}

/*
 * Disconnects the auto-fetch memory connected to #who# and removes its entry
 * from the #autoFetches# module global list.
 */
static void
AutoFetchDisconnect(Socket who) {
	int i;
	DataDescriptor des = SIMPLE_DATA(CHAR_TYPE, 1);

	/* just get out of here if no socket */
	if (who == NO_SOCKET) {
		return;
	}
	/* we operate on global variables all over the place */
	GetNWSLock(&lock);
	for (i = 0; i < autoFetchCount; i++) {
		if (autoFetches[i].memory.sd == who) {
			/* found it */
			break;
		}
	}
	if (i >= autoFetchCount) {
		ReleaseNWSLock(&lock);
		WARN("AutoFetchDisconnect: memory not found!\n");
		return;
	}

	INFO2("AutoFetchDisconnect: disconnecting from %s:%d\n", autoFetches[i].memory.name, autoFetches[i].memory.port);
	
	/* let's clean the autoFetches about this memory */
	while(autoFetches[i].series->howMany > 0) {
		DeleteRegistration(autoFetches[i].series, 0);
	}
	FREE(autoFetches[i].series);
	if (autoFetchCount > 1) {
		autoFetches[i] = autoFetches[--autoFetchCount];
	} else {
		FREE(autoFetches);
		autoFetchCount = 0;
	}
	ReleaseNWSLock(&lock);

	/* let's stop autofetch (empty string) */
	des.repetitions = 1;
	if (!SendMessageAndData(who, AUTOFETCH_BEGIN, "", &des, 1, 5)) {
		ERROR("AutoFetchDisconnect: failed to talk to memory\n");
	}

	/* Ignore the reply.  It might get mixed with an autofetch anyway */
	DROP_SOCKET(&who);
}

/*
 * Connects to #memory#, sends it #seriesList# in an AUTOFETCH_BEGIN message,
 * and places an entry for the connection in the #autoFetches# global.  Returns
 * 1 if successful, else 0.
 *
 * NOTE: we need to hook a check whenever a socket gets disconnected,
 * since it can be an autofetching socket.
 */
static int
AutoFetchConnect(	struct host_cookie *memory,
			const char *seriesList) {
	AutoFetchInfo *extendedAutoFetches;
	size_t ignored;
	DataDescriptor seriesDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	char name[256], *series;
	const char *ind;
	Object obj;
	int i;

	/* sanity check */
	if (memory == NULL || seriesList == NULL) {
		FAIL("AutoFetchConnect: NULL parameters!\n");
	}

	/* let's check if we can talk to the memory first */
	if (!ConnectToHost(memory, NULL)) {
		FAIL1("AutoFetchConnect: Unable to contact memory %s\n", HostCImage(memory));
	}

	/* since we operates on the global variables, we lock a lot */
	GetNWSLock(&lock);
	/* extend the autoFetchers */
	extendedAutoFetches = (AutoFetchInfo *) REALLOC(autoFetches, (autoFetchCount + 1) * sizeof(AutoFetchInfo), 0);
	if (extendedAutoFetches == NULL) {
		ReleaseNWSLock(&lock);
		FAIL("AutoFetchConnect: out of memory\n");
	}
	extendedAutoFetches[autoFetchCount].series = NULL;
	if (!InitRegistrations(&extendedAutoFetches[autoFetchCount].series)) {
		FREE(extendedAutoFetches);
		ReleaseNWSLock(&lock);
		FAIL("AutoFetchConnect: cannot initialize structure\n");
	}
	autoFetches = extendedAutoFetches;

	/* let's save the series/memory */
	name[0] = '\0';
	autoFetches[autoFetchCount].memory = *memory;
	for (ind = seriesList; GETWORD(name, ind, &ind); ) {
		if (SearchForName(autoFetches[autoFetchCount].series, name, 1, &i)) {
			/* we already have it */
			continue;
		}

		/* insert the element */
		obj = NewObject();
		AddNwsAttribute(&obj, "name", name);
		if (!InsertRegistration(autoFetches[autoFetchCount].series, obj, 0, i)) {
			WARN1("AutoFetchConnect: couldn't insert %s\n", name);
		}
		FreeObject(&obj);
	}
	/* generate the list of series we got in */
	series = GenerateList(autoFetches[autoFetchCount].series);
	autoFetchCount++;
	ReleaseNWSLock(&lock);
	if (series == NULL) {
		WARN("AutoFetchConnect: empty series list?\n");
		return 0;
	}

	/* let's try to autofetch the series list */
	i = 1;
	seriesDescriptor.repetitions = strlen(series) + 1;
	if (!SendMessageAndData(memory->sd, AUTOFETCH_BEGIN, (const void *)series, &seriesDescriptor, 1, -1) || !RecvMessage(memory->sd, AUTOFETCH_ACK, &ignored, -1)) {
		i = 0;
		AutoFetchDisconnect(memory->sd);
		ERROR1("AutoFetchConnect: Memory %s refused request\n", HostCImage(memory));
	}
	FREE(series);

	return i;
}


char *
NWSAPI_AutoFetchError() {
	char *ret;
	int i;
	Socket s;

	/* first of all let's check that all the sockets we have here are
	 * good to go */
	GetNWSLock(&lock);
	for (i = 0; i < autoFetchCount; i++) {
		if (!IsOkay(autoFetches[i].memory.sd)) {
			s = autoFetches[i].memory.sd;
			WARN1("AutoFetchError: lost contact with memory %s\n", autoFetches[i].memory.name);
			ReleaseNWSLock(&lock);
			AddToLostFromSocket(s);
			AutoFetchDisconnect(s);
			GetNWSLock(&lock);

			/* let's recheck this index */
			i--;
		}
	}

	if (lostSeries == NULL) {
		ret = NULL;
	} else {
		ret = GenerateList(lostSeries);
		while (lostSeries->howMany > 0) {
			DeleteRegistration(lostSeries, 0);
		}
	}
	ReleaseNWSLock(&lock);

	return ret;
}

int
NWSAPI_AutoFetchAlready(const char *seriesName) {
	int i, ret;
	
	/* sanity check */
	if (seriesName == NULL) {
		FAIL("AutoFetchAlready: NULL parameter!\n");
	}

	/* easy way out */
	if (seriesName[0] == '\0') {
		return 0;
	}

	GetNWSLock(&lock);
	/* let's see if we already have this series */
	for(i = 0; i < autoFetchCount; i++) {
		if (SearchForName(autoFetches[i].series, seriesName, 1, &ret)) {
			break;
		}
	}
	if (i < autoFetchCount) {
		ret = 1;
	} else {
		ret = 0;
	}
	ReleaseNWSLock(&lock);

	return ret;
}

/* temporary structure used to keep track of which memory has which
 * series. The 2 following functions (AddMemorySeries & StartFetch) are
 * local to the AutoFetchBegin* functions.*/
typedef struct {
	struct host_cookie memory;
	char *toFetch;
} memorySeries;
static int
AddMemorySeries(	memorySeries **l,
			int *howMany,
			struct host_cookie *memory) {
	int i;
	memorySeries *list;

	list = *l;

	/* See whether we're already have this memory for other series. */
	for(i = 0; i < *howMany; i++) {
		if (SameHost(memory, &list[i].memory)) {
			break;
		}
	}
	if (i >= *howMany) {
		/* nope it's a new memory */
		i = *howMany;
		(*howMany)++;

		/* allocate space to save the memory */
		list = REALLOC(list, sizeof(memorySeries)*(*howMany), 0);
		if (list == NULL) {
			return -1;
		}
		list[i].toFetch = strdup("");
		if (list[i].toFetch == NULL) {
			return -1;
		}

		list[i].memory = *memory;
		*l = list;
	}

	return i;
}
static int
StartFetch(	memorySeries *list,
		int howMany) {
	int i, j, len, ret;
	char *listToFetch, *c;
	Socket sock;

	ret = 0;

	if (howMany <= 0) {
		/* I guess we are done! */
		return 1;
	}
	for (i = 0; i < howMany; i++) {
		sock = NO_SOCKET;

		GetNWSLock(&lock);
		/* See whether we're already asking this memory for other
		 * series. */
		for(j = 0; j < autoFetchCount; j++) {
			if (SameHost(&list[i].memory, &autoFetches[j].memory)) {
				break;
			}
		}
		if (j >= autoFetchCount) {
			/* this is the first series for this memory */
			listToFetch = strdup(list[i].toFetch);
		} else {
			/* An existing memory.  Because we're receiving
			 * messages asynchronously, we can't reliably
			 * reuse the connection -- we might mistake the
			 * ack for a series coming in or vice versa.
			 * Instead, get rid of this connection and open a
			 * new one with the extended list.  */
			c = GenerateList(autoFetches[i].series);
			len = strlen(c) + strlen(list[i].toFetch) + 3;
			listToFetch = (char *)MALLOC(len, 0);
			if (listToFetch) {
				len = strlen(c);
				memcpy(listToFetch, c, len);
				listToFetch[len] = '\t';
				memcpy(listToFetch + len + 1, list[i].toFetch, strlen(list[i].toFetch));
				sock = autoFetches[j].memory.sd;
			}
			FREE(c);
		}

		/* let's restart the autofecthing for this memory */
		ReleaseNWSLock(&lock);
		if (listToFetch == NULL) {
			ERROR("StartFetch out of memory\n");
			break;
		}
		if (sock != NO_SOCKET) {
			AutoFetchDisconnect(sock);
		}

		if (AutoFetchConnect(&list[i].memory, listToFetch)) {
			ret = 1;
		} else {
			AddToLost(listToFetch);
		}
		FREE(listToFetch);
	}

	/* let's free memory */
	for (i = 0; i < howMany; i++) {
		FREE(list[i].toFetch);
	}
	FREE(list);

	return ret;
}

int
NWSAPI_AutoFetchBegin(const char *seriesName) {
	int i, len;
	struct host_cookie memory;
	char tmpSeries[255];
	const char *ind;
	memorySeries *list;
	int howMany;

	/* sanity check */
	if (seriesName == NULL) {
		FAIL("AutoFetchBegin: NULL parameter\n");
	}

	/* let's loop through the series */
	list = NULL;
	for (howMany = 0, ind = seriesName; GETWORD(tmpSeries, ind, &ind); ) {
		/* let's see if we already checked this series */
		if (NWSAPI_AutoFetchAlready(tmpSeries)) {
			continue;
		}

		/* let's see if we can find the memory responsible */
		if (!NwsGetSeriesMemory(tmpSeries, &memory, -1)) {
			WARN("AutoFetchBegin: unable to determine memory\n");
			AddToLost(tmpSeries);
			continue;
		}

		/* let's get the right index for this memory */
		i = AddMemorySeries(&list, &howMany, &memory);
		if (i == -1 ) {
			ABORT("AutoFetchBegin: out of memory\n");
			return 0;
		}

		/* now let's add the series name to the to be fetch list */
		len = strlen(tmpSeries) + strlen(list[i].toFetch) + 2;
		list[i].toFetch = REALLOC(list[i].toFetch, len, 0);
		if (list[i].toFetch == NULL) {
			ERROR("AutoFetchBegin: out of memory\n");
			AddToLost(seriesName);
			for (i = 0; i < howMany; i++) {
				FREE(list[i].toFetch);
			}
			FREE(list);

			return 0;
		}
		strcat(list[i].toFetch, "\t");
		strcat(list[i].toFetch, tmpSeries);
	}

	/* we got all the memories/series coupled: time to fetch */
	return StartFetch(list, howMany);
}


int
NWSAPI_AutoFetchBeginObject(ObjectSet seriesSet) {
	int i, len;
	char *name;
	struct host_cookie memory;
	memorySeries *list;
	int howMany;
	Object obj;

	/* sanity check */
	if (seriesSet == NULL) {
		FAIL("AutoFetchBeginObject: NULL parameter\n");
	}

	/* let's loop through the series */
	list = NULL;
	obj = NWSAPI_FirstObject(seriesSet);
	for (howMany = 0; obj != NULL; obj = NextObject(seriesSet, obj)) {
		/* now we can get the name and the memory */
		name = NwsAttributeValue_r(FindNwsAttribute(obj, "memory"));
		if (name == NULL) {
			ERROR("AutoFetchBeginObject: series with no memory!\n");
			continue;
		}
		Host2Cookie(name, DefaultHostPort(NAME_SERVER_HOST), &memory);
		FREE(name);
		name = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
		if (name == NULL) {
			ERROR("AutoFetchBeginObject: series with no name!\n");
			continue;
		}

		/* let's see if we already checked this series */
		if (NWSAPI_AutoFetchAlready(name)) {
			FREE(name);
			continue;
		}

		/* let's get the right index for this memory */
		i = AddMemorySeries(&list, &howMany, &memory);
		if (i == -1 ) {
			ABORT("AutoFetchBegin: out of memory\n");
		}

		/* now let's add the series name to the to be fetch list */
		len = strlen(name) + strlen(list[i].toFetch) + 2;
		list[i].toFetch = REALLOC(list[i].toFetch, len, 0);
		if (list[i].toFetch == NULL) {
			ERROR("AutoFetchBeginObject: out of memory\n");
			AddToLost(name);
			FREE(name);
			for (i = 0; i < howMany; i++) {
				FREE(list[i].toFetch);
			}
			FREE(list);

			return 0;
		}
		if (strlen(list[i].toFetch) > 1) {
			strcat(list[i].toFetch, "\t");
		}
		strcat(list[i].toFetch, name);
		FREE(name);
	}

	/* we got all the memories/series coupled: time to fetch */
	return StartFetch(list, howMany);
}

int
NWSAPI_AutoFetchMessage(	int sock,
				char *seriesName,
				unsigned int nameLen,
				NWSAPI_Measurement *meas) {
	char *autoFetchContents;
	struct nws_memory_state s;
	const char *nextWord;
	DataDescriptor des = SIMPLE_DATA(CHAR_TYPE, 0);
	char measurement[127 + 1];
	char timeStamp[127 + 1];
	Socket sender;

	nextWord = autoFetchContents = NULL;
	timeStamp[0] = measurement[0] = '\0';

	/* sanity check */
	sender = (Socket)sock;
	if (sender == NO_SOCKET) {
		FAIL("AutoFetchMessage: no valid socket\n");
	}

	/* let's get the state */
	if (!RecvData(sender, &s, stateDescriptor, stateDescriptorLength, -1)) {
		AddToLostFromSocket(sender);
		AutoFetchDisconnect(sender);
		FAIL("AutoFetchMessage: failed to receive state\n");
	}

	/* get room for the data */
	des.repetitions = s.rec_size * s.rec_count;
	autoFetchContents = (char *)MALLOC(des.repetitions + 1, 0);
	if (autoFetchContents == NULL) {
		AddToLostFromSocket(sender);
		AutoFetchDisconnect(sender);
		FAIL("AutoFetchMessage: out of memory\n");
	}

	/* let's get the message */
	if (!RecvData(sender, autoFetchContents, &des, 1, -1)) {
		AddToLostFromSocket(sender);
		AutoFetchDisconnect(sender);
		FREE(autoFetchContents);
		FAIL("AutoFetchMessage: failed to receive message\n");
	}
	autoFetchContents[des.repetitions] = '\0';

	/* let's parse it */
	if (GETWORD(timeStamp, autoFetchContents, &nextWord) && GETWORD(measurement, nextWord, &nextWord)) {
		/* let's return the values and series name */
		zstrncpy(seriesName, s.id, nameLen);
		meas->timeStamp = strtod(timeStamp, NULL);
		meas->measurement = strtod(measurement, NULL);
	} else {
		AddToLostFromSocket(sender);
		AutoFetchDisconnect(sender);
		FREE(autoFetchContents);
		FAIL("AutoFetchMessage: failed to parse message\n");
	}
	FREE(autoFetchContents);

	return 1;
}


int
NWSAPI_AutoFetchCheck(char *seriesName,
                      unsigned int nameLen,
                      NWSAPI_Measurement *seriesMeasurement,
                      long timeOut) {
	MessageHeader header;
	Socket sender;
	int i;

	/* sanity check */
	if (seriesName == NULL || seriesMeasurement == NULL || nameLen <= 0) {
		FAIL("AutoFetchCheck: wrong parameter!\n");
	}
	seriesName[0] = '\0';

	 /* let's wait for a message: a message contains only 1 series
	  * name plus 1 measurements */
	if (!IncomingRequest(timeOut, &sender)) {
		/* nope, nothing for us */
		return 0;
	}

	/* let's get the header */
	if (!RecvHeader(sender, &header, -1) ||
			header.message != STATE_FETCHED) {
		char *name;
		
		AddToLostFromSocket(sender);
		AutoFetchDisconnect(sender);

		name = PeerName_r(sender);
		ERROR1("AutoFetchCheck: failed to receive from %s\n", name);
		FREE(name);

		return 0;
	}

	/* let's get the auto fetch data */
	i = NWSAPI_AutoFetchMessage(sender, seriesName, nameLen, seriesMeasurement);

	/* done with this socket */
	SocketIsAvailable(sender);

	return i;
}


void
NWSAPI_AutoFetchEnd(const char *seriesName) {
	int i, len, j;
	struct host_cookie memory;
	char *tmp, name[255];
	const char *ind;
	Socket sock;

	/* sanity check */
	if (seriesName == NULL) {
		ERROR("NWSAPI_AutoFetchEnd: NULL parameter\n");
		return;
	}

	/* we can have multiple series: let's get them one by one */
	for (ind = seriesName; GETWORD(name, ind, &ind); ) {
		/* if we don't have it, we don't need to stop it */
		if (!NWSAPI_AutoFetchAlready(name)) {
			continue;
		}

		len = strlen(name);

		/* let's look for the series */
		GetNWSLock(&lock);
		for(sock = NO_SOCKET, tmp = NULL, i = 0; i < autoFetchCount; i++) {
			/* let's look for the series */
			if (!SearchForName(autoFetches[i].series, name, 1, &j)) {
				/* not here */
				continue;
			}

			/* let's remmeber the memory to restart */
			SAFESTRCPY(memory.name, autoFetches[i].memory.name);
			memory.port = autoFetches[i].memory.port;
			sock = autoFetches[i].memory.sd;

			/* let's remove it: if there is only one series
			 * we remove the whole memory. */
			if (autoFetches[i].series->howMany > 1) {
				DeleteRegistration(autoFetches[i].series, j);
				tmp = GenerateList(autoFetches[i].series);
			}

			/* since we found the series is time to reset the
			 * autofetching with the memory */
			break;
		}
		ReleaseNWSLock(&lock);

		if (sock != NO_SOCKET) {
			/* See comment in AutoFetchBegin() as to why we
			 * can't reuse the connection. */
			AutoFetchDisconnect(sock);
			memory.sd = NO_SOCKET;
			if (tmp != NULL && strlen(tmp) > 0) {
				if (!AutoFetchConnect(&memory, tmp)) {
					/* got problem: add the series to the
					 * orphaned series */
					AddToLost(tmp);
				}
			}
		}
		FREE(tmp);
	}

	/* we remove the series from the lostSeries no matter what: we
	 * don't care about it anymore */
	RemoveFromLost(seriesName);
}


