# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import urlparse
import urllib
import StringIO
import os
import hashlib
import time
import httplib
import logging

from ninix.home import get_normalized_path


class NetworkUpdate(object):

    __BACKUP_SUFFIX = '.BACKUP'

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        self.event_queue = []
        self.state = None
        self.backups = []
        self.newfiles = []
        self.newdirs = []

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def is_active(self):
        return self.state is not None

    def enqueue_event(self, event,
                      ref0=None, ref1=None, ref2=None, ref3=None,
                      ref4=None, ref5=None, ref6=None, ref7=None):
        self.event_queue.append(
            (event, ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7))

    def get_event(self):
        return None if not self.event_queue else self.event_queue.pop(0)

    def has_events(self):
        return 1 if self.event_queue else 0

    def start(self, homeurl, ghostdir, timeout=60):
        url = urlparse.urlparse(homeurl)
        if not (url[0] == 'http' and url[3] == url[4] == url[5] == ''):
            self.enqueue_event('OnUpdateFailure', 'bad home URL')
            self.state = None
            return
        try:
            self.host, port = url[1].split(':')
            self.port = int(port)
        except ValueError:
            self.host = url[1]
            self.port = 80
        self.path = url[2]
        self.ghostdir = ghostdir
        self.timeout = timeout
        self.state = 0

    def interrupt(self):
        self.event_queue = []
        self.request_parent(
            'NOTIFY', 'enqueue_event', 'OnUpdateFailure', 'artificial')
        self.state = None
        self.stop(revert=1)

    def stop(self, revert=0):
        self.buffer = []
        if revert:
            for path in self.backups:
                if os.path.isfile(path):
                    os.rename(path, path[:-len(self.__BACKUP_SUFFIX)])
            for path in self.newfiles:
                if os.path.isfile(path):
                    os.remove(path)
            for path in self.newdirs:
                if os.path.isdir(path):
                    os.rmdir(path)
            self.backups = []
        self.newfiles = []
        self.newdirs = []

    def clean_up(self):
        for path in self.backups:
            if os.path.isfile(path):
                os.remove(path)
        self.backups = []

    def reset_timeout(self):
        self.timestamp = time.time()

    def check_timeout(self):
        return time.time() - self.timestamp > self.timeout

    LEN_STATE = 5
    LEN_PRE = 5

    def run(self):
        if self.state is None or \
                self.request_parent('GET', 'check_event_queue'):
            return 0
        elif self.state == 0:
            self.start_updates()
        elif self.state == 1:
            self.connect()
        elif self.state == 2:
            self.wait_response()
        elif self.state == 3:
            self.get_content()
        elif self.state == 4:
            self.schedule = self.make_schedule()
            if self.schedule is None:
                return 0
            self.final_state = len(self.schedule) * self.LEN_STATE + self.LEN_PRE
        elif self.state == self.final_state:
            self.end_updates()
        elif (self.state - self.LEN_PRE) % self.LEN_STATE == 0:
            filename, checksum = self.schedule[0]
            logging.info('UPDATE: {0} {1}'.format(filename, checksum))
            self.download(os.path.join(self.path, urllib.quote(filename)),
                          event=1)
        elif (self.state - self.LEN_PRE) % self.LEN_STATE == 1:
            self.connect()
        elif (self.state - self.LEN_PRE) % self.LEN_STATE == 2:
            self.wait_response()
        elif (self.state - self.LEN_PRE) % self.LEN_STATE == 3:
            self.get_content()
        elif (self.state - self.LEN_PRE) % self.LEN_STATE == 4:
            filename, checksum = self.schedule.pop(0)
            self.update_file(unicode(filename, 'Shift_JIS'), checksum)
        return 1

    def start_updates(self):
        self.enqueue_event('OnUpdateBegin')
        self.download(os.path.join(self.path, 'updates2.dau'))

    def download(self, locator, event=0):
        self.locator = self.encode(locator)
        self.http = httplib.HTTPConnection(self.host, self.port)
        if event:
            self.enqueue_event('OnUpdate.OnDownloadBegin',
                               os.path.basename(locator),
                               self.file_number, self.num_files)
        self.state += 1
        self.reset_timeout()

    def encode(self, path):
        return ''.join([self.encode_special(c) for c in path])

    def encode_special(self, c):
        return c if '\x20' < c < '\x7e' else '%{0:02x}'.format(ord(c))

    def connect(self):
        try:
            self.http.connect()
        except:
            self.enqueue_event('OnUpdateFailure', 'connection failed')
            self.state = None
            self.stop(revert=1)
            return
        if self.check_timeout():
            self.enqueue_event('OnUpdateFailure', 'timeout')
            self.state = None
            self.stop(revert=1)
            return
        self.http.request('GET', self.locator)
        self.state += 1
        self.reset_timeout()

    def wait_response(self):
        try:
            self.response = self.http.getresponse()
        except:
            self.enqueue_event('OnUpdateFailure', 'no HTTP response')
            self.state = None
            self.stop(revert=1)
            return
        if self.check_timeout():
            self.enqueue_event('OnUpdateFailure', 'timeout')
            self.state = None
            self.stop(revert=1)
            return
        code = self.response.status
        message = self.response.reason
        if code == 200:
            pass
        elif code == 302 and self.redirect():
            return
        elif self.state == 2: # updates2.dau
            self.enqueue_event('OnUpdateFailure', str(code))
            self.state = None
            return
        else:
            filename, checksum = self.schedule.pop(0)
            logging.error(
                'failed to download {0} ({1:d} {2})'.format(
                    filename, code, message))
            self.file_number += 1
            self.state += 3
            return
        self.buffer = []
        size = self.response.getheader('content-length', None)
        if size is None:
            self.size = None
        else:
            self.size = int(size)
        self.state += 1
        self.reset_timeout()

    def redirect(self):
        location = self.response.getheader('location', None)
        if location is None:
            return 0
        url = urlparse.urlparse(location)
        if not (url[0] == 'http' and url[3] == url[4] == url[5] == ''):
            return 0
        logging.info('redirected to {0}'.format(location))
        self.http.close()
        try:
            self.host, port = url[1].split(':')
            self.port = int(port)
        except ValueError:
            self.host = url[1]
            self.port = 80
        self.path = os.path.dirname(url[2])
        self.state -= 2
        self.download(url[2])
        return 1

    def get_content(self):
        data = self.response.read()
        if not data:
            if self.check_timeout():
                self.enqueue_event('OnUpdateFailure', 'timeout')
                self.state = None
                self.stop(revert=1)
                return
            elif data is None:
                return
        elif data < 0:
            self.enqueue_event('OnUpdateFailure', 'data retrieval failed')
            self.state = None
            self.stop(revert=1)
            return
        if data:
            self.buffer.append(data)
            if self.size is not None:
                self.size = self.size - len(data)
            self.reset_timeout()
            return
        if self.size is not None and self.size > 0:
            return
        self.http.close()
        self.state += 1

    def make_checksum(self, digest):
        return ''.join(['{0:02x}'.format(ord(x)) for x in digest])

    ROOT_FILES = ['install.txt', 'delete.txt', 'readme.txt', 'thumbnail.png']

    def adjust_path(self, filename):
        filename = get_normalized_path(filename)
        if filename in self.ROOT_FILES or os.path.dirname(filename):
            return filename
        return os.path.join('ghost', 'master', filename)

    def make_schedule(self):
        schedule = self.parse_updates2_dau()
        if schedule is not None:
            self.num_files = len(schedule) - 1
            self.file_number = 0
            if self.num_files >= 0:
                self.enqueue_event('OnUpdateReady', self.num_files)
            self.state += 1
        return schedule

    def parse_updates2_dau(self):
        schedule = []
        f = StringIO.StringIO(''.join(self.buffer))
        for line in f:
            try:
                filename, checksum, newline = line.split('\001', 2)
            except ValueError:
                self.enqueue_event('OnUpdateFailure', 'broken updates2.dau')
                self.state = None
                return None
            if not filename:
                continue
            path = os.path.join(self.ghostdir, self.adjust_path(
                    unicode(filename, 'Shift_JIS')))
            try:
                with open(path, 'rb') as f:
                    data = f.read()
            except IOError: # does not exist or broken
                pass
            else:
                m = hashlib.md5()
                m.update(data)
                if checksum == self.make_checksum(m.digest()):
                    continue
            schedule.append((filename, checksum))
        self.updated_files = []
        return schedule

    def update_file(self, filename, checksum):
        data = ''.join(self.buffer)
        m = hashlib.md5()
        m.update(data)
        digest = self.make_checksum(m.digest())
        if digest == checksum:
            path = os.path.join(self.ghostdir, self.adjust_path(filename))
            subdir = os.path.dirname(path)
            if not os.path.exists(subdir):
                subroot = subdir
                while 1:
                    head, tail = os.path.split(subroot)
                    if os.path.exists(head):
                        break
                    else:
                        subroot = head
                self.newdirs.append(subroot)
                try:
                    os.makedirs(subdir)
                except OSError:
                    self.enqueue_event(
                        'OnUpdateFailure', ''.join(("can't mkdir ", subdir)))
                    self.state = None
                    self.stop(revert=1)
                    return
            if os.path.exists(path):
                if os.path.isfile(path):
                    backup = ''.join((path, self.__BACKUP_SUFFIX))
                    os.rename(path, backup)
                    self.backups.append(backup)
            else:
                self.newfiles.append(path)
            try:
                with open(path, 'wb') as f:
                    try:
                        f.write(data)
                    except IOError:
                        self.enqueue_event(
                            'OnUpdateFailure',
                            ''.join(("can't write ", os.path.basename(path))))
                        self.state = None
                        self.stop(revert=1)
                        return
            except IOError:
                self.enqueue_event(
                    'OnUpdateFailure',
                    ''.join(("can't open ", os.path.basename(path))))
                self.state = None
                self.stop(revert=1)
                return
            self.updated_files.append(filename)
            event = 'OnUpdate.OnMD5CompareComplete'
        else:
            event = 'OnUpdate.OnMD5CompareFailure'
            self.enqueue_event(event, filename, checksum, digest)
            self.state = None
            self.stop(revert=1)
            return
        self.enqueue_event(event, filename, checksum, digest)
        self.file_number += 1
        self.state += 1

    def end_updates(self):
        filelist = self.parse_delete_txt()
        if filelist:
            for filename in filelist:
                path = os.path.join(self.ghostdir, filename)
                if os.path.exists(path) and os.path.isfile(path):
                    try:
                        os.unlink(path)
                        logging.info('deleted {0}'.format(path))
                    except OSError as e:
                        logging.error(e)
        update_list = ','.join(self.updated_files)
        if not update_list:
            self.enqueue_event('OnUpdateComplete', 'none')
        else:
            self.enqueue_event('OnUpdateComplete', 'changed', update_list)
        self.state = None
        self.stop()

    def parse_delete_txt(self):
        filelist = []
        try:
            with open(os.path.join(self.ghostdir, 'delete.txt')) as f:
                for line in f:
                    line = line.strip()
                    if not line:
                        continue
                    filename = unicode(line, 'Shift_JIS')
                    filelist.append(get_normalized_path(filename))
        except IOError:
            return None
        return filelist


def test():
    import sys
    if len(sys.argv) != 3:
        raise SystemExit, 'Usage: update.py homeurl ghostdir\n'
    update = NetworkUpdate({'enqueu_event': lambda *a: None,
                            'check_event_queue': lambda *a: None,})
    update.start(sys.argv[1], sys.argv[2], timeout=60)
    while 1:
        state = update.state
        s = time.time()
        code = update.run()
        e = time.time()
        delta = e - s
        if delta > 0.1:
            print 'Warning: state = {0:d} ({1:f} sec)'.format(state, delta)
        while 1:
            event = update.get_event()
            if not event:
                break
            print event
        if code == 0:
            break
        if update.state == 5 and update.schedule:
            print 'File(s) to be update:'
            for filename, checksum in update.schedule:
                print '   ', filename
    update.stop()


if __name__ == '__main__':
    test()
