/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_RSA.h: interface for the PKI_RSA class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PKI_RSA_H
#define PKI_RSA_H



#include <PEM_DER.h>
#include <PKI_ERR.h>
#include <ExceptionNewPKI.h>

#include <openssl/pem.h>
#include <openssl/bio.h>
#include <openssl/engine.h>
#include <openssl/evp.h>
#include <openssl/rsa.h>
#include <mString.h>

/*!
  This class represents a RSA bi-key
*/
class PKI_RSA  
{
public:
	/*! \brief This is the constructor
	 */
	PKI_RSA();

	/*! \brief This is the constructor.
	 *  \param other [IN] The other key to copy from.
	 */
	PKI_RSA(const PKI_RSA & other);

	/*! \brief This is the constructor, it loads a key.
	 *  \param pem_key [IN] The RSA key, either in PEM, or FILE:filename, or ENGINE:keyid.
	 *  \param engine [IN] A pointer to the engine, can be NULL.
	 *  \exception ExceptionNewPKI An error occured.
	 */
	PKI_RSA(const mString & pem_key, ENGINE * engine);

	/*! \brief This is the constructor, it generates a key.
	 *  \param KeyBitsLen [IN] The number of bits for the new key.
	 *  \param engine [IN] A pointer to the engine, can be NULL.
	 *  \exception ExceptionNewPKI An error occured.
	 */
	PKI_RSA(int KeyBitsLen, ENGINE * engine);

	/*! \brief This is the constructor, it assigns a key.
	 *  \param RSAkey [IN] The key to assign.
	 *  \exception ExceptionNewPKI An error occured.
	 */
	PKI_RSA(const EVP_PKEY * RSAkey);

	/*! \brief This is the destructor.
	 */
	virtual ~PKI_RSA();

	/*! \brief This function loads a key.
	 *  \param pem_key [IN] The RSA key, either in PEM, or FILE:filename, or ENGINE:keyid.
	 *  \param engine [IN] A pointer to the engine, can be NULL.
	 *  \return true on success, false on failure.
	 */
	bool SetKey(const mString & pem_key, ENGINE * engine);

	/*! \brief This function assigns a key.
	 *  \param RSAkey [IN] The key to assign.
	 *  \return true on success, false on failure.
	 */
	bool SetKey(const EVP_PKEY *RSAkey);

	/*! \brief This function assigns a key.
	 *  \param RSAkey [IN] The key to assign.
	 *  \return true on success, false on failure.
	 */
	bool SetKey(const RSA *RSAkey);

	/*! \brief This function returns the public key.
	 *  \return The public key.
	 */
	const X509_PUBKEY * GetPublicKey() const;

	/*! \brief This function returns the bi key in OSSL format.
	 *  \param dup [IN] Should the RSA be duplicated, then freed by calling function.
	 *  \return The bi key in OSSL format.
	 */
	RSA * GetRSA(bool dup=false);

	/*! \brief This function returns the bi key in OSSL format.
	 *  \param dup [IN] Should the RSA be duplicated, then freed by calling function.
	 *  \return The bi key in OSSL format.
	 */
	const RSA * GetRSA(bool dup=false) const;

	/*! \brief This function returns the size of the key in bits.
	 *  \return The size of the key in bits.
	 */
	long GetRsaSize() const;

	/*! \brief This function generates a key.
	 *  \param KeyBitsLen [IN] The length in key to generate.
	 *  \param engine [IN] A pointer to the engine, can be NULL.
	 *  \return true on success, false on failure.
	 */
	bool GenerateKey(int KeyBitsLen, ENGINE * engine);

	/*! \brief This function returns the key in OpenSSL EVP_PKEY format.
	 *  \return The key in OpenSSL EVP_PKEY format.
	 */
	const EVP_PKEY * GetRsaKey() const;

	/*! \brief This function returns the key in PEM format.
	 *  \return The key in PEM format.
	 */
	const mString & GetRsaKeyPem() const;
	
	/*! \brief This operator copies one PKI_RSA into another.
	 *  \param other [IN] The other key to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=( const PKI_RSA &other );

	/*! \brief This operator is used to know if the class is set.
	 *  \return return 1 when key is operable and 0 when not.
	 */
	operator int() const;

	/*! \brief This function clears the internal datas.
	 */
	void Clear();

private:
	typedef enum en_PKI_RSA_KEY_TYPE
	{
		PKI_RSA_KEY_TYPE_NORMAL=1,
		PKI_RSA_KEY_TYPE_FILE=2,
		PKI_RSA_KEY_TYPE_ENGINE=3
	}PKI_RSA_KEY_TYPE;

	PKI_RSA_KEY_TYPE KeyType;

	EVP_PKEY *pkey;
	mString PemKey;
	mString InitialString;
	ENGINE * e;
	X509_PUBKEY * pubKey;

	void Reset();

	bool LoadKeyFromEngine(const mString & KeyFile);
	bool LoadKeyFromFile(const mString & KeyFile);

	bool StringToEVP_PKEY(const mString & PrivateKey);
	bool EVP_PKEYToString();

	static void req_cb(int p, int n, void *arg);
	static int pem_password_cb(char *buf, int size, int rwflag, void *userdata);
};

#endif // PKI_RSA_H

