/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1CONF_H
#define ASN1CONF_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <assert.h>
#include <openssl/x509.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/x509v3.h>
#include <openssl/rand.h>
#include <openssl/evp.h>
#include <openssl/err.h>
#include <PKI_CERT.h>
#include <PKI_CRL.h>
#include <PKI_PKCS12.h>
#include <PKI_P7B.h>
#include "Asn1Helper.h"
#include "Asn1HashVal.h"


#define PKI_REPOSITORIES_RESOURCE		"PkiRepositories"	//!< This is the resource name for the PKI entity's repositories conf
#define PKI_ENTITIES_LINKS_RESOURCE		"PkiEntitiesLinks"	//!< This is the resource name for the PKI entity's links
#define PKI_ENTITIES_GROUPS_RESOURCE	"PkiGroupsLinks"	//!< This is the resource name for the PKI entity's groups

#define CRYPTO_LOCK_REPS_INFO CRYPTO_NUM_LOCKS+1	//!< This is the lock ID for the repositories structure
#define CRYPTO_LOCK_INTERNAL_PKI_CRL CRYPTO_NUM_LOCKS+2		//!< This is the lock ID for the internal CRL structure
#define CRYPTO_LOCK_INTERNAL_PKI_CA CRYPTO_NUM_LOCKS+3		//!< This is the lock ID for the internal CA structure
#define CRYPTO_LOCK_ENTITY_CONF_CRYPTED CRYPTO_NUM_LOCKS+4	//!< This is the lock ID for the PKI conf structure

#define NEWPKI_NUM_LOCKS CRYPTO_NUM_LOCKS+4					//!< This is the number of locks

#define ENTITY_TYPE_RA			0	//!< The entity type RA
#define ENTITY_TYPE_CA			1	//!< The entity type CA				
#define ENTITY_TYPE_REPOSITORY	2	//!< The entity type Repository
#define ENTITY_TYPE_PUBLICATION	3	//!< The entity type Publication
#define ENTITY_TYPE_KEY_STORE	4	//!< The entity type Key Store
#define ENTITY_TYPE_PKI			5	//!< The entity type PKI
#define ENTITY_TYPE_BACKUP		6	//!< The entity type Backup
#define ENTITY_TYPE_EE			7	//!< The entity type EE


#define CONF_VERSION_0 0
#define CONF_VERSION_1 1

#define REP_CONF_VERSION CONF_VERSION_0
#define REP_CONF_PTR get_conf0()

/*!
	This structure represents the body version 0 of a repository conf
*/
typedef struct st_ENTITY_CONF_BODY_REP_0
{
	ASN1_INTEGER * Enabled;
}ENTITY_CONF_BODY_REP_0;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_REP_0)


/*!
	This structure represents a repository conf
*/
typedef struct st_ENTITY_CONF_BODY_REP
{
	int type;
	union{
		ENTITY_CONF_BODY_REP_0 * conf0;	//!< Conf version 0
		void * ptr;
	}d;
}ENTITY_CONF_BODY_REP;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_REP)


#define PKI_CONF_VERSION CONF_VERSION_0
#define PKI_CONF_PTR get_conf0()

/*!
	This structure represents the body version 0 of a PKI conf
*/
typedef struct st_ENTITY_CONF_BODY_PKI_0
{
	ASN1_INTEGER * offline;			//!< The PKI is offline
}ENTITY_CONF_BODY_PKI_0;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_PKI_0)

/*!
	This structure represents a PKI conf
*/
typedef struct st_ENTITY_CONF_BODY_PKI
{
	int type;
	union{
		ENTITY_CONF_BODY_PKI_0 * conf0; //!< Conf version 0
		void * ptr;
	}d;
}ENTITY_CONF_BODY_PKI;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_PKI)

/*!
	This structure represents a plugin
*/
typedef struct st_EXTERNAL_ENTITY_PLUG
{
	ASN1_UTF8STRING * LibraryPath;		//!< The path to the plugin
	STACK_OF(PLUG_OPTION) * Options;	//!< The options
}EXTERNAL_ENTITY_PLUG;
DECLARE_ASN1_ITEM(EXTERNAL_ENTITY_PLUG)



#define EE_CONF_VERSION CONF_VERSION_0
#define EE_CONF_PTR get_conf0()

#define EE_DISCLAIMER_DN_VALIDATION 0


#define EE_ALLOW_CREATE_PROFILE			1		//!< EE conf flag to allow web users to create profiles
#define EE_ALLOW_SERVER_SIDE_KEY_GEN	2		//!< EE conf flag to allow web users to request a PKCS#12
#define EE_ALLOW_PKCS10_IMPORT			4		//!< EE conf flag to allow web users to import a PKCS#10
#define EE_ALLOW_SC_KEY_GEN				8		//!< EE conf flag to allow web users to request a Smart Card PKCS#10 generation
#define EE_LDAP_AUTHENTICATE			16		//!< EE conf flag to have web users authenticate against an LDAP server

/*!
	This structure represents the body version 0 of a EE conf
*/
typedef struct st_ENTITY_CONF_BODY_EE_0
{
	ASN1_BIT_STRING * Flags;				//!< The Flags
	ASN1_INTEGER * MinKeyLen;				//!< The minimum key length
	ASN1_INTEGER * MinPwdLen;				//!< The minimum PKCS#12 password
	STACK_OF(DN_SPECS) * Dn_Specs;			//!< The DN specs
	STACK_OF(POLICY_VALUE) * Policies;		//!< The DN policies
	ASN1_UTF8STRING * Site_Name;			//!< The site name, used to generate the URLs sent to the End Users
	ASN1_UTF8STRING * Dn_Val_Disclaimer;	//!< The disclaimer to show on the DN validation page
	STACK_OF(ASN1_UTF8STRING) * cas;		//!< The list of CAs from the attached RA
	ASN1_UTF8STRING * ldap_server;			//!< The address of the LDAP server
	ASN1_INTEGER * ldap_port;				//!< The port of the LDAP server
	ASN1_UTF8STRING * ldap_username;		//!< The username to use during the LDAP connection
	ASN1_UTF8STRING * ldap_password;		//!< The password to use during the LDAP connection
	ASN1_UTF8STRING * ldap_base;			//!< The base of the LDAP
	ASN1_UTF8STRING * ldap_filters;			//!< Some filters to add during LDAP auto synchro search
	ASN1_UTF8STRING * ldap_attr_name;		//!< The name of the LDAP attribute that represents the user's UID
	ASN1_INTEGER * ldap_utf8;				//!< Is the LDAP server using UTF-8 ?
}ENTITY_CONF_BODY_EE_0;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_EE_0)

/*!
	This structure represents a EE conf
*/
typedef struct st_ENTITY_CONF_BODY_EE
{
	int type;
	union{
		ENTITY_CONF_BODY_EE_0 * conf0;	//!< Conf version 0
		void * ptr;
	}d;
}ENTITY_CONF_BODY_EE;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_EE)




#define CA_CONF_VERSION CONF_VERSION_1
#define CA_CONF_PTR get_conf1()

/*!
	This structure represents the body version 0 of a CA conf
*/
typedef struct st_ENTITY_CONF_BODY_CA_0
{
	STACK_OF(EXTENSION_VALUE) * Exts;		//!< The certificate extensions
	STACK_OF(EXTENSION_VALUE) * CrlExts;	//!< The CRL extensions
	ASN1_INTEGER * CrlValidityDays;			//!< The validity in days for the generated CRLs
	ASN1_INTEGER * CrlValidityHours;		//!< The validity in hours for the generated CRLs
	EXTERNAL_ENTITY_PLUG * PlugCa;			//!< A validation plugin
}ENTITY_CONF_BODY_CA_0;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_CA_0)


#define CA_ALLOW_CSR_EXTS					1	//!< CA conf flag to allow CSR extensions
#define CA_CSR_EXTS_OVERWRITE				2	//!< CA conf flag to overwrite CSR extensions

/*!
	This structure represents the body version 1 of a CA conf
*/
typedef struct st_ENTITY_CONF_BODY_CA_1
{
	ASN1_BIT_STRING * Flags;				//!< The Flags
	STACK_OF(EXTENSION_VALUE) * Exts;		//!< The certificate extensions
	STACK_OF(EXTENSION_VALUE) * CrlExts;	//!< The CRL extensions
	ASN1_INTEGER * CrlValidityDays;			//!< The validity in days for the generated CRLs
	ASN1_INTEGER * CrlValidityHours;		//!< The validity in hours for the generated CRLs
	EXTERNAL_ENTITY_PLUG * PlugCa;			//!< A validation plugin
}ENTITY_CONF_BODY_CA_1;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_CA_1)

/*!
	This structure represents a CA conf
*/
typedef struct st_ENTITY_CONF_BODY_CA
{
	int type;
	union{
		ENTITY_CONF_BODY_CA_0 * conf0;	//!< Conf version 0
		ENTITY_CONF_BODY_CA_1 * conf1;	//!< Conf version 0
		void * ptr;
	}d;
}ENTITY_CONF_BODY_CA;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_CA)

#define PUBLICATION_METHOD_INFO_TYPE_CERT 1		//!< The publication method is for a certificate
#define PUBLICATION_METHOD_INFO_TYPE_CRL 2		//!< The publication method is for a CRL

/*!
	This structure represents a publication method
*/
typedef struct st_PUBLICATION_METHOD_INFO
{
	ASN1_UTF8STRING * Name;				//!< The publication method's name
	ASN1_INTEGER * Type;				//!< The publication method's type
	ASN1_UTF8STRING * LibraryPath;		//!< The path to the publication plugin
	STACK_OF(PLUG_OPTION) * Options;	//!< The options for the publication method
}PUBLICATION_METHOD_INFO;
DECLARE_ASN1_ITEM(PUBLICATION_METHOD_INFO)

/*!
	This structure represents a list of publication methods
*/
typedef struct st_PUBLICATION_ENTRY
{
	ASN1_UTF8STRING * CaName;					//!< The name of the CA concerned by the methods
	STACK_OF(PUBLICATION_METHOD_INFO) * Methods;		//!< The publication methods
}PUBLICATION_ENTRY;
DECLARE_ASN1_ITEM(PUBLICATION_ENTRY)


#define PUB_CONF_VERSION CONF_VERSION_0
#define PUB_CONF_PTR get_conf0()

/*!
	This structure represents the body version 0 of a Publication conf
*/
typedef struct st_ENTITY_CONF_BODY_PUB_0
{
	STACK_OF(PUBLICATION_ENTRY) * Publications;		//!< The list of publication methods
	ASN1_INTEGER * OcspEnabled;						//!< Should the OCSP server be enabled
	ASN1_UTF8STRING * BindAddress;					//!< The OCSP server bind address
	ASN1_INTEGER * Port;							//!< The OCSP server port
}ENTITY_CONF_BODY_PUB_0;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_PUB_0)

/*!
	This structure represents a Publication conf
*/
typedef struct st_ENTITY_CONF_BODY_PUB
{
	int type;
	union{
		ENTITY_CONF_BODY_PUB_0 * conf0;		//!< Conf version 0
		void * ptr;
	}d;
}ENTITY_CONF_BODY_PUB;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_PUB)






#define RA_ALLOW_SERVER_SIDE_KEY_GEN		1	//!< RA conf flag to allow the RA entity to generate end-users private keys
#define RA_ALLOW_PKCS10_IMPORT				2	//!< RA conf flag to allow the RA to import an end-user PKCS#10
#define RA_ALLOW_OP_SC_KEY_GEN				4	//!< RA conf flag to allow the RA Operator to generate the end-user's private key into a Smart Card
#define RA_PUSH_KEYS_TO_KEY_ESCROW			8	//!< RA conf flag to tell the RA to push the key it generated to a key store
#define RA_REMOVE_UNKNOWN_DN_FIELDS			16	//!< RA conf flag to tell the RA to remove the DN fields that would not be present into DnSpecs
#define RA_ALLOW_LDAP_PROFILE_IMPORT		32	//!< RA conf flag to allow LDAP profiles importation
#define RA_ALLOW_LDAP_AUTO_SYNCH			64	//!< RA conf flag to tell the RA that it should synchronize automatically with the LDAP
#define RA_WARN_PROFILE_OWNER_EXPIRING		128	//!< RA conf flag to tell the RA that it should warn a profile owner when one of its certificate is about to expire
#define RA_EE_PROFILES_REQUIRE_VALIDATION	256	//!< RA conf flag to require EE profiles to be validated by an admin before being accepted


#define RA_CONF_VERSION CONF_VERSION_1
#define RA_CONF_PTR get_conf1()

/*!
	This structure represents the body version 0 of a RA conf
*/
typedef struct st_ENTITY_CONF_BODY_RA_0
{
	ASN1_BIT_STRING * Flags;				//!< The Flags
	STACK_OF(DN_SPECS) * DnSpecs;			//!< The DN specs
	STACK_OF(POLICY_VALUE) * Policies;		//!< The DN policies
	ASN1_INTEGER * MinKeyLen;				//!< The minimum key length
	ASN1_INTEGER * MinPwdLen;				//!< The minimum PKCS#12 password
	ASN1_INTEGER * default_validity;		//!< The default validity for a certificate
	ASN1_UTF8STRING * ldap_server;			//!< The address of the LDAP server
	ASN1_INTEGER * ldap_port;				//!< The port of the LDAP server
	ASN1_UTF8STRING * ldap_username;		//!< The username to use during the LDAP connection
	ASN1_UTF8STRING * ldap_password;		//!< The password to use during the LDAP connection
	ASN1_UTF8STRING * ldap_base;			//!< The base of the LDAP
	ASN1_INTEGER * ldap_profiles_group;		//!< Delegate the ownership on newly imported profiles to this group
	ASN1_UTF8STRING * ldap_filters;			//!< Some filters to add during LDAP auto synchro search
	ASN1_UTF8STRING * ldap_attr_name;		//!< The name of the LDAP attribute that represents the user's UID
	ASN1_INTEGER * ldap_utf8;				//!< Is the LDAP server using UTF-8 ?
}ENTITY_CONF_BODY_RA_0;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_RA_0)

/*!
	This structure represents the body version 0 of a RA conf
*/
typedef struct st_ENTITY_CONF_BODY_RA_1
{
	ASN1_BIT_STRING * Flags;				//!< The Flags
	STACK_OF(DN_SPECS) * DnSpecs;			//!< The DN specs
	STACK_OF(POLICY_VALUE) * Policies;		//!< The DN policies
	ASN1_INTEGER * MinKeyLen;				//!< The minimum key length
	ASN1_INTEGER * MinPwdLen;				//!< The minimum PKCS#12 password
	ASN1_INTEGER * default_validity;		//!< The default validity for a certificate
	ASN1_UTF8STRING * ldap_server;			//!< The address of the LDAP server
	ASN1_INTEGER * ldap_port;				//!< The port of the LDAP server
	ASN1_UTF8STRING * ldap_username;		//!< The username to use during the LDAP connection
	ASN1_UTF8STRING * ldap_password;		//!< The password to use during the LDAP connection
	ASN1_UTF8STRING * ldap_base;			//!< The base of the LDAP
	ASN1_INTEGER * ldap_profiles_group;		//!< Delegate the ownership on newly imported profiles to this group
	ASN1_UTF8STRING * ldap_filters;			//!< Some filters to add during LDAP auto synchro search
	ASN1_UTF8STRING * ldap_attr_name;		//!< The name of the LDAP attribute that represents the user's UID
	ASN1_INTEGER * ldap_utf8;				//!< Is the LDAP server using UTF-8 ?
	ASN1_INTEGER * ee_profiles_group;		//!< Delegate the ownership on newly imported profiles to this group
}ENTITY_CONF_BODY_RA_1;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_RA_1)

/*!
	This structure represents a RA conf
*/
typedef struct st_ENTITY_CONF_BODY_RA
{
	int type;
	union{
		ENTITY_CONF_BODY_RA_0 * conf0;	//!< Conf version 0
		ENTITY_CONF_BODY_RA_1 * conf1;	//!< Conf version 1
		void * ptr;
	}d;
}ENTITY_CONF_BODY_RA;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY_RA)




/*!
	This structure respresents the conf body
*/
typedef struct st_ENTITY_CONF_BODY
{
	int type;
	union{
		ASN1_OCTET_STRING * conf;			//!< An undefined conf
		ENTITY_CONF_BODY_RA * ra_conf;		//!< A RA conf
		ENTITY_CONF_BODY_CA * ca_conf;		//!< A CA conf
		ENTITY_CONF_BODY_REP * rep_conf;	//!< A Repository conf
		ENTITY_CONF_BODY_PKI * pki_conf;	//!< A PKI conf
		ENTITY_CONF_BODY_PUB * pub_conf;	//!< A Publication conf
		ENTITY_CONF_BODY_EE * ee_conf;		//!< An EE conf
	}d;
}ENTITY_CONF_BODY;
DECLARE_ASN1_ITEM(ENTITY_CONF_BODY)


#define REP_ENTRY_INFO_FLAG_FIREWALLED			1	//!< The repository is firewalled
#define REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO	2	//!< The repository should do a full synchro

/*!
	This structure respresents a repository information
*/
typedef struct st_REP_ENTRY_INFO
{
	X509 * RepositorySSL;			//!< The repository certificate
	ASN1_UTF8STRING * Name;			//!< The repository name
	ASN1_UTF8STRING * Address;		//!< The repository address
	ASN1_INTEGER * Port;			//!< The repository port
	ASN1_BIT_STRING	* Flags;		//!< The flags
}REP_ENTRY_INFO;
DECLARE_ASN1_ITEM(REP_ENTRY_INFO)



/*!
	This structure respresents a CA information
*/
typedef struct st_CA_ENTRY_INFO
{
	X509 * CaSSL;				//!< The CA certificate
	ASN1_UTF8STRING * Name;		//!< The CA name
}CA_ENTRY_INFO;
DECLARE_ASN1_ITEM(CA_ENTRY_INFO)

/*!
	This structure respresents a list of CAs
*/
typedef struct st_CAS_INFO
{
	STACK_OF(CA_ENTRY_INFO) * list;		//!< The list
}CAS_INFO;
DECLARE_ASN1_ITEM(CAS_INFO)

/*!
	This structure respresents a RA information
*/
typedef struct st_RA_ENTRY_INFO
{
	X509 * RaSSL;				//!< The RA certificate
	ASN1_UTF8STRING * Name;		//!< The RA name
}RA_ENTRY_INFO;
DECLARE_ASN1_ITEM(RA_ENTRY_INFO)

/*!
	This structure respresents a list of RAs
*/
typedef struct st_RAS_INFO
{
	STACK_OF(RA_ENTRY_INFO) * list;		//!< The list
}RAS_INFO;
DECLARE_ASN1_ITEM(RAS_INFO)


/*!
	This structure respresents a Publication information
*/
typedef struct st_PUB_ENTRY_INFO
{
	ASN1_UTF8STRING * Name;			//!< The Publication name
	X509 * PubSSL;					//!< The Publication certificate
	X509 * PubOCSP;					//!< The Publication OCSP certificate
}PUB_ENTRY_INFO;
DECLARE_ASN1_ITEM(PUB_ENTRY_INFO)

/*!
	This structure respresents a list of Publications
*/
typedef struct st_PUBS_INFO
{
	STACK_OF(PUB_ENTRY_INFO) * list;		//!< The list
}PUBS_INFO;
DECLARE_ASN1_ITEM(PUBS_INFO)

/*!
	This structure respresents a Key Store information
*/
typedef struct st_KEY_ENTRY_INFO
{
	X509 * KeySSL;				//!< The Key Store certificate
	ASN1_UTF8STRING * Name;		//!< The Key Store name
}KEY_ENTRY_INFO;
DECLARE_ASN1_ITEM(KEY_ENTRY_INFO)

/*!
	This structure respresents a list of Key Stores
*/
typedef struct st_KEYS_INFO
{
	STACK_OF(KEY_ENTRY_INFO) * list;		//!< The list
}KEYS_INFO;
DECLARE_ASN1_ITEM(KEYS_INFO)


/*!
	This structure respresents an EE information
*/
typedef struct st_EE_ENTRY_INFO
{
	X509 * EESSL;				//!< The EE certificate
	ASN1_UTF8STRING * Name;		//!< The EE name
}EE_ENTRY_INFO;
DECLARE_ASN1_ITEM(EE_ENTRY_INFO)

/*!
	This structure respresents a list of EEs
*/
typedef struct st_EES_INFO
{
	STACK_OF(EE_ENTRY_INFO) * list;		//!< The list
}EES_INFO;
DECLARE_ASN1_ITEM(EES_INFO)


/*!
	This structure respresents a Backup information
*/
typedef struct st_BACKUP_ENTRY_INFO
{
	X509 * BackupSSL;			//!< The Backup certificate
	ASN1_UTF8STRING * Name;		//!< The Backup name
}BACKUP_ENTRY_INFO;
DECLARE_ASN1_ITEM(BACKUP_ENTRY_INFO)

/*!
	This structure respresents a list of Backups
*/
typedef struct st_BACKUPS_INFO
{
	STACK_OF(BACKUP_ENTRY_INFO) * list;		//!< The list
}BACKUPS_INFO;
DECLARE_ASN1_ITEM(BACKUPS_INFO)


/*!
	This enumeration defines the possible ACLs
*/
typedef enum en_ACL_TYPE
{
	ACL_TYPE_AUTHENTICATE_ON_ENTITY			= 1,		//!< The user is allowed to authenticate on the entity
	ACL_TYPE_MANAGE_USERS					= 2,		//!< The user is allowed to manage the users
	ACL_TYPE_MANAGE_ENTITIES				= 3,		//!< The user is allowed to manage the entities
	ACL_TYPE_VIEW_LOGS						= 4,		//!< The user is allowed to view the logs
	ACL_TYPE_MANAGE_ACLS					= 5,		//!< The user is allowed to manage the ACLs
	ACL_TYPE_MODIFY_PKI_STATUS				= 6,		//!< The user is allowed to set the PKI online/offline
	ACL_TYPE_IMPORT_CONF					= 7,		//!< The user is allowed to import a conf
	ACL_TYPE_SEND_ADMIN_MAIL				= 8,		//!< The user is allowed to send admin mails
	ACL_TYPE_CREATE_ROOT_CA					= 9,		//!< The user is allowed to create a root CA
	ACL_TYPE_CREATE_CHILD_CA				= 10,		//!< The user is allowed to create a child CA
	ACL_TYPE_CA_SIGN_CERT					= 11,		//!< The user is allowed to sign a certificate
	ACL_TYPE_CA_REVOKE_CERT					= 12,		//!< The user is allowed to revoke a certificate
	ACL_TYPE_CA_GENERATE_CRL				= 13,		//!< The user is allowed to generate a CRL
	ACL_TYPE_IMPORT_NEWPKI_OBJECTS			= 14,		//!< The user is allowed to imports newpki objects
	ACL_TYPE_RA_MANAGE_PROFILES				= 15,		//!< The user is allowed to manage the RA profiles
	ACL_TYPE_RA_DELEGATE_PROFILE_OWNERSHIP	= 16,		//!< The user is allowed to delegate a profiles ownership
	ACL_TYPE_RA_REQUEST_CERT				= 17,		//!< The user is allowed to request a certificate
	ACL_TYPE_RA_REVOKE_CERT					= 18,		//!< The user is allowed to requets a certificate revocation
	ACL_TYPE_EE_MANAGE_PROFILES				= 19		//!< The user is allowed to manage the EE profiles
}ACL_TYPE;


/*!
	This structure respresents a group
*/
typedef struct st_USERS_GROUP
{
	ASN1_UTF8STRING * name;					//!< The name of the group
	ASN1_INTEGER * serial;					//!< The unique id of the group (will be used for the ACL)
	STACK_OF(ASN1_INTEGER) * users_serial;	//!< The list of users being part of the group
}USERS_GROUP;
DECLARE_ASN1_ITEM(USERS_GROUP)


#define ACL_ENTRY_TYPE_USER		1		//!< The ACL concerns a user
#define ACL_ENTRY_TYPE_GROUP	2		//!< The ACL concerns a group

/*!
	This structure respresents an ACL entry
*/
typedef struct st_ACL_ENTRY
{
	ASN1_UTF8STRING * name;			//!< The name of user/group
	ASN1_INTEGER * type;			//!< The type of ACL 
	ASN1_INTEGER * serial;			//!< The serial of the user/group
	ASN1_BIT_STRING * acls;			//!< The list of allowed actions
}ACL_ENTRY;
DECLARE_ASN1_ITEM(ACL_ENTRY)

/*!
	This structure respresents an PKI admin
*/
typedef struct st_PKI_ADMIN_ENTRY
{
	ASN1_INTEGER * serial;			//!< The serial of its certificate
	ASN1_UTF8STRING * email;		//!< Its email address
}PKI_ADMIN_ENTRY;
DECLARE_ASN1_ITEM(PKI_ADMIN_ENTRY)


/*!
	This structure represents the list of pki admin
*/
typedef struct st_PKI_ADMIN_SERIALS
{
	STACK_OF(PKI_ADMIN_ENTRY) * admins;		//!< The list
}PKI_ADMIN_SERIALS;
DECLARE_ASN1_ITEM(PKI_ADMIN_SERIALS)



/*!
	This structure represents a complete ACL information for an entity
*/
typedef struct st_X509_ACL
{
	STACK_OF(ACL_ENTRY) *acl_entries;		//!< The list of ACLs
	STACK_OF(ASN1_INTEGER) *acl_type;		//!< The list supported ACLs by the entity
	STACK_OF(PKI_ADMIN_ENTRY) * AdminSerials;		//!< The list of PKI admins
}X509_ACL;
DECLARE_ASN1_ITEM(X509_ACL)


/*!
	This structure represents the CRLs for the PKI internal CA
*/
typedef struct st_INTERNAL_PKI_CRL
{
	X509_CRL * RootCaCrl;		//!< The CRL for the ROOT internal CA
	X509_CRL * UsersCaCrl;		//!< The CRL for the Users internal CA
	X509_CRL * EntitiesCaCrl;	//!< The CRL for the Entitis internal CA
	X509_CRL * OcspCaCrl;		//!< The CRL for the OCSP internal CA
	int references;
}INTERNAL_PKI_CRL;
DECLARE_ASN1_ITEM(INTERNAL_PKI_CRL)

/*!
	This structure represents the Certificates for the PKI internal CA
*/
typedef struct st_INTERNAL_PKI_CA
{
	X509 * RootCa;			//!< The certificate for the ROOT internal CA
	X509 * UsersCa;			//!< The certificate for the Users internal CA
	X509 * EntitiesCa;		//!< The certificate for the Entitis internal CA
	X509 * OcspCa;			//!< The certificate for the OCSP internal CA
	X509 * PkiCert;			//!< The certificate for the PKI entity
	int references;
}INTERNAL_PKI_CA;
DECLARE_ASN1_ITEM(INTERNAL_PKI_CA)


/*!
	This structure an audit entry
*/
typedef struct st_ENTITY_AUDIT_ENTRY
{
	ASN1_UTF8STRING * object;		//!< The object name
	ASN1_INTEGER * type;			//!< The log type
	ASN1_INTEGER * status;			//!< The log status
	ASN1_UTF8STRING * user;			//!< The user that generated the log
}ENTITY_AUDIT_ENTRY;
DECLARE_ASN1_ITEM(ENTITY_AUDIT_ENTRY)


/*!
	This structure represents the smtp configuration
*/
typedef struct st_EMAIL_CONF
{
	ASN1_UTF8STRING * server;		//!< The SMTP server
	ASN1_INTEGER * port;			//!< The SMTP port
}EMAIL_CONF;
DECLARE_ASN1_ITEM(EMAIL_CONF)


/*!
	This structure is an entity conf but not crypted for it
*/
typedef struct st_ENTITY_CONF
{
	ASN1_UTF8STRING * Name;						//!< The name of the entity
	ASN1_INTEGER * version;						//!< The version of the entity conf
	X509_ACL * acls;							//!< The ACLs for the entity
	ENTITY_CONF_BODY * body;					//!< The body of the entity conf
	STACK_OF(REP_ENTRY_INFO) * Repositories;	//!< The repositories list is exported as well
	RAS_INFO * Ras;								//!< The RAs list is exported as well
	CAS_INFO * Cas;								//!< The CAs list is exported as well
	KEYS_INFO * KeyStores;						//!< The Key stores list is exported as well
	EES_INFO * EES;								//!< The EEs list is exported as well
	BACKUPS_INFO * Backups;						//!< The Backups list is exported as well
	PUBS_INFO * Publications;					//!< The Publications list is exported as well
	INTERNAL_PKI_CRL * crls;					//!< The current CRLs for the internal CAs
	EMAIL_CONF * mail_conf;						//!< The mail conf
	STACK_OF(ENTITY_AUDIT_ENTRY) * audits;		//!< The audits
	STACK_OF(USERS_GROUP) * groups;				//!< The groups
}ENTITY_CONF;
DECLARE_ASN1_ITEM(ENTITY_CONF)


/*!
	This structure is an entity conf but not crypted for it
*/
typedef struct st_ENTITY_CONF_BETA4
{
	ASN1_UTF8STRING * Name;						//!< The name of the entity
	ASN1_INTEGER * version;						//!< The version of the entity conf
	X509_ACL * acls;							//!< The ACLs for the entity
	ENTITY_CONF_BODY * body;					//!< The body of the entity conf
	STACK_OF(REP_ENTRY_INFO) * Repositories;	//!< The repositories list is exported as well
	RAS_INFO * Ras;								//!< The RAs list is exported as well
	CAS_INFO * Cas;								//!< The CAs list is exported as well
	KEYS_INFO * KeyStores;						//!< The Key stores list is exported as well
	BACKUPS_INFO * Backups;						//!< The Backups list is exported as well
	PUBS_INFO * Publications;					//!< The Publications list is exported as well
	INTERNAL_PKI_CRL * crls;					//!< The current CRLs for the internal CAs
	EMAIL_CONF * mail_conf;						//!< The mail conf
	STACK_OF(ENTITY_AUDIT_ENTRY) * audits;		//!< The audits
	STACK_OF(USERS_GROUP) * groups;				//!< The groups
}ENTITY_CONF_BETA4;
DECLARE_ASN1_ITEM(ENTITY_CONF_BETA4)


/*!
	This structure represents an entity conf, the conf is crypted 
	for the destination entity and signed by us.
	The entity will decrypt its own conf and verify the signature.
*/
typedef struct st_ENTITY_CONF_CRYPTED
{
	X509_PUBKEY * recipient;			//!< The public key of the destination entity	
	ASN1_ENCRYPT_SIGN * crypted;		//!< The crypted and signed datas
	int references;
}ENTITY_CONF_CRYPTED;
DECLARE_ASN1_ITEM(ENTITY_CONF_CRYPTED)


/*!
	This structure represents a global PKI conf.
	Necessary for the signature generation of all the confs together.
*/
typedef struct st_EXPORTED_PKI_CONF_BODY
{
	ASN1_INTEGER * version;								//!< The version of the "alltogether" configuration
	STACK_OF(ENTITY_CONF_CRYPTED) * all_confs;			//!< All the crypted and signed confs
}EXPORTED_PKI_CONF_BODY;
DECLARE_ASN1_ITEM(EXPORTED_PKI_CONF_BODY)


/*!
	This structure represents the whole pki configuration for each
	entity, it is signed by the PKI entity priv key.
	This structure is sent to the repositories, they're
	in charge of verifying the "global" signature
*/
typedef struct st_EXPORTED_PKI_CONF
{
	EXPORTED_PKI_CONF_BODY * confs;			//!< The global conf
	X509_SIG * sig;							//!< The signature of the global conf
	STACK_OF(X509_PUBKEY) * rep_path;		//!< The list of all the repositories that have already processed this conf
}EXPORTED_PKI_CONF;
DECLARE_ASN1_ITEM(EXPORTED_PKI_CONF)


/*!
	This enumeration defines the list of internal CAs.
*/
typedef enum en_INTERNAL_CA_TYPE
{
	INTERNAL_CA_TYPE_ROOT = 1,		//!< The ROOT internal CA
	INTERNAL_CA_TYPE_ENTITY = 2,	//!< The Entities internal CA
	INTERNAL_CA_TYPE_USER = 3,		//!< The Users internal CA
	INTERNAL_CA_TYPE_OCSP = 4		//!< The OCSP internal CA
}INTERNAL_CA_TYPE;

/*!
	This structure represents a profile.
*/
typedef struct st_NEWPKI_PROFILE
{
	ASN1_INTEGER * EE_Validation;		//!< Should the profile be validated by an EE entity
	X509_NAME * dn;						//!< The DN of the profile
	ASN1_INTEGER * owner_group_serial;	//!< The serial of the owning group, or 0 if the owner is a user
	ASN1_UTF8STRING * ldap_uid;			//!< The LDAP UID
}NEWPKI_PROFILE;
DECLARE_ASN1_ITEM(NEWPKI_PROFILE)

#define NEWPKI_PROFILE_CERT_STATE_WAITING			1		//!< The certification request is waiting for the return from the CA
#define NEWPKI_PROFILE_CERT_STATE_ACTIVE			2		//!< The certificate is valid
#define NEWPKI_PROFILE_CERT_STATE_REVOKED			3		//!< The certificate is revoked
#define NEWPKI_PROFILE_CERT_STATE_ERROR				4		//!< There was an error during last request for this certificate
#define NEWPKI_PROFILE_CERT_STATE_WAITING_REV		5		//!< The revocation request is waiting for the return from the CA
#define NEWPKI_PROFILE_CERT_STATE_SUSPENDED			6		//!< The certificate is suspended
#define NEWPKI_PROFILE_CERT_STATE_WAITING_SUSP		7		//!< The suspension request is waiting for the return from the CA
#define NEWPKI_PROFILE_CERT_STATE_WAITING_UNSUSP	8		//!< The unsuspension request is waiting for the return from the CA


#define NEWPKI_PROFILE_CERT_TYPE_PKCS10 1			//!< The certificate request has been imported from a PKCS#10
#define NEWPKI_PROFILE_CERT_TYPE_PKCS12 2			//!< The certificate request has been generated by the server
#define NEWPKI_PROFILE_CERT_TYPE_SC		3			//!< The certificate request has been generated by the admin

/*!
	This structure represents a certificate.
*/
typedef struct st_NEWPKI_PROFILE_DATAS_CERT
{
	ASN1_INTEGER * Id;					//!< The internal ID of the cerificate
	ASN1_INTEGER * state;				//!< The state of the certificate
	ASN1_INTEGER * type;				//!< The type of certificate
	ASN1_UTF8STRING * ca_name;			//!< The name the certificate has been requested from
	ASN1_BIT_STRING * flags;			//!< The flags, see REQUEST_CERT_FLAG_SEND_ADMIN_NOTIFICATION, etc.
	ASN1_UTF8STRING * admin_mail;		//!< The email address of the admin that requested the certificate
	X509 * cert;						//!< The optional certificate
	ASN1_UTF8STRING * error;			//!< The optional errors
	PKCS12 *p12;						//!< The optional PKCS#12
	PKCS7 * p7b;						//!< The optional PKCS#7
}NEWPKI_PROFILE_DATAS_CERT;
DECLARE_ASN1_ITEM(NEWPKI_PROFILE_DATAS_CERT)

#define NEWPKI_PROFILE_STATE_WAITING_FOR_VALIDATION			1	//!< The profile hasn't been validated by the EE entity yet
#define NEWPKI_PROFILE_STATE_VALIDATED						2	//!< The profile has been validated
#define NEWPKI_PROFILE_STATE_WAITING_FOR_ADMIN_VALIDATION	3	//!< The profile hasn't been validated by an admin yet

/*!
	This structure represents a complete profile.
*/
typedef struct st_NEWPKI_PROFILE_DATAS
{
	ASN1_INTEGER * State;							//!< The state of the profile
	ASN1_INTEGER * Id;								//!< The internal ID of the profile
	ASN1_UTF8STRING * ee_id;						//!< The optional EE ID
	NEWPKI_PROFILE * Profile;						//!< The profile
	STACK_OF(NEWPKI_PROFILE_DATAS_CERT) * Certs;	//!< All the associated certificates
}NEWPKI_PROFILE_DATAS;
DECLARE_ASN1_ITEM(NEWPKI_PROFILE_DATAS)


/*!
	This structure represents a complete EE user.
*/
typedef struct st_NEWPKI_EE_USER
{
	ASN1_INTEGER * Id;								//!< The internal ID of the user
	ASN1_UTF8STRING * email;						//!< The email address of the user
	X509_NAME * dn;									//!< The DN of the user
	ASN1_INTEGER * ra_validated;					//!< The profile has been validated by the RA
	ASN1_INTEGER * activated;						//!< The profile has been activated by the user
	ASN1_INTEGER * last_send;						//!< Last time an activation request email was sent
	ASN1_INTEGER * send_count;						//!< The number of time an activation request email was sent
	STACK_OF(NEWPKI_PROFILE_DATAS_CERT) * Certs;	//!< All the associated certificates
}NEWPKI_EE_USER;
DECLARE_ASN1_ITEM(NEWPKI_EE_USER)


/*!
  This class is the representation of EMAIL_CONF
*/
class EmailConf : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EmailConf();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EmailConf(const EmailConf & other);

	/*! \brief This is the destructor.
	 */
	~EmailConf();


	/*! \brief This function converts a EMAIL_CONF to a EmailConf.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const EMAIL_CONF * Datas);

	/*! \brief This function converts a EmailConf to a EMAIL_CONF.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(EMAIL_CONF ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of EMAIL_CONF.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::port.
	 *  \param c_port [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_port(unsigned long c_port);

	/*! \brief This function returns ASN1_INTEGER::port.
	 *  \return The value of ASN1_INTEGER::port.
	 */
	unsigned long get_port() const;


	/*! \brief This function sets ASN1_UTF8STRING::server.
	 *  \param c_server [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_server(const mString & c_server);

	/*! \brief This function returns ASN1_UTF8STRING::server.
	 *  \return The value of ASN1_UTF8STRING::server.
	 */
	const mString & get_server() const;

	/*! \brief This function returns ASN1_UTF8STRING::server.
	 *  \return The value of ASN1_UTF8STRING::server.
	 */
	mString & get_server();


	/*! \brief This function copies a EmailConf to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EmailConf & other);
	/*! \brief This member is an empty instance of EmailConf.
	 */
	static EmailConf EmptyInstance;

private:
	unsigned long m_port;
	mString m_server;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PKI_ADMIN_ENTRY
*/
class PkiAdminEntry : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PkiAdminEntry();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PkiAdminEntry(const PkiAdminEntry & other);

	/*! \brief This is the destructor.
	 */
	~PkiAdminEntry();


	/*! \brief This function converts a PKI_ADMIN_ENTRY to a PkiAdminEntry.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PKI_ADMIN_ENTRY * Datas);

	/*! \brief This function converts a PkiAdminEntry to a PKI_ADMIN_ENTRY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PKI_ADMIN_ENTRY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PKI_ADMIN_ENTRY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::email.
	 *  \param c_email [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_email(const mString & c_email);

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	const mString & get_email() const;

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	mString & get_email();


	/*! \brief This function sets ASN1_INTEGER::serial.
	 *  \param c_serial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_serial(unsigned long c_serial);

	/*! \brief This function returns ASN1_INTEGER::serial.
	 *  \return The value of ASN1_INTEGER::serial.
	 */
	unsigned long get_serial() const;


	/*! \brief This function copies a PkiAdminEntry to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PkiAdminEntry & other);
	/*! \brief This member is an empty instance of PkiAdminEntry.
	 */
	static PkiAdminEntry EmptyInstance;

private:
	mString m_email;
	unsigned long m_serial;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_REP_0
*/
class EntityConfBodyRep0 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyRep0();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyRep0(const EntityConfBodyRep0 & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyRep0();


	/*! \brief This function converts a ENTITY_CONF_BODY_REP_0 to a EntityConfBodyRep0.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_REP_0 * Datas);

	/*! \brief This function converts a EntityConfBodyRep0 to a ENTITY_CONF_BODY_REP_0.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_REP_0 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_REP_0.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::Enabled.
	 *  \param c_enabled [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_enabled(unsigned long c_enabled);

	/*! \brief This function returns ASN1_INTEGER::Enabled.
	 *  \return The value of ASN1_INTEGER::Enabled.
	 */
	unsigned long get_enabled() const;


	/*! \brief This function copies a EntityConfBodyRep0 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyRep0 & other);
	/*! \brief This member is an empty instance of EntityConfBodyRep0.
	 */
	static EntityConfBodyRep0 EmptyInstance;

private:
	unsigned long m_enabled;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_REP
*/
class EntityConfBodyRep : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyRep();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyRep(const EntityConfBodyRep & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyRep();


	/*! \brief This function converts a ENTITY_CONF_BODY_REP to a EntityConfBodyRep.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_REP * Datas);

	/*! \brief This function converts a EntityConfBodyRep to a ENTITY_CONF_BODY_REP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_REP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_REP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets ENTITY_CONF_BODY_REP_0::d.conf0.
	 *  \param c_conf0 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf0(const EntityConfBodyRep0 & c_conf0);

	/*! \brief This function returns ENTITY_CONF_BODY_REP_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_REP_0::d.conf0.
	 */
	const EntityConfBodyRep0 & get_conf0() const;

	/*! \brief This function returns ENTITY_CONF_BODY_REP_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_REP_0::d.conf0.
	 */
	EntityConfBodyRep0 & get_conf0();


	/*! \brief This function copies a EntityConfBodyRep to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyRep & other);
	/*! \brief This member is an empty instance of EntityConfBodyRep.
	 */
	static EntityConfBodyRep EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	EntityConfBodyRep0 * m_conf0;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_PKI_0
*/
class EntityConfBodyPki0 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyPki0();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyPki0(const EntityConfBodyPki0 & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyPki0();


	/*! \brief This function converts a ENTITY_CONF_BODY_PKI_0 to a EntityConfBodyPki0.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_PKI_0 * Datas);

	/*! \brief This function converts a EntityConfBodyPki0 to a ENTITY_CONF_BODY_PKI_0.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_PKI_0 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_PKI_0.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::offline.
	 *  \param c_offline [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_offline(unsigned long c_offline);

	/*! \brief This function returns ASN1_INTEGER::offline.
	 *  \return The value of ASN1_INTEGER::offline.
	 */
	unsigned long get_offline() const;


	/*! \brief This function copies a EntityConfBodyPki0 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyPki0 & other);
	/*! \brief This member is an empty instance of EntityConfBodyPki0.
	 */
	static EntityConfBodyPki0 EmptyInstance;

private:
	unsigned long m_offline;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_PKI
*/
class EntityConfBodyPki : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyPki();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyPki(const EntityConfBodyPki & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyPki();


	/*! \brief This function converts a ENTITY_CONF_BODY_PKI to a EntityConfBodyPki.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_PKI * Datas);

	/*! \brief This function converts a EntityConfBodyPki to a ENTITY_CONF_BODY_PKI.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_PKI ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_PKI.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets ENTITY_CONF_BODY_PKI_0::d.conf0.
	 *  \param c_conf0 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf0(const EntityConfBodyPki0 & c_conf0);

	/*! \brief This function returns ENTITY_CONF_BODY_PKI_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_PKI_0::d.conf0.
	 */
	const EntityConfBodyPki0 & get_conf0() const;

	/*! \brief This function returns ENTITY_CONF_BODY_PKI_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_PKI_0::d.conf0.
	 */
	EntityConfBodyPki0 & get_conf0();


	/*! \brief This function copies a EntityConfBodyPki to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyPki & other);
	/*! \brief This member is an empty instance of EntityConfBodyPki.
	 */
	static EntityConfBodyPki EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	EntityConfBodyPki0 * m_conf0;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of EXTERNAL_ENTITY_PLUG
*/
class ExternalEntityPlug : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ExternalEntityPlug();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ExternalEntityPlug(const ExternalEntityPlug & other);

	/*! \brief This is the destructor.
	 */
	~ExternalEntityPlug();


	/*! \brief This function converts a EXTERNAL_ENTITY_PLUG to a ExternalEntityPlug.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const EXTERNAL_ENTITY_PLUG * Datas);

	/*! \brief This function converts a ExternalEntityPlug to a EXTERNAL_ENTITY_PLUG.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(EXTERNAL_ENTITY_PLUG ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of EXTERNAL_ENTITY_PLUG.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::LibraryPath.
	 *  \param c_librarypath [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_librarypath(const mString & c_librarypath);

	/*! \brief This function returns ASN1_UTF8STRING::LibraryPath.
	 *  \return The value of ASN1_UTF8STRING::LibraryPath.
	 */
	const mString & get_librarypath() const;

	/*! \brief This function returns ASN1_UTF8STRING::LibraryPath.
	 *  \return The value of ASN1_UTF8STRING::LibraryPath.
	 */
	mString & get_librarypath();


	/*! \brief This function sets PLUG_OPTION::Options.
	 *  \param c_options [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_options(const mVector< PlugOption > & c_options);

	/*! \brief This function returns PLUG_OPTION::Options.
	 *  \return The value of PLUG_OPTION::Options.
	 */
	const mVector< PlugOption > & get_options() const;

	/*! \brief This function returns PLUG_OPTION::Options.
	 *  \return The value of PLUG_OPTION::Options.
	 */
	mVector< PlugOption > & get_options();


	/*! \brief This function copies a ExternalEntityPlug to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ExternalEntityPlug & other);
	/*! \brief This member is an empty instance of ExternalEntityPlug.
	 */
	static ExternalEntityPlug EmptyInstance;

private:
	mString m_librarypath;
	mVector< PlugOption > m_options;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_CA_0
*/
class EntityConfBodyCa0 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyCa0();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyCa0(const EntityConfBodyCa0 & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyCa0();


	/*! \brief This function converts a ENTITY_CONF_BODY_CA_0 to a EntityConfBodyCa0.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_CA_0 * Datas);

	/*! \brief This function converts a EntityConfBodyCa0 to a ENTITY_CONF_BODY_CA_0.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_CA_0 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_CA_0.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets EXTENSION_VALUE::CrlExts.
	 *  \param c_crlexts [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crlexts(const mVector< ExtensionValue > & c_crlexts);

	/*! \brief This function returns EXTENSION_VALUE::CrlExts.
	 *  \return The value of EXTENSION_VALUE::CrlExts.
	 */
	const mVector< ExtensionValue > & get_crlexts() const;

	/*! \brief This function returns EXTENSION_VALUE::CrlExts.
	 *  \return The value of EXTENSION_VALUE::CrlExts.
	 */
	mVector< ExtensionValue > & get_crlexts();


	/*! \brief This function sets ASN1_INTEGER::CrlValidityDays.
	 *  \param c_crlvaliditydays [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crlvaliditydays(unsigned long c_crlvaliditydays);

	/*! \brief This function returns ASN1_INTEGER::CrlValidityDays.
	 *  \return The value of ASN1_INTEGER::CrlValidityDays.
	 */
	unsigned long get_crlvaliditydays() const;


	/*! \brief This function sets ASN1_INTEGER::CrlValidityHours.
	 *  \param c_crlvalidityhours [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crlvalidityhours(unsigned long c_crlvalidityhours);

	/*! \brief This function returns ASN1_INTEGER::CrlValidityHours.
	 *  \return The value of ASN1_INTEGER::CrlValidityHours.
	 */
	unsigned long get_crlvalidityhours() const;


	/*! \brief This function sets EXTENSION_VALUE::Exts.
	 *  \param c_exts [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_exts(const mVector< ExtensionValue > & c_exts);

	/*! \brief This function returns EXTENSION_VALUE::Exts.
	 *  \return The value of EXTENSION_VALUE::Exts.
	 */
	const mVector< ExtensionValue > & get_exts() const;

	/*! \brief This function returns EXTENSION_VALUE::Exts.
	 *  \return The value of EXTENSION_VALUE::Exts.
	 */
	mVector< ExtensionValue > & get_exts();


	/*! \brief This function sets EXTERNAL_ENTITY_PLUG::PlugCa.
	 *  \param c_plugca [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_plugca(const ExternalEntityPlug & c_plugca);

	/*! \brief This function returns EXTERNAL_ENTITY_PLUG::PlugCa.
	 *  \return The value of EXTERNAL_ENTITY_PLUG::PlugCa.
	 */
	const ExternalEntityPlug & get_plugca() const;

	/*! \brief This function returns EXTERNAL_ENTITY_PLUG::PlugCa.
	 *  \return The value of EXTERNAL_ENTITY_PLUG::PlugCa.
	 */
	ExternalEntityPlug & get_plugca();


	/*! \brief This function copies a EntityConfBodyCa0 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyCa0 & other);
	/*! \brief This member is an empty instance of EntityConfBodyCa0.
	 */
	static EntityConfBodyCa0 EmptyInstance;

private:
	mVector< ExtensionValue > m_crlexts;
	unsigned long m_crlvaliditydays;
	unsigned long m_crlvalidityhours;
	mVector< ExtensionValue > m_exts;
	ExternalEntityPlug m_plugca;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_CA_1
*/
class EntityConfBodyCa1 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyCa1();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyCa1(const EntityConfBodyCa1 & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyCa1();


	/*! \brief This function converts a ENTITY_CONF_BODY_CA_1 to a EntityConfBodyCa1.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_CA_1 * Datas);

	/*! \brief This function converts a EntityConfBodyCa1 to a ENTITY_CONF_BODY_CA_1.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_CA_1 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_CA_1.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets EXTENSION_VALUE::CrlExts.
	 *  \param c_crlexts [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crlexts(const mVector< ExtensionValue > & c_crlexts);

	/*! \brief This function returns EXTENSION_VALUE::CrlExts.
	 *  \return The value of EXTENSION_VALUE::CrlExts.
	 */
	const mVector< ExtensionValue > & get_crlexts() const;

	/*! \brief This function returns EXTENSION_VALUE::CrlExts.
	 *  \return The value of EXTENSION_VALUE::CrlExts.
	 */
	mVector< ExtensionValue > & get_crlexts();


	/*! \brief This function sets ASN1_INTEGER::CrlValidityDays.
	 *  \param c_crlvaliditydays [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crlvaliditydays(unsigned long c_crlvaliditydays);

	/*! \brief This function returns ASN1_INTEGER::CrlValidityDays.
	 *  \return The value of ASN1_INTEGER::CrlValidityDays.
	 */
	unsigned long get_crlvaliditydays() const;


	/*! \brief This function sets ASN1_INTEGER::CrlValidityHours.
	 *  \param c_crlvalidityhours [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crlvalidityhours(unsigned long c_crlvalidityhours);

	/*! \brief This function returns ASN1_INTEGER::CrlValidityHours.
	 *  \return The value of ASN1_INTEGER::CrlValidityHours.
	 */
	unsigned long get_crlvalidityhours() const;


	/*! \brief This function sets EXTENSION_VALUE::Exts.
	 *  \param c_exts [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_exts(const mVector< ExtensionValue > & c_exts);

	/*! \brief This function returns EXTENSION_VALUE::Exts.
	 *  \return The value of EXTENSION_VALUE::Exts.
	 */
	const mVector< ExtensionValue > & get_exts() const;

	/*! \brief This function returns EXTENSION_VALUE::Exts.
	 *  \return The value of EXTENSION_VALUE::Exts.
	 */
	mVector< ExtensionValue > & get_exts();


	/*! \brief This function sets ASN1_BIT_STRING::Flags.
	 *  \param c_flags [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_flags(const ASN1_BIT_STRING * c_flags);

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	const ASN1_BIT_STRING * get_flags() const;

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	ASN1_BIT_STRING * get_flags();


	/*! \brief This function sets EXTERNAL_ENTITY_PLUG::PlugCa.
	 *  \param c_plugca [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_plugca(const ExternalEntityPlug & c_plugca);

	/*! \brief This function returns EXTERNAL_ENTITY_PLUG::PlugCa.
	 *  \return The value of EXTERNAL_ENTITY_PLUG::PlugCa.
	 */
	const ExternalEntityPlug & get_plugca() const;

	/*! \brief This function returns EXTERNAL_ENTITY_PLUG::PlugCa.
	 *  \return The value of EXTERNAL_ENTITY_PLUG::PlugCa.
	 */
	ExternalEntityPlug & get_plugca();


	/*! \brief This function copies a EntityConfBodyCa1 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyCa1 & other);
	/*! \brief This member is an empty instance of EntityConfBodyCa1.
	 */
	static EntityConfBodyCa1 EmptyInstance;

private:
	mVector< ExtensionValue > m_crlexts;
	unsigned long m_crlvaliditydays;
	unsigned long m_crlvalidityhours;
	mVector< ExtensionValue > m_exts;
	ASN1_BIT_STRING * m_flags;
	ExternalEntityPlug m_plugca;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_CA
*/
class EntityConfBodyCa : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyCa();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyCa(const EntityConfBodyCa & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyCa();


	/*! \brief This function converts a ENTITY_CONF_BODY_CA to a EntityConfBodyCa.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_CA * Datas);

	/*! \brief This function converts a EntityConfBodyCa to a ENTITY_CONF_BODY_CA.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_CA ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_CA.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets ENTITY_CONF_BODY_CA_0::d.conf0.
	 *  \param c_conf0 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf0(const EntityConfBodyCa0 & c_conf0);

	/*! \brief This function returns ENTITY_CONF_BODY_CA_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_CA_0::d.conf0.
	 */
	const EntityConfBodyCa0 & get_conf0() const;

	/*! \brief This function returns ENTITY_CONF_BODY_CA_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_CA_0::d.conf0.
	 */
	EntityConfBodyCa0 & get_conf0();


	/*! \brief This function sets ENTITY_CONF_BODY_CA_1::d.conf1.
	 *  \param c_conf1 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf1(const EntityConfBodyCa1 & c_conf1);

	/*! \brief This function returns ENTITY_CONF_BODY_CA_1::d.conf1.
	 *  \return The value of ENTITY_CONF_BODY_CA_1::d.conf1.
	 */
	const EntityConfBodyCa1 & get_conf1() const;

	/*! \brief This function returns ENTITY_CONF_BODY_CA_1::d.conf1.
	 *  \return The value of ENTITY_CONF_BODY_CA_1::d.conf1.
	 */
	EntityConfBodyCa1 & get_conf1();


	/*! \brief This function copies a EntityConfBodyCa to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyCa & other);
	/*! \brief This member is an empty instance of EntityConfBodyCa.
	 */
	static EntityConfBodyCa EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	EntityConfBodyCa0 * m_conf0;
	EntityConfBodyCa1 * m_conf1;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_EE_0
*/
class EntityConfBodyEe0 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyEe0();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyEe0(const EntityConfBodyEe0 & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyEe0();


	/*! \brief This function converts a ENTITY_CONF_BODY_EE_0 to a EntityConfBodyEe0.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_EE_0 * Datas);

	/*! \brief This function converts a EntityConfBodyEe0 to a ENTITY_CONF_BODY_EE_0.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_EE_0 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_EE_0.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets DN_SPECS::Dn_Specs.
	 *  \param c_dnSpecs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dnSpecs(const mVector< DnSpecs > & c_dnSpecs);

	/*! \brief This function returns DN_SPECS::Dn_Specs.
	 *  \return The value of DN_SPECS::Dn_Specs.
	 */
	const mVector< DnSpecs > & get_dnSpecs() const;

	/*! \brief This function returns DN_SPECS::Dn_Specs.
	 *  \return The value of DN_SPECS::Dn_Specs.
	 */
	mVector< DnSpecs > & get_dnSpecs();


	/*! \brief This function sets ASN1_UTF8STRING::Dn_Val_Disclaimer.
	 *  \param c_dnValDisclaimer [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dnValDisclaimer(const mString & c_dnValDisclaimer);

	/*! \brief This function returns ASN1_UTF8STRING::Dn_Val_Disclaimer.
	 *  \return The value of ASN1_UTF8STRING::Dn_Val_Disclaimer.
	 */
	const mString & get_dnValDisclaimer() const;

	/*! \brief This function returns ASN1_UTF8STRING::Dn_Val_Disclaimer.
	 *  \return The value of ASN1_UTF8STRING::Dn_Val_Disclaimer.
	 */
	mString & get_dnValDisclaimer();


	/*! \brief This function sets ASN1_BIT_STRING::Flags.
	 *  \param c_flags [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_flags(const ASN1_BIT_STRING * c_flags);

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	const ASN1_BIT_STRING * get_flags() const;

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	ASN1_BIT_STRING * get_flags();


	/*! \brief This function sets ASN1_INTEGER::MinKeyLen.
	 *  \param c_minkeylen [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_minkeylen(unsigned long c_minkeylen);

	/*! \brief This function returns ASN1_INTEGER::MinKeyLen.
	 *  \return The value of ASN1_INTEGER::MinKeyLen.
	 */
	unsigned long get_minkeylen() const;


	/*! \brief This function sets ASN1_INTEGER::MinPwdLen.
	 *  \param c_minpwdlen [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_minpwdlen(unsigned long c_minpwdlen);

	/*! \brief This function returns ASN1_INTEGER::MinPwdLen.
	 *  \return The value of ASN1_INTEGER::MinPwdLen.
	 */
	unsigned long get_minpwdlen() const;


	/*! \brief This function sets POLICY_VALUE::Policies.
	 *  \param c_policies [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_policies(const mVector< PolicyValue > & c_policies);

	/*! \brief This function returns POLICY_VALUE::Policies.
	 *  \return The value of POLICY_VALUE::Policies.
	 */
	const mVector< PolicyValue > & get_policies() const;

	/*! \brief This function returns POLICY_VALUE::Policies.
	 *  \return The value of POLICY_VALUE::Policies.
	 */
	mVector< PolicyValue > & get_policies();


	/*! \brief This function sets ASN1_UTF8STRING::Site_Name.
	 *  \param c_siteName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_siteName(const mString & c_siteName);

	/*! \brief This function returns ASN1_UTF8STRING::Site_Name.
	 *  \return The value of ASN1_UTF8STRING::Site_Name.
	 */
	const mString & get_siteName() const;

	/*! \brief This function returns ASN1_UTF8STRING::Site_Name.
	 *  \return The value of ASN1_UTF8STRING::Site_Name.
	 */
	mString & get_siteName();


	/*! \brief This function sets ASN1_UTF8STRING::cas.
	 *  \param c_cas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cas(const mVector< mString > & c_cas);

	/*! \brief This function returns ASN1_UTF8STRING::cas.
	 *  \return The value of ASN1_UTF8STRING::cas.
	 */
	const mVector< mString > & get_cas() const;

	/*! \brief This function returns ASN1_UTF8STRING::cas.
	 *  \return The value of ASN1_UTF8STRING::cas.
	 */
	mVector< mString > & get_cas();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_attr_name.
	 *  \param c_ldapAttrName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapAttrName(const mString & c_ldapAttrName);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_attr_name.
	 *  \return The value of ASN1_UTF8STRING::ldap_attr_name.
	 */
	const mString & get_ldapAttrName() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_attr_name.
	 *  \return The value of ASN1_UTF8STRING::ldap_attr_name.
	 */
	mString & get_ldapAttrName();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_base.
	 *  \param c_ldapBase [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapBase(const mString & c_ldapBase);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_base.
	 *  \return The value of ASN1_UTF8STRING::ldap_base.
	 */
	const mString & get_ldapBase() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_base.
	 *  \return The value of ASN1_UTF8STRING::ldap_base.
	 */
	mString & get_ldapBase();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_filters.
	 *  \param c_ldapFilters [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapFilters(const mString & c_ldapFilters);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_filters.
	 *  \return The value of ASN1_UTF8STRING::ldap_filters.
	 */
	const mString & get_ldapFilters() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_filters.
	 *  \return The value of ASN1_UTF8STRING::ldap_filters.
	 */
	mString & get_ldapFilters();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_password.
	 *  \param c_ldapPassword [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapPassword(const mString & c_ldapPassword);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_password.
	 *  \return The value of ASN1_UTF8STRING::ldap_password.
	 */
	const mString & get_ldapPassword() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_password.
	 *  \return The value of ASN1_UTF8STRING::ldap_password.
	 */
	mString & get_ldapPassword();


	/*! \brief This function sets ASN1_INTEGER::ldap_port.
	 *  \param c_ldapPort [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapPort(unsigned long c_ldapPort);

	/*! \brief This function returns ASN1_INTEGER::ldap_port.
	 *  \return The value of ASN1_INTEGER::ldap_port.
	 */
	unsigned long get_ldapPort() const;


	/*! \brief This function sets ASN1_UTF8STRING::ldap_server.
	 *  \param c_ldapServer [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapServer(const mString & c_ldapServer);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_server.
	 *  \return The value of ASN1_UTF8STRING::ldap_server.
	 */
	const mString & get_ldapServer() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_server.
	 *  \return The value of ASN1_UTF8STRING::ldap_server.
	 */
	mString & get_ldapServer();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_username.
	 *  \param c_ldapUsername [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUsername(const mString & c_ldapUsername);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_username.
	 *  \return The value of ASN1_UTF8STRING::ldap_username.
	 */
	const mString & get_ldapUsername() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_username.
	 *  \return The value of ASN1_UTF8STRING::ldap_username.
	 */
	mString & get_ldapUsername();


	/*! \brief This function sets ASN1_INTEGER::ldap_utf8.
	 *  \param c_ldapUtf8 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUtf8(unsigned long c_ldapUtf8);

	/*! \brief This function returns ASN1_INTEGER::ldap_utf8.
	 *  \return The value of ASN1_INTEGER::ldap_utf8.
	 */
	unsigned long get_ldapUtf8() const;


	/*! \brief This function copies a EntityConfBodyEe0 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyEe0 & other);
	/*! \brief This member is an empty instance of EntityConfBodyEe0.
	 */
	static EntityConfBodyEe0 EmptyInstance;

private:
	mVector< DnSpecs > m_dnSpecs;
	mString m_dnValDisclaimer;
	ASN1_BIT_STRING * m_flags;
	unsigned long m_minkeylen;
	unsigned long m_minpwdlen;
	mVector< PolicyValue > m_policies;
	mString m_siteName;
	mVector< mString > m_cas;
	mString m_ldapAttrName;
	mString m_ldapBase;
	mString m_ldapFilters;
	mString m_ldapPassword;
	unsigned long m_ldapPort;
	mString m_ldapServer;
	mString m_ldapUsername;
	unsigned long m_ldapUtf8;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_EE
*/
class EntityConfBodyEe : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyEe();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyEe(const EntityConfBodyEe & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyEe();


	/*! \brief This function converts a ENTITY_CONF_BODY_EE to a EntityConfBodyEe.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_EE * Datas);

	/*! \brief This function converts a EntityConfBodyEe to a ENTITY_CONF_BODY_EE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_EE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_EE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets ENTITY_CONF_BODY_EE_0::d.conf0.
	 *  \param c_conf0 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf0(const EntityConfBodyEe0 & c_conf0);

	/*! \brief This function returns ENTITY_CONF_BODY_EE_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_EE_0::d.conf0.
	 */
	const EntityConfBodyEe0 & get_conf0() const;

	/*! \brief This function returns ENTITY_CONF_BODY_EE_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_EE_0::d.conf0.
	 */
	EntityConfBodyEe0 & get_conf0();


	/*! \brief This function copies a EntityConfBodyEe to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyEe & other);
	/*! \brief This member is an empty instance of EntityConfBodyEe.
	 */
	static EntityConfBodyEe EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	EntityConfBodyEe0 * m_conf0;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PUBLICATION_METHOD_INFO
*/
class PublicationMethodInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PublicationMethodInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PublicationMethodInfo(const PublicationMethodInfo & other);

	/*! \brief This is the destructor.
	 */
	~PublicationMethodInfo();


	/*! \brief This function converts a PUBLICATION_METHOD_INFO to a PublicationMethodInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PUBLICATION_METHOD_INFO * Datas);

	/*! \brief This function converts a PublicationMethodInfo to a PUBLICATION_METHOD_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PUBLICATION_METHOD_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PUBLICATION_METHOD_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::LibraryPath.
	 *  \param c_librarypath [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_librarypath(const mString & c_librarypath);

	/*! \brief This function returns ASN1_UTF8STRING::LibraryPath.
	 *  \return The value of ASN1_UTF8STRING::LibraryPath.
	 */
	const mString & get_librarypath() const;

	/*! \brief This function returns ASN1_UTF8STRING::LibraryPath.
	 *  \return The value of ASN1_UTF8STRING::LibraryPath.
	 */
	mString & get_librarypath();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function sets PLUG_OPTION::Options.
	 *  \param c_options [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_options(const mVector< PlugOption > & c_options);

	/*! \brief This function returns PLUG_OPTION::Options.
	 *  \return The value of PLUG_OPTION::Options.
	 */
	const mVector< PlugOption > & get_options() const;

	/*! \brief This function returns PLUG_OPTION::Options.
	 *  \return The value of PLUG_OPTION::Options.
	 */
	mVector< PlugOption > & get_options();


	/*! \brief This function sets ASN1_INTEGER::Type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::Type.
	 *  \return The value of ASN1_INTEGER::Type.
	 */
	unsigned long get_type() const;


	/*! \brief This function copies a PublicationMethodInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PublicationMethodInfo & other);
	/*! \brief This member is an empty instance of PublicationMethodInfo.
	 */
	static PublicationMethodInfo EmptyInstance;

private:
	mString m_librarypath;
	mString m_name;
	mVector< PlugOption > m_options;
	unsigned long m_type;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PUBLICATION_ENTRY
*/
class PublicationEntry : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PublicationEntry();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PublicationEntry(const PublicationEntry & other);

	/*! \brief This is the destructor.
	 */
	~PublicationEntry();


	/*! \brief This function converts a PUBLICATION_ENTRY to a PublicationEntry.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PUBLICATION_ENTRY * Datas);

	/*! \brief This function converts a PublicationEntry to a PUBLICATION_ENTRY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PUBLICATION_ENTRY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PUBLICATION_ENTRY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::CaName.
	 *  \param c_caname [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caname(const mString & c_caname);

	/*! \brief This function returns ASN1_UTF8STRING::CaName.
	 *  \return The value of ASN1_UTF8STRING::CaName.
	 */
	const mString & get_caname() const;

	/*! \brief This function returns ASN1_UTF8STRING::CaName.
	 *  \return The value of ASN1_UTF8STRING::CaName.
	 */
	mString & get_caname();


	/*! \brief This function sets PUBLICATION_METHOD_INFO::Methods.
	 *  \param c_methods [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_methods(const mVector< PublicationMethodInfo > & c_methods);

	/*! \brief This function returns PUBLICATION_METHOD_INFO::Methods.
	 *  \return The value of PUBLICATION_METHOD_INFO::Methods.
	 */
	const mVector< PublicationMethodInfo > & get_methods() const;

	/*! \brief This function returns PUBLICATION_METHOD_INFO::Methods.
	 *  \return The value of PUBLICATION_METHOD_INFO::Methods.
	 */
	mVector< PublicationMethodInfo > & get_methods();


	/*! \brief This function copies a PublicationEntry to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PublicationEntry & other);
	/*! \brief This member is an empty instance of PublicationEntry.
	 */
	static PublicationEntry EmptyInstance;

private:
	mString m_caname;
	mVector< PublicationMethodInfo > m_methods;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_PUB_0
*/
class EntityConfBodyPub0 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyPub0();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyPub0(const EntityConfBodyPub0 & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyPub0();


	/*! \brief This function converts a ENTITY_CONF_BODY_PUB_0 to a EntityConfBodyPub0.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_PUB_0 * Datas);

	/*! \brief This function converts a EntityConfBodyPub0 to a ENTITY_CONF_BODY_PUB_0.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_PUB_0 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_PUB_0.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::BindAddress.
	 *  \param c_bindaddress [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_bindaddress(const mString & c_bindaddress);

	/*! \brief This function returns ASN1_UTF8STRING::BindAddress.
	 *  \return The value of ASN1_UTF8STRING::BindAddress.
	 */
	const mString & get_bindaddress() const;

	/*! \brief This function returns ASN1_UTF8STRING::BindAddress.
	 *  \return The value of ASN1_UTF8STRING::BindAddress.
	 */
	mString & get_bindaddress();


	/*! \brief This function sets ASN1_INTEGER::OcspEnabled.
	 *  \param c_ocspenabled [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ocspenabled(unsigned long c_ocspenabled);

	/*! \brief This function returns ASN1_INTEGER::OcspEnabled.
	 *  \return The value of ASN1_INTEGER::OcspEnabled.
	 */
	unsigned long get_ocspenabled() const;


	/*! \brief This function sets ASN1_INTEGER::Port.
	 *  \param c_port [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_port(unsigned long c_port);

	/*! \brief This function returns ASN1_INTEGER::Port.
	 *  \return The value of ASN1_INTEGER::Port.
	 */
	unsigned long get_port() const;


	/*! \brief This function sets PUBLICATION_ENTRY::Publications.
	 *  \param c_publications [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_publications(const mVector< PublicationEntry > & c_publications);

	/*! \brief This function returns PUBLICATION_ENTRY::Publications.
	 *  \return The value of PUBLICATION_ENTRY::Publications.
	 */
	const mVector< PublicationEntry > & get_publications() const;

	/*! \brief This function returns PUBLICATION_ENTRY::Publications.
	 *  \return The value of PUBLICATION_ENTRY::Publications.
	 */
	mVector< PublicationEntry > & get_publications();


	/*! \brief This function copies a EntityConfBodyPub0 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyPub0 & other);
	/*! \brief This member is an empty instance of EntityConfBodyPub0.
	 */
	static EntityConfBodyPub0 EmptyInstance;

private:
	mString m_bindaddress;
	unsigned long m_ocspenabled;
	unsigned long m_port;
	mVector< PublicationEntry > m_publications;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_PUB
*/
class EntityConfBodyPub : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyPub();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyPub(const EntityConfBodyPub & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyPub();


	/*! \brief This function converts a ENTITY_CONF_BODY_PUB to a EntityConfBodyPub.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_PUB * Datas);

	/*! \brief This function converts a EntityConfBodyPub to a ENTITY_CONF_BODY_PUB.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_PUB ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_PUB.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets ENTITY_CONF_BODY_PUB_0::d.conf0.
	 *  \param c_conf0 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf0(const EntityConfBodyPub0 & c_conf0);

	/*! \brief This function returns ENTITY_CONF_BODY_PUB_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_PUB_0::d.conf0.
	 */
	const EntityConfBodyPub0 & get_conf0() const;

	/*! \brief This function returns ENTITY_CONF_BODY_PUB_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_PUB_0::d.conf0.
	 */
	EntityConfBodyPub0 & get_conf0();


	/*! \brief This function copies a EntityConfBodyPub to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyPub & other);
	/*! \brief This member is an empty instance of EntityConfBodyPub.
	 */
	static EntityConfBodyPub EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	EntityConfBodyPub0 * m_conf0;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_RA_0
*/
class EntityConfBodyRa0 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyRa0();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyRa0(const EntityConfBodyRa0 & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyRa0();


	/*! \brief This function converts a ENTITY_CONF_BODY_RA_0 to a EntityConfBodyRa0.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_RA_0 * Datas);

	/*! \brief This function converts a EntityConfBodyRa0 to a ENTITY_CONF_BODY_RA_0.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_RA_0 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_RA_0.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets DN_SPECS::DnSpecs.
	 *  \param c_dnspecs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dnspecs(const mVector< DnSpecs > & c_dnspecs);

	/*! \brief This function returns DN_SPECS::DnSpecs.
	 *  \return The value of DN_SPECS::DnSpecs.
	 */
	const mVector< DnSpecs > & get_dnspecs() const;

	/*! \brief This function returns DN_SPECS::DnSpecs.
	 *  \return The value of DN_SPECS::DnSpecs.
	 */
	mVector< DnSpecs > & get_dnspecs();


	/*! \brief This function sets ASN1_BIT_STRING::Flags.
	 *  \param c_flags [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_flags(const ASN1_BIT_STRING * c_flags);

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	const ASN1_BIT_STRING * get_flags() const;

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	ASN1_BIT_STRING * get_flags();


	/*! \brief This function sets ASN1_INTEGER::MinKeyLen.
	 *  \param c_minkeylen [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_minkeylen(unsigned long c_minkeylen);

	/*! \brief This function returns ASN1_INTEGER::MinKeyLen.
	 *  \return The value of ASN1_INTEGER::MinKeyLen.
	 */
	unsigned long get_minkeylen() const;


	/*! \brief This function sets ASN1_INTEGER::MinPwdLen.
	 *  \param c_minpwdlen [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_minpwdlen(unsigned long c_minpwdlen);

	/*! \brief This function returns ASN1_INTEGER::MinPwdLen.
	 *  \return The value of ASN1_INTEGER::MinPwdLen.
	 */
	unsigned long get_minpwdlen() const;


	/*! \brief This function sets POLICY_VALUE::Policies.
	 *  \param c_policies [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_policies(const mVector< PolicyValue > & c_policies);

	/*! \brief This function returns POLICY_VALUE::Policies.
	 *  \return The value of POLICY_VALUE::Policies.
	 */
	const mVector< PolicyValue > & get_policies() const;

	/*! \brief This function returns POLICY_VALUE::Policies.
	 *  \return The value of POLICY_VALUE::Policies.
	 */
	mVector< PolicyValue > & get_policies();


	/*! \brief This function sets ASN1_INTEGER::default_validity.
	 *  \param c_defaultValidity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_defaultValidity(unsigned long c_defaultValidity);

	/*! \brief This function returns ASN1_INTEGER::default_validity.
	 *  \return The value of ASN1_INTEGER::default_validity.
	 */
	unsigned long get_defaultValidity() const;


	/*! \brief This function sets ASN1_UTF8STRING::ldap_attr_name.
	 *  \param c_ldapAttrName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapAttrName(const mString & c_ldapAttrName);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_attr_name.
	 *  \return The value of ASN1_UTF8STRING::ldap_attr_name.
	 */
	const mString & get_ldapAttrName() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_attr_name.
	 *  \return The value of ASN1_UTF8STRING::ldap_attr_name.
	 */
	mString & get_ldapAttrName();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_base.
	 *  \param c_ldapBase [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapBase(const mString & c_ldapBase);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_base.
	 *  \return The value of ASN1_UTF8STRING::ldap_base.
	 */
	const mString & get_ldapBase() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_base.
	 *  \return The value of ASN1_UTF8STRING::ldap_base.
	 */
	mString & get_ldapBase();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_filters.
	 *  \param c_ldapFilters [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapFilters(const mString & c_ldapFilters);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_filters.
	 *  \return The value of ASN1_UTF8STRING::ldap_filters.
	 */
	const mString & get_ldapFilters() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_filters.
	 *  \return The value of ASN1_UTF8STRING::ldap_filters.
	 */
	mString & get_ldapFilters();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_password.
	 *  \param c_ldapPassword [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapPassword(const mString & c_ldapPassword);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_password.
	 *  \return The value of ASN1_UTF8STRING::ldap_password.
	 */
	const mString & get_ldapPassword() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_password.
	 *  \return The value of ASN1_UTF8STRING::ldap_password.
	 */
	mString & get_ldapPassword();


	/*! \brief This function sets ASN1_INTEGER::ldap_port.
	 *  \param c_ldapPort [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapPort(unsigned long c_ldapPort);

	/*! \brief This function returns ASN1_INTEGER::ldap_port.
	 *  \return The value of ASN1_INTEGER::ldap_port.
	 */
	unsigned long get_ldapPort() const;


	/*! \brief This function sets ASN1_INTEGER::ldap_profiles_group.
	 *  \param c_ldapProfilesGroup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapProfilesGroup(unsigned long c_ldapProfilesGroup);

	/*! \brief This function returns ASN1_INTEGER::ldap_profiles_group.
	 *  \return The value of ASN1_INTEGER::ldap_profiles_group.
	 */
	unsigned long get_ldapProfilesGroup() const;


	/*! \brief This function sets ASN1_UTF8STRING::ldap_server.
	 *  \param c_ldapServer [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapServer(const mString & c_ldapServer);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_server.
	 *  \return The value of ASN1_UTF8STRING::ldap_server.
	 */
	const mString & get_ldapServer() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_server.
	 *  \return The value of ASN1_UTF8STRING::ldap_server.
	 */
	mString & get_ldapServer();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_username.
	 *  \param c_ldapUsername [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUsername(const mString & c_ldapUsername);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_username.
	 *  \return The value of ASN1_UTF8STRING::ldap_username.
	 */
	const mString & get_ldapUsername() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_username.
	 *  \return The value of ASN1_UTF8STRING::ldap_username.
	 */
	mString & get_ldapUsername();


	/*! \brief This function sets ASN1_INTEGER::ldap_utf8.
	 *  \param c_ldapUtf8 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUtf8(unsigned long c_ldapUtf8);

	/*! \brief This function returns ASN1_INTEGER::ldap_utf8.
	 *  \return The value of ASN1_INTEGER::ldap_utf8.
	 */
	unsigned long get_ldapUtf8() const;


	/*! \brief This function copies a EntityConfBodyRa0 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyRa0 & other);
	/*! \brief This member is an empty instance of EntityConfBodyRa0.
	 */
	static EntityConfBodyRa0 EmptyInstance;

private:
	mVector< DnSpecs > m_dnspecs;
	ASN1_BIT_STRING * m_flags;
	unsigned long m_minkeylen;
	unsigned long m_minpwdlen;
	mVector< PolicyValue > m_policies;
	unsigned long m_defaultValidity;
	mString m_ldapAttrName;
	mString m_ldapBase;
	mString m_ldapFilters;
	mString m_ldapPassword;
	unsigned long m_ldapPort;
	unsigned long m_ldapProfilesGroup;
	mString m_ldapServer;
	mString m_ldapUsername;
	unsigned long m_ldapUtf8;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_RA_1
*/
class EntityConfBodyRa1 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyRa1();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyRa1(const EntityConfBodyRa1 & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyRa1();


	/*! \brief This function converts a ENTITY_CONF_BODY_RA_1 to a EntityConfBodyRa1.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_RA_1 * Datas);

	/*! \brief This function converts a EntityConfBodyRa1 to a ENTITY_CONF_BODY_RA_1.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_RA_1 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_RA_1.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets DN_SPECS::DnSpecs.
	 *  \param c_dnspecs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dnspecs(const mVector< DnSpecs > & c_dnspecs);

	/*! \brief This function returns DN_SPECS::DnSpecs.
	 *  \return The value of DN_SPECS::DnSpecs.
	 */
	const mVector< DnSpecs > & get_dnspecs() const;

	/*! \brief This function returns DN_SPECS::DnSpecs.
	 *  \return The value of DN_SPECS::DnSpecs.
	 */
	mVector< DnSpecs > & get_dnspecs();


	/*! \brief This function sets ASN1_BIT_STRING::Flags.
	 *  \param c_flags [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_flags(const ASN1_BIT_STRING * c_flags);

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	const ASN1_BIT_STRING * get_flags() const;

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	ASN1_BIT_STRING * get_flags();


	/*! \brief This function sets ASN1_INTEGER::MinKeyLen.
	 *  \param c_minkeylen [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_minkeylen(unsigned long c_minkeylen);

	/*! \brief This function returns ASN1_INTEGER::MinKeyLen.
	 *  \return The value of ASN1_INTEGER::MinKeyLen.
	 */
	unsigned long get_minkeylen() const;


	/*! \brief This function sets ASN1_INTEGER::MinPwdLen.
	 *  \param c_minpwdlen [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_minpwdlen(unsigned long c_minpwdlen);

	/*! \brief This function returns ASN1_INTEGER::MinPwdLen.
	 *  \return The value of ASN1_INTEGER::MinPwdLen.
	 */
	unsigned long get_minpwdlen() const;


	/*! \brief This function sets POLICY_VALUE::Policies.
	 *  \param c_policies [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_policies(const mVector< PolicyValue > & c_policies);

	/*! \brief This function returns POLICY_VALUE::Policies.
	 *  \return The value of POLICY_VALUE::Policies.
	 */
	const mVector< PolicyValue > & get_policies() const;

	/*! \brief This function returns POLICY_VALUE::Policies.
	 *  \return The value of POLICY_VALUE::Policies.
	 */
	mVector< PolicyValue > & get_policies();


	/*! \brief This function sets ASN1_INTEGER::default_validity.
	 *  \param c_defaultValidity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_defaultValidity(unsigned long c_defaultValidity);

	/*! \brief This function returns ASN1_INTEGER::default_validity.
	 *  \return The value of ASN1_INTEGER::default_validity.
	 */
	unsigned long get_defaultValidity() const;


	/*! \brief This function sets ASN1_INTEGER::ee_profiles_group.
	 *  \param c_eeProfilesGroup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeProfilesGroup(unsigned long c_eeProfilesGroup);

	/*! \brief This function returns ASN1_INTEGER::ee_profiles_group.
	 *  \return The value of ASN1_INTEGER::ee_profiles_group.
	 */
	unsigned long get_eeProfilesGroup() const;


	/*! \brief This function sets ASN1_UTF8STRING::ldap_attr_name.
	 *  \param c_ldapAttrName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapAttrName(const mString & c_ldapAttrName);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_attr_name.
	 *  \return The value of ASN1_UTF8STRING::ldap_attr_name.
	 */
	const mString & get_ldapAttrName() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_attr_name.
	 *  \return The value of ASN1_UTF8STRING::ldap_attr_name.
	 */
	mString & get_ldapAttrName();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_base.
	 *  \param c_ldapBase [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapBase(const mString & c_ldapBase);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_base.
	 *  \return The value of ASN1_UTF8STRING::ldap_base.
	 */
	const mString & get_ldapBase() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_base.
	 *  \return The value of ASN1_UTF8STRING::ldap_base.
	 */
	mString & get_ldapBase();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_filters.
	 *  \param c_ldapFilters [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapFilters(const mString & c_ldapFilters);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_filters.
	 *  \return The value of ASN1_UTF8STRING::ldap_filters.
	 */
	const mString & get_ldapFilters() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_filters.
	 *  \return The value of ASN1_UTF8STRING::ldap_filters.
	 */
	mString & get_ldapFilters();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_password.
	 *  \param c_ldapPassword [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapPassword(const mString & c_ldapPassword);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_password.
	 *  \return The value of ASN1_UTF8STRING::ldap_password.
	 */
	const mString & get_ldapPassword() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_password.
	 *  \return The value of ASN1_UTF8STRING::ldap_password.
	 */
	mString & get_ldapPassword();


	/*! \brief This function sets ASN1_INTEGER::ldap_port.
	 *  \param c_ldapPort [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapPort(unsigned long c_ldapPort);

	/*! \brief This function returns ASN1_INTEGER::ldap_port.
	 *  \return The value of ASN1_INTEGER::ldap_port.
	 */
	unsigned long get_ldapPort() const;


	/*! \brief This function sets ASN1_INTEGER::ldap_profiles_group.
	 *  \param c_ldapProfilesGroup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapProfilesGroup(unsigned long c_ldapProfilesGroup);

	/*! \brief This function returns ASN1_INTEGER::ldap_profiles_group.
	 *  \return The value of ASN1_INTEGER::ldap_profiles_group.
	 */
	unsigned long get_ldapProfilesGroup() const;


	/*! \brief This function sets ASN1_UTF8STRING::ldap_server.
	 *  \param c_ldapServer [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapServer(const mString & c_ldapServer);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_server.
	 *  \return The value of ASN1_UTF8STRING::ldap_server.
	 */
	const mString & get_ldapServer() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_server.
	 *  \return The value of ASN1_UTF8STRING::ldap_server.
	 */
	mString & get_ldapServer();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_username.
	 *  \param c_ldapUsername [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUsername(const mString & c_ldapUsername);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_username.
	 *  \return The value of ASN1_UTF8STRING::ldap_username.
	 */
	const mString & get_ldapUsername() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_username.
	 *  \return The value of ASN1_UTF8STRING::ldap_username.
	 */
	mString & get_ldapUsername();


	/*! \brief This function sets ASN1_INTEGER::ldap_utf8.
	 *  \param c_ldapUtf8 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUtf8(unsigned long c_ldapUtf8);

	/*! \brief This function returns ASN1_INTEGER::ldap_utf8.
	 *  \return The value of ASN1_INTEGER::ldap_utf8.
	 */
	unsigned long get_ldapUtf8() const;


	/*! \brief This function copies a EntityConfBodyRa1 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyRa1 & other);
	/*! \brief This member is an empty instance of EntityConfBodyRa1.
	 */
	static EntityConfBodyRa1 EmptyInstance;

private:
	mVector< DnSpecs > m_dnspecs;
	ASN1_BIT_STRING * m_flags;
	unsigned long m_minkeylen;
	unsigned long m_minpwdlen;
	mVector< PolicyValue > m_policies;
	unsigned long m_defaultValidity;
	unsigned long m_eeProfilesGroup;
	mString m_ldapAttrName;
	mString m_ldapBase;
	mString m_ldapFilters;
	mString m_ldapPassword;
	unsigned long m_ldapPort;
	unsigned long m_ldapProfilesGroup;
	mString m_ldapServer;
	mString m_ldapUsername;
	unsigned long m_ldapUtf8;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY_RA
*/
class EntityConfBodyRa : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBodyRa();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBodyRa(const EntityConfBodyRa & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBodyRa();


	/*! \brief This function converts a ENTITY_CONF_BODY_RA to a EntityConfBodyRa.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY_RA * Datas);

	/*! \brief This function converts a EntityConfBodyRa to a ENTITY_CONF_BODY_RA.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY_RA ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY_RA.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets ENTITY_CONF_BODY_RA_0::d.conf0.
	 *  \param c_conf0 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf0(const EntityConfBodyRa0 & c_conf0);

	/*! \brief This function returns ENTITY_CONF_BODY_RA_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_RA_0::d.conf0.
	 */
	const EntityConfBodyRa0 & get_conf0() const;

	/*! \brief This function returns ENTITY_CONF_BODY_RA_0::d.conf0.
	 *  \return The value of ENTITY_CONF_BODY_RA_0::d.conf0.
	 */
	EntityConfBodyRa0 & get_conf0();


	/*! \brief This function sets ENTITY_CONF_BODY_RA_1::d.conf1.
	 *  \param c_conf1 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf1(const EntityConfBodyRa1 & c_conf1);

	/*! \brief This function returns ENTITY_CONF_BODY_RA_1::d.conf1.
	 *  \return The value of ENTITY_CONF_BODY_RA_1::d.conf1.
	 */
	const EntityConfBodyRa1 & get_conf1() const;

	/*! \brief This function returns ENTITY_CONF_BODY_RA_1::d.conf1.
	 *  \return The value of ENTITY_CONF_BODY_RA_1::d.conf1.
	 */
	EntityConfBodyRa1 & get_conf1();


	/*! \brief This function copies a EntityConfBodyRa to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBodyRa & other);
	/*! \brief This member is an empty instance of EntityConfBodyRa.
	 */
	static EntityConfBodyRa EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	EntityConfBodyRa0 * m_conf0;
	EntityConfBodyRa1 * m_conf1;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BODY
*/
class EntityConfBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBody(const EntityConfBody & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBody();


	/*! \brief This function converts a ENTITY_CONF_BODY to a EntityConfBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BODY * Datas);

	/*! \brief This function converts a EntityConfBody to a ENTITY_CONF_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets ENTITY_CONF_BODY_CA::d.ca_conf.
	 *  \param c_caConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caConf(const EntityConfBodyCa & c_caConf);

	/*! \brief This function returns ENTITY_CONF_BODY_CA::d.ca_conf.
	 *  \return The value of ENTITY_CONF_BODY_CA::d.ca_conf.
	 */
	const EntityConfBodyCa & get_caConf() const;

	/*! \brief This function returns ENTITY_CONF_BODY_CA::d.ca_conf.
	 *  \return The value of ENTITY_CONF_BODY_CA::d.ca_conf.
	 */
	EntityConfBodyCa & get_caConf();


	/*! \brief This function sets ASN1_OCTET_STRING::d.conf.
	 *  \param c_conf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf(const Asn1OctetString & c_conf);

	/*! \brief This function returns ASN1_OCTET_STRING::d.conf.
	 *  \return The value of ASN1_OCTET_STRING::d.conf.
	 */
	const Asn1OctetString & get_conf() const;

	/*! \brief This function returns ASN1_OCTET_STRING::d.conf.
	 *  \return The value of ASN1_OCTET_STRING::d.conf.
	 */
	Asn1OctetString & get_conf();


	/*! \brief This function sets ENTITY_CONF_BODY_EE::d.ee_conf.
	 *  \param c_eeConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeConf(const EntityConfBodyEe & c_eeConf);

	/*! \brief This function returns ENTITY_CONF_BODY_EE::d.ee_conf.
	 *  \return The value of ENTITY_CONF_BODY_EE::d.ee_conf.
	 */
	const EntityConfBodyEe & get_eeConf() const;

	/*! \brief This function returns ENTITY_CONF_BODY_EE::d.ee_conf.
	 *  \return The value of ENTITY_CONF_BODY_EE::d.ee_conf.
	 */
	EntityConfBodyEe & get_eeConf();


	/*! \brief This function sets ENTITY_CONF_BODY_PKI::d.pki_conf.
	 *  \param c_pkiConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pkiConf(const EntityConfBodyPki & c_pkiConf);

	/*! \brief This function returns ENTITY_CONF_BODY_PKI::d.pki_conf.
	 *  \return The value of ENTITY_CONF_BODY_PKI::d.pki_conf.
	 */
	const EntityConfBodyPki & get_pkiConf() const;

	/*! \brief This function returns ENTITY_CONF_BODY_PKI::d.pki_conf.
	 *  \return The value of ENTITY_CONF_BODY_PKI::d.pki_conf.
	 */
	EntityConfBodyPki & get_pkiConf();


	/*! \brief This function sets ENTITY_CONF_BODY_PUB::d.pub_conf.
	 *  \param c_pubConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pubConf(const EntityConfBodyPub & c_pubConf);

	/*! \brief This function returns ENTITY_CONF_BODY_PUB::d.pub_conf.
	 *  \return The value of ENTITY_CONF_BODY_PUB::d.pub_conf.
	 */
	const EntityConfBodyPub & get_pubConf() const;

	/*! \brief This function returns ENTITY_CONF_BODY_PUB::d.pub_conf.
	 *  \return The value of ENTITY_CONF_BODY_PUB::d.pub_conf.
	 */
	EntityConfBodyPub & get_pubConf();


	/*! \brief This function sets ENTITY_CONF_BODY_RA::d.ra_conf.
	 *  \param c_raConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_raConf(const EntityConfBodyRa & c_raConf);

	/*! \brief This function returns ENTITY_CONF_BODY_RA::d.ra_conf.
	 *  \return The value of ENTITY_CONF_BODY_RA::d.ra_conf.
	 */
	const EntityConfBodyRa & get_raConf() const;

	/*! \brief This function returns ENTITY_CONF_BODY_RA::d.ra_conf.
	 *  \return The value of ENTITY_CONF_BODY_RA::d.ra_conf.
	 */
	EntityConfBodyRa & get_raConf();


	/*! \brief This function sets ENTITY_CONF_BODY_REP::d.rep_conf.
	 *  \param c_repConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repConf(const EntityConfBodyRep & c_repConf);

	/*! \brief This function returns ENTITY_CONF_BODY_REP::d.rep_conf.
	 *  \return The value of ENTITY_CONF_BODY_REP::d.rep_conf.
	 */
	const EntityConfBodyRep & get_repConf() const;

	/*! \brief This function returns ENTITY_CONF_BODY_REP::d.rep_conf.
	 *  \return The value of ENTITY_CONF_BODY_REP::d.rep_conf.
	 */
	EntityConfBodyRep & get_repConf();


	/*! \brief This function copies a EntityConfBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBody & other);
	/*! \brief This member is an empty instance of EntityConfBody.
	 */
	static EntityConfBody EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	EntityConfBodyCa * m_caConf;
	Asn1OctetString * m_conf;
	EntityConfBodyEe * m_eeConf;
	EntityConfBodyPki * m_pkiConf;
	EntityConfBodyPub * m_pubConf;
	EntityConfBodyRa * m_raConf;
	EntityConfBodyRep * m_repConf;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of USERS_GROUP
*/
class UsersGroup : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	UsersGroup();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	UsersGroup(const UsersGroup & other);

	/*! \brief This is the destructor.
	 */
	~UsersGroup();


	/*! \brief This function converts a USERS_GROUP to a UsersGroup.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const USERS_GROUP * Datas);

	/*! \brief This function converts a UsersGroup to a USERS_GROUP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(USERS_GROUP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of USERS_GROUP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	mString & get_name();


	/*! \brief This function sets ASN1_INTEGER::serial.
	 *  \param c_serial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_serial(unsigned long c_serial);

	/*! \brief This function returns ASN1_INTEGER::serial.
	 *  \return The value of ASN1_INTEGER::serial.
	 */
	unsigned long get_serial() const;


	/*! \brief This function sets ASN1_INTEGER::users_serial.
	 *  \param c_usersSerial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_usersSerial(const mVector< unsigned long > & c_usersSerial);

	/*! \brief This function returns ASN1_INTEGER::users_serial.
	 *  \return The value of ASN1_INTEGER::users_serial.
	 */
	const mVector< unsigned long > & get_usersSerial() const;

	/*! \brief This function returns ASN1_INTEGER::users_serial.
	 *  \return The value of ASN1_INTEGER::users_serial.
	 */
	mVector< unsigned long > & get_usersSerial();


	/*! \brief This function copies a UsersGroup to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const UsersGroup & other);
	/*! \brief This member is an empty instance of UsersGroup.
	 */
	static UsersGroup EmptyInstance;

private:
	mString m_name;
	unsigned long m_serial;
	mVector< unsigned long > m_usersSerial;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ACL_ENTRY
*/
class AclEntry : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	AclEntry();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	AclEntry(const AclEntry & other);

	/*! \brief This is the destructor.
	 */
	~AclEntry();


	/*! \brief This function converts a ACL_ENTRY to a AclEntry.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ACL_ENTRY * Datas);

	/*! \brief This function converts a AclEntry to a ACL_ENTRY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ACL_ENTRY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ACL_ENTRY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_BIT_STRING::acls.
	 *  \param c_acls [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_acls(const ASN1_BIT_STRING * c_acls);

	/*! \brief This function returns ASN1_BIT_STRING::acls.
	 *  \return The value of ASN1_BIT_STRING::acls.
	 */
	const ASN1_BIT_STRING * get_acls() const;

	/*! \brief This function returns ASN1_BIT_STRING::acls.
	 *  \return The value of ASN1_BIT_STRING::acls.
	 */
	ASN1_BIT_STRING * get_acls();


	/*! \brief This function sets ASN1_UTF8STRING::name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	mString & get_name();


	/*! \brief This function sets ASN1_INTEGER::serial.
	 *  \param c_serial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_serial(unsigned long c_serial);

	/*! \brief This function returns ASN1_INTEGER::serial.
	 *  \return The value of ASN1_INTEGER::serial.
	 */
	unsigned long get_serial() const;


	/*! \brief This function sets ASN1_INTEGER::type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::type.
	 *  \return The value of ASN1_INTEGER::type.
	 */
	unsigned long get_type() const;


	/*! \brief This function copies a AclEntry to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const AclEntry & other);
	/*! \brief This member is an empty instance of AclEntry.
	 */
	static AclEntry EmptyInstance;

private:
	ASN1_BIT_STRING * m_acls;
	mString m_name;
	unsigned long m_serial;
	unsigned long m_type;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of X509_ACL
*/
class X509Acl : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	X509Acl();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	X509Acl(const X509Acl & other);

	/*! \brief This is the destructor.
	 */
	~X509Acl();


	/*! \brief This function converts a X509_ACL to a X509Acl.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const X509_ACL * Datas);

	/*! \brief This function converts a X509Acl to a X509_ACL.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(X509_ACL ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of X509_ACL.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets PKI_ADMIN_ENTRY::AdminSerials.
	 *  \param c_adminserials [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_adminserials(const mVector< PkiAdminEntry > & c_adminserials);

	/*! \brief This function returns PKI_ADMIN_ENTRY::AdminSerials.
	 *  \return The value of PKI_ADMIN_ENTRY::AdminSerials.
	 */
	const mVector< PkiAdminEntry > & get_adminserials() const;

	/*! \brief This function returns PKI_ADMIN_ENTRY::AdminSerials.
	 *  \return The value of PKI_ADMIN_ENTRY::AdminSerials.
	 */
	mVector< PkiAdminEntry > & get_adminserials();


	/*! \brief This function sets ACL_ENTRY::acl_entries.
	 *  \param c_aclEntries [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_aclEntries(const mVector< AclEntry > & c_aclEntries);

	/*! \brief This function returns ACL_ENTRY::acl_entries.
	 *  \return The value of ACL_ENTRY::acl_entries.
	 */
	const mVector< AclEntry > & get_aclEntries() const;

	/*! \brief This function returns ACL_ENTRY::acl_entries.
	 *  \return The value of ACL_ENTRY::acl_entries.
	 */
	mVector< AclEntry > & get_aclEntries();


	/*! \brief This function sets ASN1_INTEGER::acl_type.
	 *  \param c_aclType [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_aclType(const mVector< unsigned long > & c_aclType);

	/*! \brief This function returns ASN1_INTEGER::acl_type.
	 *  \return The value of ASN1_INTEGER::acl_type.
	 */
	const mVector< unsigned long > & get_aclType() const;

	/*! \brief This function returns ASN1_INTEGER::acl_type.
	 *  \return The value of ASN1_INTEGER::acl_type.
	 */
	mVector< unsigned long > & get_aclType();


	/*! \brief This function copies a X509Acl to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const X509Acl & other);
	/*! \brief This member is an empty instance of X509Acl.
	 */
	static X509Acl EmptyInstance;

private:
	mVector< PkiAdminEntry > m_adminserials;
	mVector< AclEntry > m_aclEntries;
	mVector< unsigned long > m_aclType;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of REP_ENTRY_INFO
*/
class RepEntryInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	RepEntryInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	RepEntryInfo(const RepEntryInfo & other);

	/*! \brief This is the destructor.
	 */
	~RepEntryInfo();


	/*! \brief This function converts a REP_ENTRY_INFO to a RepEntryInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const REP_ENTRY_INFO * Datas);

	/*! \brief This function converts a RepEntryInfo to a REP_ENTRY_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(REP_ENTRY_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of REP_ENTRY_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::Address.
	 *  \param c_address [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_address(const mString & c_address);

	/*! \brief This function returns ASN1_UTF8STRING::Address.
	 *  \return The value of ASN1_UTF8STRING::Address.
	 */
	const mString & get_address() const;

	/*! \brief This function returns ASN1_UTF8STRING::Address.
	 *  \return The value of ASN1_UTF8STRING::Address.
	 */
	mString & get_address();


	/*! \brief This function sets ASN1_BIT_STRING::Flags.
	 *  \param c_flags [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_flags(const ASN1_BIT_STRING * c_flags);

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	const ASN1_BIT_STRING * get_flags() const;

	/*! \brief This function returns ASN1_BIT_STRING::Flags.
	 *  \return The value of ASN1_BIT_STRING::Flags.
	 */
	ASN1_BIT_STRING * get_flags();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function sets ASN1_INTEGER::Port.
	 *  \param c_port [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_port(unsigned long c_port);

	/*! \brief This function returns ASN1_INTEGER::Port.
	 *  \return The value of ASN1_INTEGER::Port.
	 */
	unsigned long get_port() const;


	/*! \brief This function sets X509::RepositorySSL.
	 *  \param c_repositoryssl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repositoryssl(const PKI_CERT & c_repositoryssl);

	/*! \brief This function returns X509::RepositorySSL.
	 *  \return The value of X509::RepositorySSL.
	 */
	const PKI_CERT & get_repositoryssl() const;

	/*! \brief This function returns X509::RepositorySSL.
	 *  \return The value of X509::RepositorySSL.
	 */
	PKI_CERT & get_repositoryssl();


	/*! \brief This function copies a RepEntryInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const RepEntryInfo & other);
	/*! \brief This member is an empty instance of RepEntryInfo.
	 */
	static RepEntryInfo EmptyInstance;

private:
	mString m_address;
	ASN1_BIT_STRING * m_flags;
	mString m_name;
	unsigned long m_port;
	PKI_CERT m_repositoryssl;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CA_ENTRY_INFO
*/
class CaEntryInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	CaEntryInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	CaEntryInfo(const CaEntryInfo & other);

	/*! \brief This is the destructor.
	 */
	~CaEntryInfo();


	/*! \brief This function converts a CA_ENTRY_INFO to a CaEntryInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CA_ENTRY_INFO * Datas);

	/*! \brief This function converts a CaEntryInfo to a CA_ENTRY_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CA_ENTRY_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CA_ENTRY_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::CaSSL.
	 *  \param c_cassl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cassl(const PKI_CERT & c_cassl);

	/*! \brief This function returns X509::CaSSL.
	 *  \return The value of X509::CaSSL.
	 */
	const PKI_CERT & get_cassl() const;

	/*! \brief This function returns X509::CaSSL.
	 *  \return The value of X509::CaSSL.
	 */
	PKI_CERT & get_cassl();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function copies a CaEntryInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const CaEntryInfo & other);
	/*! \brief This member is an empty instance of CaEntryInfo.
	 */
	static CaEntryInfo EmptyInstance;

private:
	PKI_CERT m_cassl;
	mString m_name;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of CAS_INFO
*/
class CasInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	CasInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	CasInfo(const CasInfo & other);

	/*! \brief This is the destructor.
	 */
	~CasInfo();


	/*! \brief This function converts a CAS_INFO to a CasInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const CAS_INFO * Datas);

	/*! \brief This function converts a CasInfo to a CAS_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(CAS_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of CAS_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets CA_ENTRY_INFO::list.
	 *  \param c_list [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_list(const mVector< CaEntryInfo > & c_list);

	/*! \brief This function returns CA_ENTRY_INFO::list.
	 *  \return The value of CA_ENTRY_INFO::list.
	 */
	const mVector< CaEntryInfo > & get_list() const;

	/*! \brief This function returns CA_ENTRY_INFO::list.
	 *  \return The value of CA_ENTRY_INFO::list.
	 */
	mVector< CaEntryInfo > & get_list();


	/*! \brief This function copies a CasInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const CasInfo & other);
	/*! \brief This member is an empty instance of CasInfo.
	 */
	static CasInfo EmptyInstance;

private:
	mVector< CaEntryInfo > m_list;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of RA_ENTRY_INFO
*/
class RaEntryInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	RaEntryInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	RaEntryInfo(const RaEntryInfo & other);

	/*! \brief This is the destructor.
	 */
	~RaEntryInfo();


	/*! \brief This function converts a RA_ENTRY_INFO to a RaEntryInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const RA_ENTRY_INFO * Datas);

	/*! \brief This function converts a RaEntryInfo to a RA_ENTRY_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(RA_ENTRY_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of RA_ENTRY_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function sets X509::RaSSL.
	 *  \param c_rassl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_rassl(const PKI_CERT & c_rassl);

	/*! \brief This function returns X509::RaSSL.
	 *  \return The value of X509::RaSSL.
	 */
	const PKI_CERT & get_rassl() const;

	/*! \brief This function returns X509::RaSSL.
	 *  \return The value of X509::RaSSL.
	 */
	PKI_CERT & get_rassl();


	/*! \brief This function copies a RaEntryInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const RaEntryInfo & other);
	/*! \brief This member is an empty instance of RaEntryInfo.
	 */
	static RaEntryInfo EmptyInstance;

private:
	mString m_name;
	PKI_CERT m_rassl;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of RAS_INFO
*/
class RasInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	RasInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	RasInfo(const RasInfo & other);

	/*! \brief This is the destructor.
	 */
	~RasInfo();


	/*! \brief This function converts a RAS_INFO to a RasInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const RAS_INFO * Datas);

	/*! \brief This function converts a RasInfo to a RAS_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(RAS_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of RAS_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets RA_ENTRY_INFO::list.
	 *  \param c_list [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_list(const mVector< RaEntryInfo > & c_list);

	/*! \brief This function returns RA_ENTRY_INFO::list.
	 *  \return The value of RA_ENTRY_INFO::list.
	 */
	const mVector< RaEntryInfo > & get_list() const;

	/*! \brief This function returns RA_ENTRY_INFO::list.
	 *  \return The value of RA_ENTRY_INFO::list.
	 */
	mVector< RaEntryInfo > & get_list();


	/*! \brief This function copies a RasInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const RasInfo & other);
	/*! \brief This member is an empty instance of RasInfo.
	 */
	static RasInfo EmptyInstance;

private:
	mVector< RaEntryInfo > m_list;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of KEY_ENTRY_INFO
*/
class KeyEntryInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	KeyEntryInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	KeyEntryInfo(const KeyEntryInfo & other);

	/*! \brief This is the destructor.
	 */
	~KeyEntryInfo();


	/*! \brief This function converts a KEY_ENTRY_INFO to a KeyEntryInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const KEY_ENTRY_INFO * Datas);

	/*! \brief This function converts a KeyEntryInfo to a KEY_ENTRY_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(KEY_ENTRY_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of KEY_ENTRY_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::KeySSL.
	 *  \param c_keyssl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_keyssl(const PKI_CERT & c_keyssl);

	/*! \brief This function returns X509::KeySSL.
	 *  \return The value of X509::KeySSL.
	 */
	const PKI_CERT & get_keyssl() const;

	/*! \brief This function returns X509::KeySSL.
	 *  \return The value of X509::KeySSL.
	 */
	PKI_CERT & get_keyssl();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function copies a KeyEntryInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const KeyEntryInfo & other);
	/*! \brief This member is an empty instance of KeyEntryInfo.
	 */
	static KeyEntryInfo EmptyInstance;

private:
	PKI_CERT m_keyssl;
	mString m_name;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of KEYS_INFO
*/
class KeysInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	KeysInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	KeysInfo(const KeysInfo & other);

	/*! \brief This is the destructor.
	 */
	~KeysInfo();


	/*! \brief This function converts a KEYS_INFO to a KeysInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const KEYS_INFO * Datas);

	/*! \brief This function converts a KeysInfo to a KEYS_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(KEYS_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of KEYS_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets KEY_ENTRY_INFO::list.
	 *  \param c_list [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_list(const mVector< KeyEntryInfo > & c_list);

	/*! \brief This function returns KEY_ENTRY_INFO::list.
	 *  \return The value of KEY_ENTRY_INFO::list.
	 */
	const mVector< KeyEntryInfo > & get_list() const;

	/*! \brief This function returns KEY_ENTRY_INFO::list.
	 *  \return The value of KEY_ENTRY_INFO::list.
	 */
	mVector< KeyEntryInfo > & get_list();


	/*! \brief This function copies a KeysInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const KeysInfo & other);
	/*! \brief This member is an empty instance of KeysInfo.
	 */
	static KeysInfo EmptyInstance;

private:
	mVector< KeyEntryInfo > m_list;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of EE_ENTRY_INFO
*/
class EeEntryInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EeEntryInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EeEntryInfo(const EeEntryInfo & other);

	/*! \brief This is the destructor.
	 */
	~EeEntryInfo();


	/*! \brief This function converts a EE_ENTRY_INFO to a EeEntryInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const EE_ENTRY_INFO * Datas);

	/*! \brief This function converts a EeEntryInfo to a EE_ENTRY_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(EE_ENTRY_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of EE_ENTRY_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::EESSL.
	 *  \param c_eessl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eessl(const PKI_CERT & c_eessl);

	/*! \brief This function returns X509::EESSL.
	 *  \return The value of X509::EESSL.
	 */
	const PKI_CERT & get_eessl() const;

	/*! \brief This function returns X509::EESSL.
	 *  \return The value of X509::EESSL.
	 */
	PKI_CERT & get_eessl();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function copies a EeEntryInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EeEntryInfo & other);
	/*! \brief This member is an empty instance of EeEntryInfo.
	 */
	static EeEntryInfo EmptyInstance;

private:
	PKI_CERT m_eessl;
	mString m_name;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of EES_INFO
*/
class EesInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EesInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EesInfo(const EesInfo & other);

	/*! \brief This is the destructor.
	 */
	~EesInfo();


	/*! \brief This function converts a EES_INFO to a EesInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const EES_INFO * Datas);

	/*! \brief This function converts a EesInfo to a EES_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(EES_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of EES_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets EE_ENTRY_INFO::list.
	 *  \param c_list [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_list(const mVector< EeEntryInfo > & c_list);

	/*! \brief This function returns EE_ENTRY_INFO::list.
	 *  \return The value of EE_ENTRY_INFO::list.
	 */
	const mVector< EeEntryInfo > & get_list() const;

	/*! \brief This function returns EE_ENTRY_INFO::list.
	 *  \return The value of EE_ENTRY_INFO::list.
	 */
	mVector< EeEntryInfo > & get_list();


	/*! \brief This function copies a EesInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EesInfo & other);
	/*! \brief This member is an empty instance of EesInfo.
	 */
	static EesInfo EmptyInstance;

private:
	mVector< EeEntryInfo > m_list;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of BACKUP_ENTRY_INFO
*/
class BackupEntryInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	BackupEntryInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	BackupEntryInfo(const BackupEntryInfo & other);

	/*! \brief This is the destructor.
	 */
	~BackupEntryInfo();


	/*! \brief This function converts a BACKUP_ENTRY_INFO to a BackupEntryInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const BACKUP_ENTRY_INFO * Datas);

	/*! \brief This function converts a BackupEntryInfo to a BACKUP_ENTRY_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(BACKUP_ENTRY_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of BACKUP_ENTRY_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::BackupSSL.
	 *  \param c_backupssl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_backupssl(const PKI_CERT & c_backupssl);

	/*! \brief This function returns X509::BackupSSL.
	 *  \return The value of X509::BackupSSL.
	 */
	const PKI_CERT & get_backupssl() const;

	/*! \brief This function returns X509::BackupSSL.
	 *  \return The value of X509::BackupSSL.
	 */
	PKI_CERT & get_backupssl();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function copies a BackupEntryInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const BackupEntryInfo & other);
	/*! \brief This member is an empty instance of BackupEntryInfo.
	 */
	static BackupEntryInfo EmptyInstance;

private:
	PKI_CERT m_backupssl;
	mString m_name;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of BACKUPS_INFO
*/
class BackupsInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	BackupsInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	BackupsInfo(const BackupsInfo & other);

	/*! \brief This is the destructor.
	 */
	~BackupsInfo();


	/*! \brief This function converts a BACKUPS_INFO to a BackupsInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const BACKUPS_INFO * Datas);

	/*! \brief This function converts a BackupsInfo to a BACKUPS_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(BACKUPS_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of BACKUPS_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets BACKUP_ENTRY_INFO::list.
	 *  \param c_list [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_list(const mVector< BackupEntryInfo > & c_list);

	/*! \brief This function returns BACKUP_ENTRY_INFO::list.
	 *  \return The value of BACKUP_ENTRY_INFO::list.
	 */
	const mVector< BackupEntryInfo > & get_list() const;

	/*! \brief This function returns BACKUP_ENTRY_INFO::list.
	 *  \return The value of BACKUP_ENTRY_INFO::list.
	 */
	mVector< BackupEntryInfo > & get_list();


	/*! \brief This function copies a BackupsInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const BackupsInfo & other);
	/*! \brief This member is an empty instance of BackupsInfo.
	 */
	static BackupsInfo EmptyInstance;

private:
	mVector< BackupEntryInfo > m_list;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PUB_ENTRY_INFO
*/
class PubEntryInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PubEntryInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PubEntryInfo(const PubEntryInfo & other);

	/*! \brief This is the destructor.
	 */
	~PubEntryInfo();


	/*! \brief This function converts a PUB_ENTRY_INFO to a PubEntryInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PUB_ENTRY_INFO * Datas);

	/*! \brief This function converts a PubEntryInfo to a PUB_ENTRY_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PUB_ENTRY_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PUB_ENTRY_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function sets X509::PubOCSP.
	 *  \param c_pubocsp [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pubocsp(const PKI_CERT & c_pubocsp);

	/*! \brief This function returns X509::PubOCSP.
	 *  \return The value of X509::PubOCSP.
	 */
	const PKI_CERT & get_pubocsp() const;

	/*! \brief This function returns X509::PubOCSP.
	 *  \return The value of X509::PubOCSP.
	 */
	PKI_CERT & get_pubocsp();


	/*! \brief This function sets X509::PubSSL.
	 *  \param c_pubssl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pubssl(const PKI_CERT & c_pubssl);

	/*! \brief This function returns X509::PubSSL.
	 *  \return The value of X509::PubSSL.
	 */
	const PKI_CERT & get_pubssl() const;

	/*! \brief This function returns X509::PubSSL.
	 *  \return The value of X509::PubSSL.
	 */
	PKI_CERT & get_pubssl();


	/*! \brief This function copies a PubEntryInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PubEntryInfo & other);
	/*! \brief This member is an empty instance of PubEntryInfo.
	 */
	static PubEntryInfo EmptyInstance;

private:
	mString m_name;
	PKI_CERT m_pubocsp;
	PKI_CERT m_pubssl;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PUBS_INFO
*/
class PubsInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PubsInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PubsInfo(const PubsInfo & other);

	/*! \brief This is the destructor.
	 */
	~PubsInfo();


	/*! \brief This function converts a PUBS_INFO to a PubsInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PUBS_INFO * Datas);

	/*! \brief This function converts a PubsInfo to a PUBS_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PUBS_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PUBS_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets PUB_ENTRY_INFO::list.
	 *  \param c_list [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_list(const mVector< PubEntryInfo > & c_list);

	/*! \brief This function returns PUB_ENTRY_INFO::list.
	 *  \return The value of PUB_ENTRY_INFO::list.
	 */
	const mVector< PubEntryInfo > & get_list() const;

	/*! \brief This function returns PUB_ENTRY_INFO::list.
	 *  \return The value of PUB_ENTRY_INFO::list.
	 */
	mVector< PubEntryInfo > & get_list();


	/*! \brief This function copies a PubsInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PubsInfo & other);
	/*! \brief This member is an empty instance of PubsInfo.
	 */
	static PubsInfo EmptyInstance;

private:
	mVector< PubEntryInfo > m_list;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of INTERNAL_PKI_CRL
*/
class InternalPkiCrl : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	InternalPkiCrl();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	InternalPkiCrl(const InternalPkiCrl & other);

	/*! \brief This is the destructor.
	 */
	~InternalPkiCrl();


	/*! \brief This function converts a INTERNAL_PKI_CRL to a InternalPkiCrl.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const INTERNAL_PKI_CRL * Datas);

	/*! \brief This function converts a InternalPkiCrl to a INTERNAL_PKI_CRL.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(INTERNAL_PKI_CRL ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of INTERNAL_PKI_CRL.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_CRL::EntitiesCaCrl.
	 *  \param c_entitiescacrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitiescacrl(const PKI_CRL & c_entitiescacrl);

	/*! \brief This function returns X509_CRL::EntitiesCaCrl.
	 *  \return The value of X509_CRL::EntitiesCaCrl.
	 */
	const PKI_CRL & get_entitiescacrl() const;

	/*! \brief This function returns X509_CRL::EntitiesCaCrl.
	 *  \return The value of X509_CRL::EntitiesCaCrl.
	 */
	PKI_CRL & get_entitiescacrl();


	/*! \brief This function sets X509_CRL::OcspCaCrl.
	 *  \param c_ocspcacrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ocspcacrl(const PKI_CRL & c_ocspcacrl);

	/*! \brief This function returns X509_CRL::OcspCaCrl.
	 *  \return The value of X509_CRL::OcspCaCrl.
	 */
	const PKI_CRL & get_ocspcacrl() const;

	/*! \brief This function returns X509_CRL::OcspCaCrl.
	 *  \return The value of X509_CRL::OcspCaCrl.
	 */
	PKI_CRL & get_ocspcacrl();


	/*! \brief This function sets X509_CRL::RootCaCrl.
	 *  \param c_rootcacrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_rootcacrl(const PKI_CRL & c_rootcacrl);

	/*! \brief This function returns X509_CRL::RootCaCrl.
	 *  \return The value of X509_CRL::RootCaCrl.
	 */
	const PKI_CRL & get_rootcacrl() const;

	/*! \brief This function returns X509_CRL::RootCaCrl.
	 *  \return The value of X509_CRL::RootCaCrl.
	 */
	PKI_CRL & get_rootcacrl();


	/*! \brief This function sets X509_CRL::UsersCaCrl.
	 *  \param c_userscacrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_userscacrl(const PKI_CRL & c_userscacrl);

	/*! \brief This function returns X509_CRL::UsersCaCrl.
	 *  \return The value of X509_CRL::UsersCaCrl.
	 */
	const PKI_CRL & get_userscacrl() const;

	/*! \brief This function returns X509_CRL::UsersCaCrl.
	 *  \return The value of X509_CRL::UsersCaCrl.
	 */
	PKI_CRL & get_userscacrl();


	/*! \brief This function copies a InternalPkiCrl to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const InternalPkiCrl & other);
	/*! \brief This member is an empty instance of InternalPkiCrl.
	 */
	static InternalPkiCrl EmptyInstance;

private:
	PKI_CRL m_entitiescacrl;
	PKI_CRL m_ocspcacrl;
	PKI_CRL m_rootcacrl;
	PKI_CRL m_userscacrl;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of INTERNAL_PKI_CA
*/
class InternalPkiCa : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	InternalPkiCa();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	InternalPkiCa(const InternalPkiCa & other);

	/*! \brief This is the destructor.
	 */
	~InternalPkiCa();


	/*! \brief This function converts a INTERNAL_PKI_CA to a InternalPkiCa.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const INTERNAL_PKI_CA * Datas);

	/*! \brief This function converts a InternalPkiCa to a INTERNAL_PKI_CA.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(INTERNAL_PKI_CA ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of INTERNAL_PKI_CA.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::EntitiesCa.
	 *  \param c_entitiesca [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitiesca(const PKI_CERT & c_entitiesca);

	/*! \brief This function returns X509::EntitiesCa.
	 *  \return The value of X509::EntitiesCa.
	 */
	const PKI_CERT & get_entitiesca() const;

	/*! \brief This function returns X509::EntitiesCa.
	 *  \return The value of X509::EntitiesCa.
	 */
	PKI_CERT & get_entitiesca();


	/*! \brief This function sets X509::OcspCa.
	 *  \param c_ocspca [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ocspca(const PKI_CERT & c_ocspca);

	/*! \brief This function returns X509::OcspCa.
	 *  \return The value of X509::OcspCa.
	 */
	const PKI_CERT & get_ocspca() const;

	/*! \brief This function returns X509::OcspCa.
	 *  \return The value of X509::OcspCa.
	 */
	PKI_CERT & get_ocspca();


	/*! \brief This function sets X509::PkiCert.
	 *  \param c_pkicert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pkicert(const PKI_CERT & c_pkicert);

	/*! \brief This function returns X509::PkiCert.
	 *  \return The value of X509::PkiCert.
	 */
	const PKI_CERT & get_pkicert() const;

	/*! \brief This function returns X509::PkiCert.
	 *  \return The value of X509::PkiCert.
	 */
	PKI_CERT & get_pkicert();


	/*! \brief This function sets X509::RootCa.
	 *  \param c_rootca [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_rootca(const PKI_CERT & c_rootca);

	/*! \brief This function returns X509::RootCa.
	 *  \return The value of X509::RootCa.
	 */
	const PKI_CERT & get_rootca() const;

	/*! \brief This function returns X509::RootCa.
	 *  \return The value of X509::RootCa.
	 */
	PKI_CERT & get_rootca();


	/*! \brief This function sets X509::UsersCa.
	 *  \param c_usersca [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_usersca(const PKI_CERT & c_usersca);

	/*! \brief This function returns X509::UsersCa.
	 *  \return The value of X509::UsersCa.
	 */
	const PKI_CERT & get_usersca() const;

	/*! \brief This function returns X509::UsersCa.
	 *  \return The value of X509::UsersCa.
	 */
	PKI_CERT & get_usersca();


	/*! \brief This function copies a InternalPkiCa to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const InternalPkiCa & other);
	/*! \brief This member is an empty instance of InternalPkiCa.
	 */
	static InternalPkiCa EmptyInstance;

private:
	PKI_CERT m_entitiesca;
	PKI_CERT m_ocspca;
	PKI_CERT m_pkicert;
	PKI_CERT m_rootca;
	PKI_CERT m_usersca;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_AUDIT_ENTRY
*/
class EntityAuditEntry : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityAuditEntry();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityAuditEntry(const EntityAuditEntry & other);

	/*! \brief This is the destructor.
	 */
	~EntityAuditEntry();


	/*! \brief This function converts a ENTITY_AUDIT_ENTRY to a EntityAuditEntry.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_AUDIT_ENTRY * Datas);

	/*! \brief This function converts a EntityAuditEntry to a ENTITY_AUDIT_ENTRY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_AUDIT_ENTRY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_AUDIT_ENTRY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::object.
	 *  \param c_object [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_object(const mString & c_object);

	/*! \brief This function returns ASN1_UTF8STRING::object.
	 *  \return The value of ASN1_UTF8STRING::object.
	 */
	const mString & get_object() const;

	/*! \brief This function returns ASN1_UTF8STRING::object.
	 *  \return The value of ASN1_UTF8STRING::object.
	 */
	mString & get_object();


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function sets ASN1_INTEGER::type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::type.
	 *  \return The value of ASN1_INTEGER::type.
	 */
	unsigned long get_type() const;


	/*! \brief This function sets ASN1_UTF8STRING::user.
	 *  \param c_user [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_user(const mString & c_user);

	/*! \brief This function returns ASN1_UTF8STRING::user.
	 *  \return The value of ASN1_UTF8STRING::user.
	 */
	const mString & get_user() const;

	/*! \brief This function returns ASN1_UTF8STRING::user.
	 *  \return The value of ASN1_UTF8STRING::user.
	 */
	mString & get_user();


	/*! \brief This function copies a EntityAuditEntry to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityAuditEntry & other);
	/*! \brief This member is an empty instance of EntityAuditEntry.
	 */
	static EntityAuditEntry EmptyInstance;

private:
	mString m_object;
	unsigned long m_status;
	unsigned long m_type;
	mString m_user;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF
*/
class EntityConf : public NewPKIObject, public NewPKISignCryptObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConf();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConf(const EntityConf & other);

	/*! \brief This is the destructor.
	 */
	~EntityConf();


	/*! \brief This function converts a ENTITY_CONF to a EntityConf.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF * Datas);

	/*! \brief This function converts a EntityConf to a ENTITY_CONF.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets BACKUPS_INFO::Backups.
	 *  \param c_backups [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_backups(const BackupsInfo & c_backups);

	/*! \brief This function returns BACKUPS_INFO::Backups.
	 *  \return The value of BACKUPS_INFO::Backups.
	 */
	const BackupsInfo & get_backups() const;

	/*! \brief This function returns BACKUPS_INFO::Backups.
	 *  \return The value of BACKUPS_INFO::Backups.
	 */
	BackupsInfo & get_backups();


	/*! \brief This function sets CAS_INFO::Cas.
	 *  \param c_cas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cas(const CasInfo & c_cas);

	/*! \brief This function returns CAS_INFO::Cas.
	 *  \return The value of CAS_INFO::Cas.
	 */
	const CasInfo & get_cas() const;

	/*! \brief This function returns CAS_INFO::Cas.
	 *  \return The value of CAS_INFO::Cas.
	 */
	CasInfo & get_cas();


	/*! \brief This function sets EES_INFO::EES.
	 *  \param c_ees [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ees(const EesInfo & c_ees);

	/*! \brief This function returns EES_INFO::EES.
	 *  \return The value of EES_INFO::EES.
	 */
	const EesInfo & get_ees() const;

	/*! \brief This function returns EES_INFO::EES.
	 *  \return The value of EES_INFO::EES.
	 */
	EesInfo & get_ees();


	/*! \brief This function sets KEYS_INFO::KeyStores.
	 *  \param c_keystores [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_keystores(const KeysInfo & c_keystores);

	/*! \brief This function returns KEYS_INFO::KeyStores.
	 *  \return The value of KEYS_INFO::KeyStores.
	 */
	const KeysInfo & get_keystores() const;

	/*! \brief This function returns KEYS_INFO::KeyStores.
	 *  \return The value of KEYS_INFO::KeyStores.
	 */
	KeysInfo & get_keystores();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function sets PUBS_INFO::Publications.
	 *  \param c_publications [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_publications(const PubsInfo & c_publications);

	/*! \brief This function returns PUBS_INFO::Publications.
	 *  \return The value of PUBS_INFO::Publications.
	 */
	const PubsInfo & get_publications() const;

	/*! \brief This function returns PUBS_INFO::Publications.
	 *  \return The value of PUBS_INFO::Publications.
	 */
	PubsInfo & get_publications();


	/*! \brief This function sets RAS_INFO::Ras.
	 *  \param c_ras [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ras(const RasInfo & c_ras);

	/*! \brief This function returns RAS_INFO::Ras.
	 *  \return The value of RAS_INFO::Ras.
	 */
	const RasInfo & get_ras() const;

	/*! \brief This function returns RAS_INFO::Ras.
	 *  \return The value of RAS_INFO::Ras.
	 */
	RasInfo & get_ras();


	/*! \brief This function sets REP_ENTRY_INFO::Repositories.
	 *  \param c_repositories [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repositories(const mVector< RepEntryInfo > & c_repositories);

	/*! \brief This function returns REP_ENTRY_INFO::Repositories.
	 *  \return The value of REP_ENTRY_INFO::Repositories.
	 */
	const mVector< RepEntryInfo > & get_repositories() const;

	/*! \brief This function returns REP_ENTRY_INFO::Repositories.
	 *  \return The value of REP_ENTRY_INFO::Repositories.
	 */
	mVector< RepEntryInfo > & get_repositories();


	/*! \brief This function sets X509_ACL::acls.
	 *  \param c_acls [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_acls(const X509Acl & c_acls);

	/*! \brief This function returns X509_ACL::acls.
	 *  \return The value of X509_ACL::acls.
	 */
	const X509Acl & get_acls() const;

	/*! \brief This function returns X509_ACL::acls.
	 *  \return The value of X509_ACL::acls.
	 */
	X509Acl & get_acls();


	/*! \brief This function sets ENTITY_AUDIT_ENTRY::audits.
	 *  \param c_audits [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_audits(const mVector< EntityAuditEntry > & c_audits);

	/*! \brief This function returns ENTITY_AUDIT_ENTRY::audits.
	 *  \return The value of ENTITY_AUDIT_ENTRY::audits.
	 */
	const mVector< EntityAuditEntry > & get_audits() const;

	/*! \brief This function returns ENTITY_AUDIT_ENTRY::audits.
	 *  \return The value of ENTITY_AUDIT_ENTRY::audits.
	 */
	mVector< EntityAuditEntry > & get_audits();


	/*! \brief This function sets ENTITY_CONF_BODY::body.
	 *  \param c_body [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_body(const EntityConfBody & c_body);

	/*! \brief This function returns ENTITY_CONF_BODY::body.
	 *  \return The value of ENTITY_CONF_BODY::body.
	 */
	const EntityConfBody & get_body() const;

	/*! \brief This function returns ENTITY_CONF_BODY::body.
	 *  \return The value of ENTITY_CONF_BODY::body.
	 */
	EntityConfBody & get_body();


	/*! \brief This function sets INTERNAL_PKI_CRL::crls.
	 *  \param c_crls [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crls(const InternalPkiCrl & c_crls);

	/*! \brief This function returns INTERNAL_PKI_CRL::crls.
	 *  \return The value of INTERNAL_PKI_CRL::crls.
	 */
	const InternalPkiCrl & get_crls() const;

	/*! \brief This function returns INTERNAL_PKI_CRL::crls.
	 *  \return The value of INTERNAL_PKI_CRL::crls.
	 */
	InternalPkiCrl & get_crls();


	/*! \brief This function sets USERS_GROUP::groups.
	 *  \param c_groups [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_groups(const mVector< UsersGroup > & c_groups);

	/*! \brief This function returns USERS_GROUP::groups.
	 *  \return The value of USERS_GROUP::groups.
	 */
	const mVector< UsersGroup > & get_groups() const;

	/*! \brief This function returns USERS_GROUP::groups.
	 *  \return The value of USERS_GROUP::groups.
	 */
	mVector< UsersGroup > & get_groups();


	/*! \brief This function sets EMAIL_CONF::mail_conf.
	 *  \param c_mailConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_mailConf(const EmailConf & c_mailConf);

	/*! \brief This function returns EMAIL_CONF::mail_conf.
	 *  \return The value of EMAIL_CONF::mail_conf.
	 */
	const EmailConf & get_mailConf() const;

	/*! \brief This function returns EMAIL_CONF::mail_conf.
	 *  \return The value of EMAIL_CONF::mail_conf.
	 */
	EmailConf & get_mailConf();


	/*! \brief This function sets ASN1_INTEGER::version.
	 *  \param c_version [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_version(unsigned long c_version);

	/*! \brief This function returns ASN1_INTEGER::version.
	 *  \return The value of ASN1_INTEGER::version.
	 */
	unsigned long get_version() const;


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a EntityConf to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConf & other);
	/*! \brief This member is an empty instance of EntityConf.
	 */
	static EntityConf EmptyInstance;

private:
	BackupsInfo m_backups;
	CasInfo m_cas;
	EesInfo m_ees;
	KeysInfo m_keystores;
	mString m_name;
	PubsInfo m_publications;
	RasInfo m_ras;
	mVector< RepEntryInfo > m_repositories;
	X509Acl m_acls;
	mVector< EntityAuditEntry > m_audits;
	EntityConfBody m_body;
	InternalPkiCrl m_crls;
	mVector< UsersGroup > m_groups;
	EmailConf m_mailConf;
	unsigned long m_version;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_BETA4
*/
class EntityConfBeta4 : public NewPKIObject, public NewPKISignCryptObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfBeta4();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfBeta4(const EntityConfBeta4 & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfBeta4();


	/*! \brief This function converts a ENTITY_CONF_BETA4 to a EntityConfBeta4.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_BETA4 * Datas);

	/*! \brief This function converts a EntityConfBeta4 to a ENTITY_CONF_BETA4.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_BETA4 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_BETA4.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets BACKUPS_INFO::Backups.
	 *  \param c_backups [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_backups(const BackupsInfo & c_backups);

	/*! \brief This function returns BACKUPS_INFO::Backups.
	 *  \return The value of BACKUPS_INFO::Backups.
	 */
	const BackupsInfo & get_backups() const;

	/*! \brief This function returns BACKUPS_INFO::Backups.
	 *  \return The value of BACKUPS_INFO::Backups.
	 */
	BackupsInfo & get_backups();


	/*! \brief This function sets CAS_INFO::Cas.
	 *  \param c_cas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cas(const CasInfo & c_cas);

	/*! \brief This function returns CAS_INFO::Cas.
	 *  \return The value of CAS_INFO::Cas.
	 */
	const CasInfo & get_cas() const;

	/*! \brief This function returns CAS_INFO::Cas.
	 *  \return The value of CAS_INFO::Cas.
	 */
	CasInfo & get_cas();


	/*! \brief This function sets KEYS_INFO::KeyStores.
	 *  \param c_keystores [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_keystores(const KeysInfo & c_keystores);

	/*! \brief This function returns KEYS_INFO::KeyStores.
	 *  \return The value of KEYS_INFO::KeyStores.
	 */
	const KeysInfo & get_keystores() const;

	/*! \brief This function returns KEYS_INFO::KeyStores.
	 *  \return The value of KEYS_INFO::KeyStores.
	 */
	KeysInfo & get_keystores();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function sets PUBS_INFO::Publications.
	 *  \param c_publications [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_publications(const PubsInfo & c_publications);

	/*! \brief This function returns PUBS_INFO::Publications.
	 *  \return The value of PUBS_INFO::Publications.
	 */
	const PubsInfo & get_publications() const;

	/*! \brief This function returns PUBS_INFO::Publications.
	 *  \return The value of PUBS_INFO::Publications.
	 */
	PubsInfo & get_publications();


	/*! \brief This function sets RAS_INFO::Ras.
	 *  \param c_ras [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ras(const RasInfo & c_ras);

	/*! \brief This function returns RAS_INFO::Ras.
	 *  \return The value of RAS_INFO::Ras.
	 */
	const RasInfo & get_ras() const;

	/*! \brief This function returns RAS_INFO::Ras.
	 *  \return The value of RAS_INFO::Ras.
	 */
	RasInfo & get_ras();


	/*! \brief This function sets REP_ENTRY_INFO::Repositories.
	 *  \param c_repositories [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repositories(const mVector< RepEntryInfo > & c_repositories);

	/*! \brief This function returns REP_ENTRY_INFO::Repositories.
	 *  \return The value of REP_ENTRY_INFO::Repositories.
	 */
	const mVector< RepEntryInfo > & get_repositories() const;

	/*! \brief This function returns REP_ENTRY_INFO::Repositories.
	 *  \return The value of REP_ENTRY_INFO::Repositories.
	 */
	mVector< RepEntryInfo > & get_repositories();


	/*! \brief This function sets X509_ACL::acls.
	 *  \param c_acls [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_acls(const X509Acl & c_acls);

	/*! \brief This function returns X509_ACL::acls.
	 *  \return The value of X509_ACL::acls.
	 */
	const X509Acl & get_acls() const;

	/*! \brief This function returns X509_ACL::acls.
	 *  \return The value of X509_ACL::acls.
	 */
	X509Acl & get_acls();


	/*! \brief This function sets ENTITY_AUDIT_ENTRY::audits.
	 *  \param c_audits [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_audits(const mVector< EntityAuditEntry > & c_audits);

	/*! \brief This function returns ENTITY_AUDIT_ENTRY::audits.
	 *  \return The value of ENTITY_AUDIT_ENTRY::audits.
	 */
	const mVector< EntityAuditEntry > & get_audits() const;

	/*! \brief This function returns ENTITY_AUDIT_ENTRY::audits.
	 *  \return The value of ENTITY_AUDIT_ENTRY::audits.
	 */
	mVector< EntityAuditEntry > & get_audits();


	/*! \brief This function sets ENTITY_CONF_BODY::body.
	 *  \param c_body [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_body(const EntityConfBody & c_body);

	/*! \brief This function returns ENTITY_CONF_BODY::body.
	 *  \return The value of ENTITY_CONF_BODY::body.
	 */
	const EntityConfBody & get_body() const;

	/*! \brief This function returns ENTITY_CONF_BODY::body.
	 *  \return The value of ENTITY_CONF_BODY::body.
	 */
	EntityConfBody & get_body();


	/*! \brief This function sets INTERNAL_PKI_CRL::crls.
	 *  \param c_crls [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crls(const InternalPkiCrl & c_crls);

	/*! \brief This function returns INTERNAL_PKI_CRL::crls.
	 *  \return The value of INTERNAL_PKI_CRL::crls.
	 */
	const InternalPkiCrl & get_crls() const;

	/*! \brief This function returns INTERNAL_PKI_CRL::crls.
	 *  \return The value of INTERNAL_PKI_CRL::crls.
	 */
	InternalPkiCrl & get_crls();


	/*! \brief This function sets USERS_GROUP::groups.
	 *  \param c_groups [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_groups(const mVector< UsersGroup > & c_groups);

	/*! \brief This function returns USERS_GROUP::groups.
	 *  \return The value of USERS_GROUP::groups.
	 */
	const mVector< UsersGroup > & get_groups() const;

	/*! \brief This function returns USERS_GROUP::groups.
	 *  \return The value of USERS_GROUP::groups.
	 */
	mVector< UsersGroup > & get_groups();


	/*! \brief This function sets EMAIL_CONF::mail_conf.
	 *  \param c_mailConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_mailConf(const EmailConf & c_mailConf);

	/*! \brief This function returns EMAIL_CONF::mail_conf.
	 *  \return The value of EMAIL_CONF::mail_conf.
	 */
	const EmailConf & get_mailConf() const;

	/*! \brief This function returns EMAIL_CONF::mail_conf.
	 *  \return The value of EMAIL_CONF::mail_conf.
	 */
	EmailConf & get_mailConf();


	/*! \brief This function sets ASN1_INTEGER::version.
	 *  \param c_version [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_version(unsigned long c_version);

	/*! \brief This function returns ASN1_INTEGER::version.
	 *  \return The value of ASN1_INTEGER::version.
	 */
	unsigned long get_version() const;


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a EntityConfBeta4 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfBeta4 & other);
	/*! \brief This member is an empty instance of EntityConfBeta4.
	 */
	static EntityConfBeta4 EmptyInstance;

private:
	BackupsInfo m_backups;
	CasInfo m_cas;
	KeysInfo m_keystores;
	mString m_name;
	PubsInfo m_publications;
	RasInfo m_ras;
	mVector< RepEntryInfo > m_repositories;
	X509Acl m_acls;
	mVector< EntityAuditEntry > m_audits;
	EntityConfBody m_body;
	InternalPkiCrl m_crls;
	mVector< UsersGroup > m_groups;
	EmailConf m_mailConf;
	unsigned long m_version;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CONF_CRYPTED
*/
class EntityConfCrypted : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityConfCrypted();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityConfCrypted(const EntityConfCrypted & other);

	/*! \brief This is the destructor.
	 */
	~EntityConfCrypted();


	/*! \brief This function converts a ENTITY_CONF_CRYPTED to a EntityConfCrypted.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CONF_CRYPTED * Datas);

	/*! \brief This function converts a EntityConfCrypted to a ENTITY_CONF_CRYPTED.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CONF_CRYPTED ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CONF_CRYPTED.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_ENCRYPT_SIGN::crypted.
	 *  \param c_crypted [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crypted(const Asn1EncryptSign & c_crypted);

	/*! \brief This function returns ASN1_ENCRYPT_SIGN::crypted.
	 *  \return The value of ASN1_ENCRYPT_SIGN::crypted.
	 */
	const Asn1EncryptSign & get_crypted() const;

	/*! \brief This function returns ASN1_ENCRYPT_SIGN::crypted.
	 *  \return The value of ASN1_ENCRYPT_SIGN::crypted.
	 */
	Asn1EncryptSign & get_crypted();


	/*! \brief This function sets X509_PUBKEY::recipient.
	 *  \param c_recipient [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_recipient(const X509_PUBKEY * c_recipient);

	/*! \brief This function returns X509_PUBKEY::recipient.
	 *  \return The value of X509_PUBKEY::recipient.
	 */
	const X509_PUBKEY * get_recipient() const;

	/*! \brief This function returns X509_PUBKEY::recipient.
	 *  \return The value of X509_PUBKEY::recipient.
	 */
	X509_PUBKEY * get_recipient();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a EntityConfCrypted to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityConfCrypted & other);
	/*! \brief This member is an empty instance of EntityConfCrypted.
	 */
	static EntityConfCrypted EmptyInstance;

private:
	Asn1EncryptSign m_crypted;
	X509_PUBKEY * m_recipient;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of EXPORTED_PKI_CONF_BODY
*/
class ExportedPkiConfBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ExportedPkiConfBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ExportedPkiConfBody(const ExportedPkiConfBody & other);

	/*! \brief This is the destructor.
	 */
	~ExportedPkiConfBody();


	/*! \brief This function converts a EXPORTED_PKI_CONF_BODY to a ExportedPkiConfBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const EXPORTED_PKI_CONF_BODY * Datas);

	/*! \brief This function converts a ExportedPkiConfBody to a EXPORTED_PKI_CONF_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(EXPORTED_PKI_CONF_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of EXPORTED_PKI_CONF_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ENTITY_CONF_CRYPTED::all_confs.
	 *  \param c_allConfs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_allConfs(const mVector< EntityConfCrypted > & c_allConfs);

	/*! \brief This function returns ENTITY_CONF_CRYPTED::all_confs.
	 *  \return The value of ENTITY_CONF_CRYPTED::all_confs.
	 */
	const mVector< EntityConfCrypted > & get_allConfs() const;

	/*! \brief This function returns ENTITY_CONF_CRYPTED::all_confs.
	 *  \return The value of ENTITY_CONF_CRYPTED::all_confs.
	 */
	mVector< EntityConfCrypted > & get_allConfs();


	/*! \brief This function sets ASN1_INTEGER::version.
	 *  \param c_version [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_version(unsigned long c_version);

	/*! \brief This function returns ASN1_INTEGER::version.
	 *  \return The value of ASN1_INTEGER::version.
	 */
	unsigned long get_version() const;


	/*! \brief This function copies a ExportedPkiConfBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ExportedPkiConfBody & other);
	/*! \brief This member is an empty instance of ExportedPkiConfBody.
	 */
	static ExportedPkiConfBody EmptyInstance;

private:
	mVector< EntityConfCrypted > m_allConfs;
	unsigned long m_version;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of EXPORTED_PKI_CONF
*/
class ExportedPkiConf : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ExportedPkiConf();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ExportedPkiConf(const ExportedPkiConf & other);

	/*! \brief This is the destructor.
	 */
	~ExportedPkiConf();


	/*! \brief This function converts a EXPORTED_PKI_CONF to a ExportedPkiConf.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const EXPORTED_PKI_CONF * Datas);

	/*! \brief This function converts a ExportedPkiConf to a EXPORTED_PKI_CONF.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(EXPORTED_PKI_CONF ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of EXPORTED_PKI_CONF.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets EXPORTED_PKI_CONF_BODY::confs.
	 *  \param c_confs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_confs(const ExportedPkiConfBody & c_confs);

	/*! \brief This function returns EXPORTED_PKI_CONF_BODY::confs.
	 *  \return The value of EXPORTED_PKI_CONF_BODY::confs.
	 */
	const ExportedPkiConfBody & get_confs() const;

	/*! \brief This function returns EXPORTED_PKI_CONF_BODY::confs.
	 *  \return The value of EXPORTED_PKI_CONF_BODY::confs.
	 */
	ExportedPkiConfBody & get_confs();


	/*! \brief This function sets X509_PUBKEY::rep_path.
	 *  \param c_repPath [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repPath(const STACK_OF(X509_PUBKEY) * c_repPath);

	/*! \brief This function returns X509_PUBKEY::rep_path.
	 *  \return The value of X509_PUBKEY::rep_path.
	 */
	const STACK_OF(X509_PUBKEY) * get_repPath() const;

	/*! \brief This function returns X509_PUBKEY::rep_path.
	 *  \return The value of X509_PUBKEY::rep_path.
	 */
	STACK_OF(X509_PUBKEY) * get_repPath();


	/*! \brief This function sets X509_SIG::sig.
	 *  \param c_sig [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sig(const X509_SIG * c_sig);

	/*! \brief This function returns X509_SIG::sig.
	 *  \return The value of X509_SIG::sig.
	 */
	const X509_SIG * get_sig() const;

	/*! \brief This function returns X509_SIG::sig.
	 *  \return The value of X509_SIG::sig.
	 */
	X509_SIG * get_sig();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a ExportedPkiConf to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ExportedPkiConf & other);
	/*! \brief This member is an empty instance of ExportedPkiConf.
	 */
	static ExportedPkiConf EmptyInstance;

private:
	ExportedPkiConfBody m_confs;
	STACK_OF(X509_PUBKEY) * m_repPath;
	X509_SIG * m_sig;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_PROFILE
*/
class NewpkiProfile : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiProfile();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiProfile(const NewpkiProfile & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiProfile();


	/*! \brief This function converts a NEWPKI_PROFILE to a NewpkiProfile.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_PROFILE * Datas);

	/*! \brief This function converts a NewpkiProfile to a NEWPKI_PROFILE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_PROFILE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_PROFILE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::EE_Validation.
	 *  \param c_eeValidation [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeValidation(unsigned long c_eeValidation);

	/*! \brief This function returns ASN1_INTEGER::EE_Validation.
	 *  \return The value of ASN1_INTEGER::EE_Validation.
	 */
	unsigned long get_eeValidation() const;


	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets ASN1_UTF8STRING::ldap_uid.
	 *  \param c_ldapUid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapUid(const mString & c_ldapUid);

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	const mString & get_ldapUid() const;

	/*! \brief This function returns ASN1_UTF8STRING::ldap_uid.
	 *  \return The value of ASN1_UTF8STRING::ldap_uid.
	 */
	mString & get_ldapUid();


	/*! \brief This function sets ASN1_INTEGER::owner_group_serial.
	 *  \param c_ownerGroupSerial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ownerGroupSerial(unsigned long c_ownerGroupSerial);

	/*! \brief This function returns ASN1_INTEGER::owner_group_serial.
	 *  \return The value of ASN1_INTEGER::owner_group_serial.
	 */
	unsigned long get_ownerGroupSerial() const;


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a NewpkiProfile to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiProfile & other);
	/*! \brief This member is an empty instance of NewpkiProfile.
	 */
	static NewpkiProfile EmptyInstance;

private:
	unsigned long m_eeValidation;
	X509_NAME * m_dn;
	mString m_ldapUid;
	unsigned long m_ownerGroupSerial;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_PROFILE_DATAS_CERT
*/
class NewpkiProfileDatasCert : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiProfileDatasCert();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiProfileDatasCert(const NewpkiProfileDatasCert & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiProfileDatasCert();


	/*! \brief This function converts a NEWPKI_PROFILE_DATAS_CERT to a NewpkiProfileDatasCert.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_PROFILE_DATAS_CERT * Datas);

	/*! \brief This function converts a NewpkiProfileDatasCert to a NEWPKI_PROFILE_DATAS_CERT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_PROFILE_DATAS_CERT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_PROFILE_DATAS_CERT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::Id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::Id.
	 *  \return The value of ASN1_INTEGER::Id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets ASN1_UTF8STRING::admin_mail.
	 *  \param c_adminMail [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_adminMail(const mString & c_adminMail);

	/*! \brief This function returns ASN1_UTF8STRING::admin_mail.
	 *  \return The value of ASN1_UTF8STRING::admin_mail.
	 */
	const mString & get_adminMail() const;

	/*! \brief This function returns ASN1_UTF8STRING::admin_mail.
	 *  \return The value of ASN1_UTF8STRING::admin_mail.
	 */
	mString & get_adminMail();


	/*! \brief This function sets ASN1_UTF8STRING::ca_name.
	 *  \param c_caName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caName(const mString & c_caName);

	/*! \brief This function returns ASN1_UTF8STRING::ca_name.
	 *  \return The value of ASN1_UTF8STRING::ca_name.
	 */
	const mString & get_caName() const;

	/*! \brief This function returns ASN1_UTF8STRING::ca_name.
	 *  \return The value of ASN1_UTF8STRING::ca_name.
	 */
	mString & get_caName();


	/*! \brief This function sets X509::cert.
	 *  \param c_cert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cert(const PKI_CERT & c_cert);

	/*! \brief This function returns X509::cert.
	 *  \return The value of X509::cert.
	 */
	const PKI_CERT & get_cert() const;

	/*! \brief This function returns X509::cert.
	 *  \return The value of X509::cert.
	 */
	PKI_CERT & get_cert();


	/*! \brief This function sets ASN1_UTF8STRING::error.
	 *  \param c_error [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_error(const mString & c_error);

	/*! \brief This function returns ASN1_UTF8STRING::error.
	 *  \return The value of ASN1_UTF8STRING::error.
	 */
	const mString & get_error() const;

	/*! \brief This function returns ASN1_UTF8STRING::error.
	 *  \return The value of ASN1_UTF8STRING::error.
	 */
	mString & get_error();


	/*! \brief This function sets ASN1_BIT_STRING::flags.
	 *  \param c_flags [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_flags(const ASN1_BIT_STRING * c_flags);

	/*! \brief This function returns ASN1_BIT_STRING::flags.
	 *  \return The value of ASN1_BIT_STRING::flags.
	 */
	const ASN1_BIT_STRING * get_flags() const;

	/*! \brief This function returns ASN1_BIT_STRING::flags.
	 *  \return The value of ASN1_BIT_STRING::flags.
	 */
	ASN1_BIT_STRING * get_flags();


	/*! \brief This function sets PKCS12::p12.
	 *  \param c_p12 [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p12(const PKI_PKCS12 & c_p12);

	/*! \brief This function returns PKCS12::p12.
	 *  \return The value of PKCS12::p12.
	 */
	const PKI_PKCS12 & get_p12() const;

	/*! \brief This function returns PKCS12::p12.
	 *  \return The value of PKCS12::p12.
	 */
	PKI_PKCS12 & get_p12();


	/*! \brief This function sets PKCS7::p7b.
	 *  \param c_p7b [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p7b(const PKI_P7B & c_p7b);

	/*! \brief This function returns PKCS7::p7b.
	 *  \return The value of PKCS7::p7b.
	 */
	const PKI_P7B & get_p7b() const;

	/*! \brief This function returns PKCS7::p7b.
	 *  \return The value of PKCS7::p7b.
	 */
	PKI_P7B & get_p7b();


	/*! \brief This function sets ASN1_INTEGER::state.
	 *  \param c_state [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_state(unsigned long c_state);

	/*! \brief This function returns ASN1_INTEGER::state.
	 *  \return The value of ASN1_INTEGER::state.
	 */
	unsigned long get_state() const;


	/*! \brief This function sets ASN1_INTEGER::type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::type.
	 *  \return The value of ASN1_INTEGER::type.
	 */
	unsigned long get_type() const;


	/*! \brief This function copies a NewpkiProfileDatasCert to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiProfileDatasCert & other);
	/*! \brief This member is an empty instance of NewpkiProfileDatasCert.
	 */
	static NewpkiProfileDatasCert EmptyInstance;

private:
	unsigned long m_id;
	mString m_adminMail;
	mString m_caName;
	PKI_CERT m_cert;
	mString m_error;
	ASN1_BIT_STRING * m_flags;
	PKI_PKCS12 m_p12;
	PKI_P7B m_p7b;
	unsigned long m_state;
	unsigned long m_type;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_PROFILE_DATAS
*/
class NewpkiProfileDatas : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiProfileDatas();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiProfileDatas(const NewpkiProfileDatas & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiProfileDatas();


	/*! \brief This function converts a NEWPKI_PROFILE_DATAS to a NewpkiProfileDatas.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_PROFILE_DATAS * Datas);

	/*! \brief This function converts a NewpkiProfileDatas to a NEWPKI_PROFILE_DATAS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_PROFILE_DATAS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_PROFILE_DATAS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets NEWPKI_PROFILE_DATAS_CERT::Certs.
	 *  \param c_certs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certs(const mVector< NewpkiProfileDatasCert > & c_certs);

	/*! \brief This function returns NEWPKI_PROFILE_DATAS_CERT::Certs.
	 *  \return The value of NEWPKI_PROFILE_DATAS_CERT::Certs.
	 */
	const mVector< NewpkiProfileDatasCert > & get_certs() const;

	/*! \brief This function returns NEWPKI_PROFILE_DATAS_CERT::Certs.
	 *  \return The value of NEWPKI_PROFILE_DATAS_CERT::Certs.
	 */
	mVector< NewpkiProfileDatasCert > & get_certs();


	/*! \brief This function sets ASN1_INTEGER::Id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::Id.
	 *  \return The value of ASN1_INTEGER::Id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets NEWPKI_PROFILE::Profile.
	 *  \param c_profile [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profile(const NewpkiProfile & c_profile);

	/*! \brief This function returns NEWPKI_PROFILE::Profile.
	 *  \return The value of NEWPKI_PROFILE::Profile.
	 */
	const NewpkiProfile & get_profile() const;

	/*! \brief This function returns NEWPKI_PROFILE::Profile.
	 *  \return The value of NEWPKI_PROFILE::Profile.
	 */
	NewpkiProfile & get_profile();


	/*! \brief This function sets ASN1_INTEGER::State.
	 *  \param c_state [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_state(unsigned long c_state);

	/*! \brief This function returns ASN1_INTEGER::State.
	 *  \return The value of ASN1_INTEGER::State.
	 */
	unsigned long get_state() const;


	/*! \brief This function sets ASN1_UTF8STRING::ee_id.
	 *  \param c_eeId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeId(const mString & c_eeId);

	/*! \brief This function returns ASN1_UTF8STRING::ee_id.
	 *  \return The value of ASN1_UTF8STRING::ee_id.
	 */
	const mString & get_eeId() const;

	/*! \brief This function returns ASN1_UTF8STRING::ee_id.
	 *  \return The value of ASN1_UTF8STRING::ee_id.
	 */
	mString & get_eeId();


	/*! \brief This function copies a NewpkiProfileDatas to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiProfileDatas & other);
	/*! \brief This member is an empty instance of NewpkiProfileDatas.
	 */
	static NewpkiProfileDatas EmptyInstance;

private:
	mVector< NewpkiProfileDatasCert > m_certs;
	unsigned long m_id;
	NewpkiProfile m_profile;
	unsigned long m_state;
	mString m_eeId;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_EE_USER
*/
class NewpkiEeUser : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiEeUser();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiEeUser(const NewpkiEeUser & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiEeUser();


	/*! \brief This function converts a NEWPKI_EE_USER to a NewpkiEeUser.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_EE_USER * Datas);

	/*! \brief This function converts a NewpkiEeUser to a NEWPKI_EE_USER.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_EE_USER ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_EE_USER.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets NEWPKI_PROFILE_DATAS_CERT::Certs.
	 *  \param c_certs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certs(const mVector< NewpkiProfileDatasCert > & c_certs);

	/*! \brief This function returns NEWPKI_PROFILE_DATAS_CERT::Certs.
	 *  \return The value of NEWPKI_PROFILE_DATAS_CERT::Certs.
	 */
	const mVector< NewpkiProfileDatasCert > & get_certs() const;

	/*! \brief This function returns NEWPKI_PROFILE_DATAS_CERT::Certs.
	 *  \return The value of NEWPKI_PROFILE_DATAS_CERT::Certs.
	 */
	mVector< NewpkiProfileDatasCert > & get_certs();


	/*! \brief This function sets ASN1_INTEGER::Id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::Id.
	 *  \return The value of ASN1_INTEGER::Id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets ASN1_INTEGER::activated.
	 *  \param c_activated [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_activated(unsigned long c_activated);

	/*! \brief This function returns ASN1_INTEGER::activated.
	 *  \return The value of ASN1_INTEGER::activated.
	 */
	unsigned long get_activated() const;


	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets ASN1_UTF8STRING::email.
	 *  \param c_email [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_email(const mString & c_email);

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	const mString & get_email() const;

	/*! \brief This function returns ASN1_UTF8STRING::email.
	 *  \return The value of ASN1_UTF8STRING::email.
	 */
	mString & get_email();


	/*! \brief This function sets ASN1_INTEGER::last_send.
	 *  \param c_lastSend [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastSend(unsigned long c_lastSend);

	/*! \brief This function returns ASN1_INTEGER::last_send.
	 *  \return The value of ASN1_INTEGER::last_send.
	 */
	unsigned long get_lastSend() const;


	/*! \brief This function sets ASN1_INTEGER::ra_validated.
	 *  \param c_raValidated [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_raValidated(unsigned long c_raValidated);

	/*! \brief This function returns ASN1_INTEGER::ra_validated.
	 *  \return The value of ASN1_INTEGER::ra_validated.
	 */
	unsigned long get_raValidated() const;


	/*! \brief This function sets ASN1_INTEGER::send_count.
	 *  \param c_sendCount [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sendCount(unsigned long c_sendCount);

	/*! \brief This function returns ASN1_INTEGER::send_count.
	 *  \return The value of ASN1_INTEGER::send_count.
	 */
	unsigned long get_sendCount() const;


	/*! \brief This function copies a NewpkiEeUser to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiEeUser & other);
	/*! \brief This member is an empty instance of NewpkiEeUser.
	 */
	static NewpkiEeUser EmptyInstance;

private:
	mVector< NewpkiProfileDatasCert > m_certs;
	unsigned long m_id;
	unsigned long m_activated;
	X509_NAME * m_dn;
	mString m_email;
	unsigned long m_lastSend;
	unsigned long m_raValidated;
	unsigned long m_sendCount;
	void resetAll();
	void freeAll();
protected:
};


/*! \brief This function returns the string representation of an ACL.
 *  \param Type [IN] The ACL.
 *  \return The string representation of an ACL.
 */
char * GetACL_String(ACL_TYPE Type);

bool Upgrade_EntityConf_From_Beta4_To_2_0_0(const EntityConfBeta4 & Beta4Conf, EntityConf & Conf);

#endif // #ifndef ASN1CONF_H
